/*
 * File:     MediaNavPanel.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.prefs.gui;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.prefs.PreferenceEditor;

import mpi.eudico.client.annotator.util.FileUtility;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.io.File;

import java.util.HashMap;
import java.util.Map;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;


/**
 * Panel showing option to change the media navigation setting: <br>
 * - frame forward/backward jumps to the begin of next/previous frame (default
 * it jumps with the amount of ms of the duration of a single frame)  - a
 * default location (directory) to search for media files (after the
 * "traditional" location)
 *
 * @author Han Sloetjes
 * @version 2.0, Dec 2008
 */
public class MediaNavPanel extends JPanel implements PreferenceEditor,
    ActionListener {
    private boolean origFrameStepToFrameBegin = false;
    private String curMediaLocation = "-";
    private boolean origVideoSameSize = false;
    private JCheckBox frameStepCB;
    private JCheckBox videosSameSizeCB;
    private JLabel setDirLabel;
    private JLabel curDirLabel;
    private JButton defaultDirButton;
    private JButton resetDirButton;

    /**
     * Creates a new MediaNavPanel instance
     */
    public MediaNavPanel() {
        super();
        readPrefs();
        initComponents();
    }

    private void readPrefs() {
        Object val = Preferences.get("MediaNavigation.FrameStepToFrameBegin",
                null);

        if (val instanceof Boolean) {
            origFrameStepToFrameBegin = ((Boolean) val).booleanValue();
        }

        val = Preferences.get("DefaultMediaLocation", null);

        if (val instanceof String) {
            curMediaLocation = (String) val;
        }

        val = Preferences.get("Media.VideosSameSize", null);

        if (val instanceof Boolean) {
            origVideoSameSize = ((Boolean) val).booleanValue();
        }
    }

    private void initComponents() {
        setLayout(new GridBagLayout());

        Insets insets = new Insets(2, 0, 2, 0);
        frameStepCB = new JCheckBox(ElanLocale.getString(
                    "PreferencesDialog.MediaNav.FrameBegin"),
                origFrameStepToFrameBegin);
        frameStepCB.setFont(frameStepCB.getFont().deriveFont(Font.PLAIN));

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        gbc.insets = insets;
        gbc.gridwidth = 3;
        add(new JLabel(ElanLocale.getString(
                    "PreferencesDialog.Category.MediaNav")), gbc);

        gbc.gridy = 1;
        add(frameStepCB, gbc);

        gbc.gridy = 2;
        gbc.insets = new Insets(12, 0, 2, 0);
        add(new JLabel(ElanLocale.getString(
                    "PreferencesDialog.Media.VideoDisplay")), gbc);
        videosSameSizeCB = new JCheckBox(ElanLocale.getString(
                    "PreferencesDialog.Media.VideoSize"), origVideoSameSize);
        gbc.gridy = 3;
        gbc.insets = insets;
        videosSameSizeCB.setFont(videosSameSizeCB.getFont().deriveFont(Font.PLAIN));
        add(videosSameSizeCB, gbc);

        gbc.gridy = 4;
        gbc.insets = new Insets(12, 0, 2, 0);
        add(new JLabel(ElanLocale.getString("PreferencesDialog.Media.Location")),
            gbc);
        setDirLabel = new JLabel(ElanLocale.getString(
                    "PreferencesDialog.Media.DefaultLoc"));
        setDirLabel.setFont(setDirLabel.getFont().deriveFont(Font.PLAIN));
        gbc.gridy = 5;
        gbc.gridwidth = 1;
        gbc.weightx = 0.5;
        gbc.insets = insets;
        add(setDirLabel, gbc);
        curDirLabel = new JLabel(curMediaLocation);
        curDirLabel.setFont(curDirLabel.getFont().deriveFont(Font.PLAIN));
        gbc.gridy = 6;
        add(curDirLabel, gbc);

        defaultDirButton = new JButton(ElanLocale.getString("Button.Browse"));
        gbc.gridx = 1;
        gbc.gridy = 5;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 0.0;
        gbc.gridheight = 2;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        add(defaultDirButton, gbc);
        defaultDirButton.addActionListener(this);

        resetDirButton = new JButton();

        ImageIcon resetIcon = null;

        // add reset icon
        try {
            resetIcon = new ImageIcon(this.getClass().getResource("/mpi/eudico/client/annotator/resources/Remove.gif"));
            resetDirButton.setIcon(resetIcon);
        } catch (Exception ex) {
            resetDirButton.setText("X");
        }

        resetDirButton.setToolTipText(ElanLocale.getString(
                "PreferencesDialog.Reset"));
        resetDirButton.setPreferredSize(new Dimension(
                resetDirButton.getPreferredSize().width,
                defaultDirButton.getPreferredSize().height));
        gbc.gridx = 2;
        gbc.gridy = 5;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 0.0;
        gbc.gridheight = 2;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = new Insets(2, 10, 2, 0);
        add(resetDirButton, gbc);
        resetDirButton.addActionListener(this);

        gbc.gridy = 7;
        gbc.gridwidth = 3;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weighty = 1.0;
        gbc.weightx = 1.0;
        add(new JPanel(), gbc); // filler
    }

    /**
     * Returns the changed pref.
     *
     * @return a map with the changed pref, or null
     */
    public Map getChangedPreferences() {
        if (isChanged()) {
            Map chMap = new HashMap(3);

            if (frameStepCB.isSelected() != origFrameStepToFrameBegin) {
                chMap.put("MediaNavigation.FrameStepToFrameBegin",
                    new Boolean(frameStepCB.isSelected()));
            }

            if (videosSameSizeCB.isSelected() != origVideoSameSize) {
                chMap.put("Media.VideosSameSize",
                    new Boolean(videosSameSizeCB.isSelected()));
            }

            if ((curDirLabel.getText() != null) &&
                    !curDirLabel.getText().equals("-")) {
                chMap.put("DefaultMediaLocation", curDirLabel.getText());
            } else {
                chMap.put("DefaultMediaLocation", null);
            }

            return chMap;
        }

        return null;
    }

    /**
     * Returns whether anything has changed.
     *
     * @return whether anything has changed
     */
    public boolean isChanged() {
        return ((frameStepCB.isSelected() != origFrameStepToFrameBegin) ||
        !curMediaLocation.equals(curDirLabel.getText()) ||
        (videosSameSizeCB.isSelected() != origVideoSameSize));
    }

    /**
     * Action event handling
     *
     * @param e the event
     */
    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == defaultDirButton) {
            // show a folder file chooser, set the current def. location
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(ElanLocale.getString(
                    "PreferencesDialog.Media.DefaultLoc"));

            File startDir = new File(System.getProperty("user.home"));

            if (curMediaLocation.length() > 1) {
                File dir = new File(FileUtility.urlToAbsPath(curMediaLocation));

                if (dir.exists() && dir.isDirectory()) {
                    startDir = dir;
                }
            }

            chooser.setCurrentDirectory(startDir);
            chooser.setMultiSelectionEnabled(false);
            chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

            int option = chooser.showDialog(this,
                    ElanLocale.getString("Button.Select"));

            if (option == JFileChooser.APPROVE_OPTION) {
                File selFile = chooser.getSelectedFile();

                if (selFile != null) {
                    curDirLabel.setText(selFile.getAbsolutePath());
                    curDirLabel.setText(FileUtility.pathToURLString(
                            selFile.getAbsolutePath()));
                }
            }
        } else if (e.getSource() == resetDirButton) {
            curDirLabel.setText("-");
        }
    }
}
