/*
 * File:     PlatformPanel.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.prefs.gui;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.prefs.PreferenceEditor;

import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;

import java.util.HashMap;
import java.util.Map;

import javax.swing.ButtonGroup;
import javax.swing.ImageIcon;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;


/**
 * A panel for OS specific preference settings.
 *
 * @author Han Sloetjes
 */
public class PlatformPanel extends JPanel implements PreferenceEditor {
    private JCheckBox macScreenBarCB;
    private boolean origMacUseScreenBar = false;
    private JCheckBox macLAndFCB;
    private boolean origMacLF = false;
    private JRadioButton cocoaQTB;
    private JRadioButton qtB;
    private String origMacPrefFramework = "CocoaQT";
    private boolean origPermDetached = false;
    private JCheckBox permDetachedCB;

    // windows
    private JRadioButton dsShowRB;
    private JRadioButton winQTRB;
    private JRadioButton jmfRB;
    private String origWinPrefFramework = "NativeWindows";

    /**
     * Creates a new PlatformPanel instance
     */
    public PlatformPanel() {
        super();
        readPrefs();
        initComponents();
    }

    private void readPrefs() {
        Object val = Preferences.get("OS.Mac.useScreenMenuBar", null);

        if (val instanceof Boolean) {
            origMacUseScreenBar = ((Boolean) val).booleanValue();
        }

        val = Preferences.get("UseMacLF", null);

        if (val instanceof Boolean) {
            origMacLF = ((Boolean) val).booleanValue();
        }

        val = Preferences.get("Mac.PrefMediaFramework", null);

        if (val instanceof String) {
            origMacPrefFramework = (String) val;
        }

        val = Preferences.get("PreferredMediaWindow", null);

        if (val instanceof Boolean) {
            origPermDetached = ((Boolean) val).booleanValue();
        }

        val = Preferences.get("Windows.PrefMediaFramework", null);

        if (val instanceof String) {
            origWinPrefFramework = (String) val;
        }
    }

    private void initComponents() {
        setLayout(new GridBagLayout());

        Insets insets = new Insets(2, 0, 2, 0);

        macScreenBarCB = new JCheckBox(ElanLocale.getString(
                    "PreferencesDialog.OS.Mac.ScreenMenuBar"));
        macScreenBarCB.setSelected(origMacUseScreenBar);

        Font plainFont = macScreenBarCB.getFont().deriveFont(Font.PLAIN);
        macScreenBarCB.setFont(plainFont);

        //JLabel macLabel = new JLabel(ElanLocale.getString("PreferencesDialog.OS.Mac"));
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        gbc.gridwidth = 2;
        gbc.insets = insets;
        add(new JLabel(ElanLocale.getString("PreferencesDialog.OS.Mac")), gbc);

        gbc.gridy = 1;
        gbc.gridwidth = 1;

        //gbc.fill = GridBagConstraints.NONE;
        //gbc.weightx = 0.0;
        add(macScreenBarCB, gbc);

        JLabel relaunchLabel = new JLabel();
        ImageIcon relaunchIcon = null;

        // add relaunch icon
        try {
            relaunchIcon = new ImageIcon(this.getClass().getResource("/toolbarButtonGraphics/general/Refresh16.gif"));
            relaunchLabel.setIcon(relaunchIcon);
        } catch (Exception ex) {
            relaunchLabel.setText(ElanLocale.getString(
                    "PreferencesDialog.Relaunch"));
        }

        relaunchLabel.setToolTipText(ElanLocale.getString(
                "PreferencesDialog.Relaunch.Tooltip"));
        macScreenBarCB.setToolTipText(ElanLocale.getString(
                "PreferencesDialog.Relaunch.Tooltip"));

        gbc.gridx = 1;
        gbc.gridwidth = 1;
        gbc.fill = GridBagConstraints.NONE;
        gbc.anchor = GridBagConstraints.EAST;
        gbc.weightx = 0.0;
        add(relaunchLabel, gbc);

        macLAndFCB = new JCheckBox(ElanLocale.getString(
                    "PreferencesDialog.OS.Mac.LF"));
        macLAndFCB.setSelected(origMacLF);
        macLAndFCB.setFont(plainFont);
        gbc.gridy = 2;
        gbc.gridx = 0;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        add(macLAndFCB, gbc);

        JLabel relaunchLabel2 = new JLabel();

        if (relaunchIcon != null) {
            relaunchLabel2.setIcon(relaunchIcon);
        } else {
            relaunchLabel2.setText(ElanLocale.getString(
                    "PreferencesDialog.Relaunch"));
        }

        relaunchLabel2.setToolTipText(ElanLocale.getString(
                "PreferencesDialog.Relaunch.Tooltip"));
        macLAndFCB.setToolTipText(ElanLocale.getString(
                "PreferencesDialog.Relaunch.Tooltip"));

        gbc.gridx = 1;
        gbc.gridwidth = 1;
        gbc.fill = GridBagConstraints.NONE;
        gbc.anchor = GridBagConstraints.EAST;
        gbc.weightx = 0.0;
        add(relaunchLabel2, gbc);

        JLabel frameworkLabel = new JLabel(ElanLocale.getString(
                    "Player.Framework"));
        frameworkLabel.setFont(plainFont);
        cocoaQTB = new JRadioButton(ElanLocale.getString(
                    "PreferencesDialog.Media.Cocoa"));
        cocoaQTB.setFont(plainFont);
        qtB = new JRadioButton(ElanLocale.getString(
                    "PreferencesDialog.Media.QTJ"));
        qtB.setFont(plainFont);

        if (origMacPrefFramework.equals("CocoaQT")) {
            cocoaQTB.setSelected(true);
        } else {
            qtB.setSelected(true);
        }

        ButtonGroup gr = new ButtonGroup();
        gr.add(cocoaQTB);
        gr.add(qtB);

        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.gridwidth = 2;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.weightx = 1.0;
        gbc.insets = new Insets(12, 0, 2, 0);
        add(frameworkLabel, gbc);

        gbc.gridy = 4;
        gbc.insets = insets;
        add(cocoaQTB, gbc);
        gbc.gridy = 5;
        add(qtB, gbc);

        permDetachedCB = new JCheckBox(ElanLocale.getString(
                    "PreferencesDialog.OS.Mac.DetachedMedia"), origPermDetached);
        permDetachedCB.setFont(plainFont);

        gbc.gridy = 6;
        add(permDetachedCB, gbc);

        // add Windows stuff
        gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        gbc.gridwidth = 2;
        gbc.gridy = 7;
        gbc.insets = new Insets(12, 0, 2, 0);
        add(new JLabel(ElanLocale.getString("PreferencesDialog.OS.Windows")),
            gbc);

        gbc.insets = insets;
        gbc.gridy = 8;

        JLabel winMedia = new JLabel(ElanLocale.getString("Player.Framework"));
        winMedia.setFont(plainFont);
        add(winMedia, gbc);

        ButtonGroup winBG = new ButtonGroup();
        dsShowRB = new JRadioButton(ElanLocale.getString(
                    "PreferencesDialog.Media.WMP"), true);
        winQTRB = new JRadioButton(ElanLocale.getString(
                    "PreferencesDialog.Media.QTJ"));
        jmfRB = new JRadioButton(ElanLocale.getString(
                    "PreferencesDialog.Media.JMF"));
        winBG.add(dsShowRB);
        winBG.add(winQTRB);
        winBG.add(jmfRB);

        if (origWinPrefFramework.equals("QT")) {
            winQTRB.setSelected(true);
        } else if (origWinPrefFramework.equals("JMF")) {
            jmfRB.setSelected(true);
        }

        dsShowRB.setFont(plainFont);
        winQTRB.setFont(plainFont);
        jmfRB.setFont(plainFont);

        gbc.insets = insets;
        gbc.gridy = 9;
        add(dsShowRB, gbc);
        gbc.gridy = 10;
        add(winQTRB, gbc);
        gbc.gridy = 11;
        add(jmfRB, gbc);

        gbc.gridy = 12;
        gbc.gridx = 0;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weighty = 1.0;
        add(new JPanel(), gbc); // filler
    }

    /**
     * @see mpi.eudico.client.annotator.prefs.PreferenceEditor#getChangedPreferences()
     */
    public Map getChangedPreferences() {
        if (isChanged()) {
            Map chMap = new HashMap(4);

            if (macScreenBarCB.isSelected() != origMacUseScreenBar) {
                chMap.put("OS.Mac.useScreenMenuBar",
                    new Boolean(macScreenBarCB.isSelected()));
            }

            if (macLAndFCB.isSelected() != origMacLF) {
                chMap.put("UseMacLF", new Boolean(macLAndFCB.isSelected()));
            }

            String tmp = "CocoaQT";

            if (qtB.isSelected()) {
                tmp = "QT";
            }

            if (!origMacPrefFramework.equals(tmp)) {
                chMap.put("Mac.PrefMediaFramework", tmp);

                //apply immediately
                System.setProperty("PreferredMediaFramework", tmp);
            }

            if (origPermDetached != permDetachedCB.isSelected()) {
                chMap.put("PreferredMediaWindow",
                    new Boolean(permDetachedCB.isSelected()));
            }

            String winTmp = "NativeWindows";

            if (winQTRB.isSelected()) {
                winTmp = "QT";
            } else if (jmfRB.isSelected()) {
                winTmp = "JMF";
            }

            if (!origWinPrefFramework.equals(winTmp)) {
                chMap.put("Windows.PrefMediaFramework", winTmp);

                //apply immediately
                System.setProperty("PreferredMediaFramework", winTmp);
            }

            return chMap;
        }

        return null;
    }

    /**
     * @see mpi.eudico.client.annotator.prefs.PreferenceEditor#isChanged()
     */
    public boolean isChanged() {
        if ((macScreenBarCB.isSelected() != origMacUseScreenBar) ||
                (macLAndFCB.isSelected() != origMacLF) ||
                (permDetachedCB.isSelected() != origPermDetached)) {
            return true;
        }

        String tmp = "CocoaQT";

        if (qtB.isSelected()) {
            tmp = "QT";
        }

        if (!origMacPrefFramework.equals(tmp)) {
            return true;
        }

        String winTmp = "NativeWindows";

        if (winQTRB.isSelected()) {
            winTmp = "QT";
        } else if (jmfRB.isSelected()) {
            winTmp = "JMF";
        }

        if (!origWinPrefFramework.equals(winTmp)) {
            return true;
        }

        return false;
    }
}
