/*
 * File:     Recognizer.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.recognizer.api;

import java.util.ArrayList;
import java.util.Locale;

import javax.swing.JPanel;


/**
 * Interface to be implemented by (audio) Recognizers that want to be hosted by ELAN
 * 
 * @author albertr
 *
 */
public interface Recognizer {
    /**
 * Set the mediaFile paths for the recognizer.
 * 
 * @param mediaFilePaths an ArrayList with full paths to the media files the recognizer must handle.
 * @return boolean to flag if the recognizer can(true) or can not(false) handle the media
 */
    public boolean setMedia(ArrayList mediaFilePaths);

    /**
 * Gives a name by which the recognizer can be chosen by a user among other recognizers.
 * 
 * @return the name for the recognizer
 */
    public String getName();

    /**
 * Will be called by software that controls the recognizer to give it a call back handle
 * The RecognizerHost Object has methods that allow the recognizer to communicate with
 * ELAN about selections an segmentations.
 * 
 * @param host the recognizerHost Object the Recognizer can communicate with
 */
    public void setRecognizerHost(RecognizerHost host);

    /**
 * This method can return a control panel for the recognizers parameters.
 * 
 * @return JPanel with the recognizers control elements on it or null if such a Panel does not exist
 */
    public JPanel getControlPanel();

    /**
 * Called to check if this recognizer needs example selections.
 * 
 * @return boolean that flags if the recognizer needs examples.
 */
    public boolean needsExamples();

    /**
 * Can be implemented optionally to support locale changes in the user interface
 * 
 * @param locale
 */
    public void updateLocale(Locale locale);

    /**
 * When this method is called the recognizer should start executing.
 * This method runs in a separate Thread that is controlled by ELAN.
 * 
 * Typically the recognizer then:
 * 1. Asks the RecognizerHost for the example selections
 * 2. Starts processing the data to produce segments
 *    and gives progress information to the RecognizerHost
 * 3. When processing is ready it sends the Segmentation(s) to the RecognizerHost
 *
 */
    public void start();

    /**
 * When this method is called the recognizer should stop executing.
 * It would be advisable to give the segmentations made up to that moment
 * to the RecognizerHost. 
 */
    public void stop();
}
