/*
 * File:     BoundarySegmentation.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.recognizer.data;

import java.util.*;


/**
 * Utility class used by SignalViewer to find Segment boundaries fast
 * 
 * There are no nested segments allowed so each segment must end before the beginning of the next segment
 * This implementation does not check the above constraint. 
 * 
 * @author albertr
 *
 */
public class BoundarySegmentation {
    private static long INFINITY = Long.MAX_VALUE;
    private TreeSet boundaries;

    /**
     * Creates a new BoundarySegmentation instance
     *
     * @param segmentation DOCUMENT ME!
     */
    public BoundarySegmentation(Segmentation segmentation) {
        boundaries = new TreeSet();

        ArrayList segments = segmentation.getSegments();

        for (int i = 0; i < segments.size(); i++) {
            Segment segment = (Segment) segments.get(i);
            Boundary begin = new Boundary(segment.beginTime, segment.label);

            // need to remove an eventual end boundary with the same time of the previous segment
            boundaries.remove(begin);

            // because add only adds if the element is not already present
            boundaries.add(begin);

            Boundary end = new Boundary(segment.endTime, "");
            boundaries.add(end);
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param time DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public long boundaryTimeBefore(long time) {
        Boundary bound = new Boundary(time, "");
        SortedSet head = boundaries.headSet(bound);

        if (head.size() > 0) {
            return ((Boundary) head.last()).time;
        } else {
            return 0;
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param time DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public long boundaryTimeAfter(long time) {
        Boundary bound = new Boundary(time, "");
        SortedSet tail = boundaries.tailSet(bound);

        if (tail.size() > 0) {
            return ((Boundary) tail.first()).time;
        } else {
            return INFINITY;
        }
    }

    /**
 * 
 * @param beginTime
 * @param endTime
 * @return null if there is no boundary between beginTime and endTime, label String otherwise (can be "")
 */
    public Boundary boundaryBetween(long beginTime, long endTime) {
        if (beginTime >= endTime) {
            return null;
        }

        Boundary from = new Boundary(beginTime, "");
        Boundary to = new Boundary(endTime, "");
        SortedSet sub = boundaries.subSet(from, to);

        if (sub.size() == 0) {
            return null;
        } else {
            return (Boundary) sub.last();
        }
    }
}
