/*
 * File:     Segmentation.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.recognizer.data;

import java.util.ArrayList;


/**
 * A segmentation consists of:
 *  a name
 *     an ArrayList of Segments
 *  an ArrayList of MediaDescriptors
 *  
 * The Name should be chosen unique if a recognizer produces more than one Segementation
 * 
 * The Segments should contain Segment objects that have no time overlap
 * 
 * Some recognizers work with more than one media file (combined video and audio)
 * or they use more than one channel (channel1 and channel2 in a stereo audio file)
 * These recognizer should add more than one MediaDescriptor to the Segmentation
 * 
 *  
 * @author albertr
 *
 */
public class Segmentation {
    private String name;
    private ArrayList segments;
    private ArrayList mediaDescriptors;

    /**
 * Construct a Segmentation
 * 
 * @param name the Segmentations name
 * @param segments the list of Segment objects
 * @param mediaFilePath the file path of the media file the recognizer used
 */
    public Segmentation(String name, ArrayList segments, String mediaFilePath) {
        this.name = name;
        this.segments = segments;

        MediaDescriptor descriptor = new MediaDescriptor();
        descriptor.mediaFilePath = mediaFilePath;
        descriptor.channel = 1;
        mediaDescriptors = new ArrayList();
        mediaDescriptors.add(descriptor);
    }

    /**
 * 
 * Construct a Segmentation
 * 
 * @param name the Segmentations name
 * @param segments the list of Segment objects
 * @param mediaFilePath the file path of the media file the recognizer used
 * @param channel the channel of the media file the recognizer used 
 */
    public Segmentation(String name, ArrayList segments, String mediaFilePath,
        int channel) {
        this.name = name;
        this.segments = segments;

        MediaDescriptor descriptor = new MediaDescriptor();
        descriptor.mediaFilePath = mediaFilePath;
        descriptor.channel = channel;
        mediaDescriptors = new ArrayList();
        mediaDescriptors.add(descriptor);
    }

    /**
 * Construct a Segmentation
 * 
 * @param name the Segmentations name
 * @param segments the list of Segment objects
 * @param descriptor the MediaDescriptor that describes the media file used by the recognizer
 */
    public Segmentation(String name, ArrayList segments,
        MediaDescriptor descriptor) {
        this.name = name;
        this.segments = segments;
        mediaDescriptors = new ArrayList();
        mediaDescriptors.add(descriptor);
    }

    /**
 * Add another MediaDescriptor to the Segmentation. 
 * Should be used for recognizers that use more than one media file or more than one channel
 * 
 * @param descriptor the MediaDescriptor that describes the media file used by the recognizer
 */
    public void addMediaDescriptor(MediaDescriptor descriptor) {
        mediaDescriptors.add(descriptor);
    }

    /**
 * 
 * @return the Segmentations name
 */
    public String getName() {
        return name;
    }

    /**
 * 
 * @return the ArrayList with Segment objects
 */
    public ArrayList getSegments() {
        return segments;
    }

    /**
 * 
 * @return the ArrayList with MediaDescriptor objects
 */
    public ArrayList getMediaDescriptors() {
        return mediaDescriptors;
    }
}
