/*
 * File:     AudioRecognizerPanel.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.recognizer.gui;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.ElanLocaleListener;
import mpi.eudico.client.annotator.Selection;
import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.client.annotator.recognizer.api.*;
import mpi.eudico.client.annotator.recognizer.data.*;

import mpi.eudico.client.util.WAVHeader;
import mpi.eudico.client.util.WAVSampler;

import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.awt.*;
import java.awt.event.*;

import java.util.*;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;


/**
 * DOCUMENT ME!
 * $Id: jalopy_gnu_src_dist.xml 16145 2009-08-24 11:55:06Z hasloe $
 * @author $Author$
 * @version $Revision$
 */
public class AudioRecognizerPanel extends JComponent implements ActionListener,
    Runnable, RecognizerHost, ElanLocaleListener {
    private ViewerManager2 viewerManager;
    private JPanel controlPanel;
    private JPanel progressPanel;
    private JPanel recognizerAndProgressPanel;
    private ArrayList audioFilePaths;
    private String visibleAudioFilePath;
    private SelectionPanel selectionPanel;
    private SegmentationPanel segmentationPanel;
    private JComboBox recognizerList;
    private HashMap recognizers;
    private Recognizer currentRecognizer;
    private HashMap segmentations;
    private JProgressBar progressBar;
    private JButton startStopButton;
    private boolean isRunning;
    private boolean notMono;

    /**
 *  TODO  update an initialize when another recognizer is chosen, segmentations.clear() etc etc.
 */
    public AudioRecognizerPanel(ViewerManager2 viewerManager,
        ArrayList audioFilePaths) {
        super();

        this.viewerManager = viewerManager;
        this.audioFilePaths = audioFilePaths;
        visibleAudioFilePath = (String) audioFilePaths.get(0);

        segmentations = new HashMap();

        try {
            WAVSampler wavSampler = new WAVSampler(visibleAudioFilePath);
            WAVHeader wavHeader = wavSampler.getWavHeader();
            notMono = wavHeader.getNumberOfChannels() > 1;
        } catch (Exception e) {
            e.printStackTrace();
        }

        try {
            setLayout(new BorderLayout());

            recognizerList = new JComboBox();
            recognizerList.addActionListener(this);
            add(recognizerList, BorderLayout.NORTH);

            JPanel selSegPanel = new JPanel();
            selSegPanel.setLayout(new BoxLayout(selSegPanel, BoxLayout.Y_AXIS));
            selectionPanel = new SelectionPanel(notMono, viewerManager);
            selSegPanel.add(selectionPanel);
            segmentationPanel = new SegmentationPanel(this, viewerManager);
            selSegPanel.add(segmentationPanel);
            add(selSegPanel, BorderLayout.WEST);

            // start/stop and progress information panel
            progressPanel = new JPanel();
            progressPanel.setLayout(new BoxLayout(progressPanel,
                    BoxLayout.X_AXIS));
            progressBar = new JProgressBar(0, 100);
            progressBar.setStringPainted(true);
            progressPanel.add(Box.createHorizontalStrut(15));
            progressPanel.add(progressBar);
            progressPanel.add(Box.createHorizontalStrut(10));
            startStopButton = new JButton(ElanLocale.getString(
                        "Recognizer.RecognizerPanel.Start"));
            startStopButton.addActionListener(this);
            progressPanel.add(startStopButton);

            recognizerAndProgressPanel = new JPanel();
            recognizerAndProgressPanel.setLayout(new BoxLayout(
                    recognizerAndProgressPanel, BoxLayout.Y_AXIS));
            add(recognizerAndProgressPanel, BorderLayout.CENTER);

            updateGui();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
 * Set the correct stereo mode for the selection panel
 * Fill the combo box with the available recognizers
 * Set the current selected recognizers gui on the Panel
 * @param stereoMode
 */
    private void updateGui() {
        selectionPanel.setStereoMode(notMono);

        if (currentRecognizer != null) {
            recognizerAndProgressPanel.removeAll();
            currentRecognizer = null;
        }

        recognizers = AvailabilityDetector.getAudioRecognizers(audioFilePaths);
        recognizerList.removeAllItems();

        Set names = recognizers.keySet();
        Iterator iter = names.iterator();

        while (iter.hasNext()) {
            recognizerList.addItem(iter.next());
        }

        if (recognizerList.getItemCount() > 0) {
            recognizerList.setSelectedIndex(0);
            currentRecognizer = (Recognizer) recognizers.get(recognizerList.getItemAt(
                        0));
            currentRecognizer.setRecognizerHost(this);

            controlPanel = currentRecognizer.getControlPanel();

            if (controlPanel != null) {
                recognizerAndProgressPanel.add(controlPanel);
            }

            recognizerAndProgressPanel.add(Box.createVerticalStrut(10));
            recognizerAndProgressPanel.add(progressPanel);
        } else {
            recognizerList.addItem(ElanLocale.getString(
                    "Recognizer.RecognizerPanel.No.Recognizers"));
        }

        progressBar.setString("");
        progressBar.setValue(0);

        validate();
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void actionPerformed(ActionEvent e) {
        Object source = e.getSource();

        if (source.equals(startStopButton)) {
            if (isRunning) {
                startStopButton.setText(ElanLocale.getString(
                        "Recognizer.RecognizerPanel.Start"));

                if (currentRecognizer != null) {
                    currentRecognizer.stop();
                }

                isRunning = false;
                progressBar.setString(ElanLocale.getString(
                        "Recognizer.RecognizerPanel.Canceled"));
            } else {
                if (currentRecognizer.needsExamples() &&
                        (getSelections().size() < 1)) {
                    JOptionPane.showMessageDialog(this,
                        ElanLocale.getString(
                            "Recognizer.RecognizerPanel.Warning.Selection"),
                        currentRecognizer.getName() + " " +
                        ElanLocale.getString(
                            "Recognizer.RecognizerPanel.Warning.Selection2"),
                        JOptionPane.PLAIN_MESSAGE);

                    return;
                }

                startStopButton.setText(ElanLocale.getString("Button.Cancel"));
                isRunning = true;
                progressBar.setString(ElanLocale.getString(
                        "Recognizer.RecognizerPanel.Recognizing"));
                viewerManager.getSignalViewer().setSegmentationChannel1(null);
                viewerManager.getSignalViewer().setSegmentationChannel2(null);
                new Thread(this).start();
            }
        } else if ((currentRecognizer != null) &&
                source.equals(recognizerList)) {
            if (isBusy()) {
                // tell the user that the current recognizer is still running
                JOptionPane.showMessageDialog(this,
                    currentRecognizer.getName() + ": " +
                    ElanLocale.getString(
                        "Recognizer.RecognizerPanel.Warning.Busy"),
                    currentRecognizer.getName() + " " +
                    ElanLocale.getString(
                        "Recognizer.RecognizerPanel.Warning.Busy2"),
                    JOptionPane.PLAIN_MESSAGE);

                // restore the current recognizers name in the combo box
                recognizerList.setSelectedItem(currentRecognizer.getName());

                return;
            }

            // reset the current segmentations
            segmentations = new HashMap();
            viewerManager.getSignalViewer().setSegmentationChannel1(null);
            viewerManager.getSignalViewer().setSegmentationChannel2(null);

            // remove current recognizer GUI
            recognizerAndProgressPanel.removeAll();

            // set the new current recognizer
            currentRecognizer = (Recognizer) recognizers.get(recognizerList.getSelectedItem());
            currentRecognizer.setRecognizerHost(this);

            // rebuild the recognizer GUI
            controlPanel = currentRecognizer.getControlPanel();

            if (controlPanel != null) {
                recognizerAndProgressPanel.add(controlPanel);
            }

            recognizerAndProgressPanel.add(progressPanel);
            progressBar.setString("");
            progressBar.setValue(0);

            validate();
        }
    }

    /**
 * This method will run in a separate Thread to decouple the recognizer from ELAN
 */
    public void run() {
        if (currentRecognizer != null) {
            currentRecognizer.start();
        }
    }

    /**
 * Takes care of giving running recognizers a chance to stop gracefully
 * ELAN should call this method before it quits
 *
 */
    public void stopRecognizers() {
        if (currentRecognizer != null) {
            currentRecognizer.stop();
        }
    }

    /**
 * Tells if there is one or more recognizer busy
 * 
 * @return true if there is some recognizing going on
 */
    public boolean isBusy() {
        return isRunning;
    }

    /**
     * DOCUMENT ME!
     */
    public void updateLocale() {
        selectionPanel.updateLocale();
        segmentationPanel.updateLocale();

        if (currentRecognizer != null) {
            currentRecognizer.updateLocale(ElanLocale.getLocale());
        }

        startStopButton.setText(ElanLocale.getString(
                "Recognizer.RecognizerPanel.Start"));
    }

    /**
 * @return the audioFilePath
 */
    public ArrayList getAudioFilePaths() {
        return audioFilePaths;
    }

    /**
 * @param audioFilePath the audioFilePath to set
 */
    public void setAudioFilePaths(ArrayList audioFilePaths) {
        this.audioFilePaths = audioFilePaths;
        visibleAudioFilePath = (String) audioFilePaths.get(0);

        try {
            WAVSampler wavSampler = new WAVSampler(visibleAudioFilePath);
            WAVHeader wavHeader = wavSampler.getWavHeader();
            notMono = wavHeader.getNumberOfChannels() > 1;
        } catch (Exception e) {
            e.printStackTrace();
        }

        segmentations = new HashMap();
        updateGui();
    }

    //
    // RecognizerHost interface implementation
    //

    /**
 * Add a segmentation to the 
 */
    public void addSegmentation(Segmentation segmentation) {
        segmentations.put(segmentation.getName(), segmentation);

        // make sure the SignalViewer knows about the new segmentation
        // CHECK IF THE MEDIA FILE IS THE VISIBLE ONE!!!
        int channel = 0;
        ArrayList mediaDescriptors = segmentation.getMediaDescriptors();

        for (int i = 0; i < mediaDescriptors.size(); i++) {
            MediaDescriptor descriptor = (MediaDescriptor) mediaDescriptors.get(i);
            channel += descriptor.channel;
        }

        if (channel == 1) {
            if (notMono) {
                viewerManager.getSignalViewer().setSegmentationChannel1(new BoundarySegmentation(
                        segmentation));
            } else {
                viewerManager.getSignalViewer().setSegmentation(new BoundarySegmentation(
                        segmentation));
            }
        } else if (channel == 2) {
            viewerManager.getSignalViewer().setSegmentationChannel2(new BoundarySegmentation(
                    segmentation));
        } else if (channel == 3) { // something for combined channel result?
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public ArrayList getSegmentations() {
        return new ArrayList(segmentations.values());
    }

    /**
 * By calling this method a recognizer gives information about the progress of its recognition task
 * 
 * @param progress  a float between 0 and 1 with 1 meaning that the task is completed
 */
    public void setProgress(float progress) {
        progressBar.setValue((int) (100 * progress));

        if (progress >= 1) {
            progressBar.setString(ElanLocale.getString(
                    "Recognizer.RecognizerPanel.Ready"));
            startStopButton.setText(ElanLocale.getString(
                    "Recognizer.RecognizerPanel.Start"));
            isRunning = false;
        }
    }

    /**
 * By calling this method the recognizer gets the selections for an audio signal 
 * 
 * @return An ArrayList with Selection Objects 
 */
    public ArrayList getSelections() {
        return selectionPanel.getSelections();
    }
}
