/*
 * File:     SelectionPanel.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.recognizer.gui;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Selection;
import mpi.eudico.client.annotator.ViewerManager2;

import mpi.util.TimeFormatter;

import java.awt.*;
import java.awt.event.*;

import java.util.*;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;


/**
 * DOCUMENT ME!
 * $Id: jalopy_gnu_src_dist.xml 16145 2009-08-24 11:55:06Z hasloe $
 * @author $Author$
 * @version $Revision$
 */
public class SelectionPanel extends JPanel implements ActionListener {
    private Selection selection;
    private ViewerManager2 viewerManager;
    private TitledBorder border;
    private JPanel buttonPanel;
    private JButton addSelection;
    private JButton addSelection1;
    private JButton addSelection2;
    private JButton removeSelection;
    private JList selectionList;
    private Vector selections;

    /**
     * Creates a new SelectionPanel instance
     *
     * @param stereoMode DOCUMENT ME!
     * @param viewerManager DOCUMENT ME!
     */
    public SelectionPanel(boolean stereoMode, ViewerManager2 viewerManager) {
        super();

        this.viewerManager = viewerManager;
        this.selection = viewerManager.getSelection();

        border = new TitledBorder(ElanLocale.getString(
                    "Recognizer.SelectionsPanel.Title"));
        setBorder(border);
        setLayout(new BorderLayout());

        buttonPanel = new JPanel();
        buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));

        addSelection = new JButton(ElanLocale.getString(
                    "Recognizer.SelectionsPanel.Add"));
        addSelection.addActionListener(this);

        addSelection1 = new JButton(ElanLocale.getString(
                    "Recognizer.SelectionsPanel.Add.Channel1"));
        addSelection1.addActionListener(this);

        addSelection2 = new JButton(ElanLocale.getString(
                    "Recognizer.SelectionsPanel.Add.Channel2"));
        addSelection2.addActionListener(this);

        removeSelection = new JButton(ElanLocale.getString(
                    "Recognizer.SelectionsPanel.Remove"));
        removeSelection.addActionListener(this);

        if (stereoMode) {
            buttonPanel.add(addSelection1);
            buttonPanel.add(addSelection2);
            buttonPanel.add(removeSelection);
        } else {
            buttonPanel.add(addSelection);
            buttonPanel.add(removeSelection);
        }

        add(buttonPanel, BorderLayout.NORTH);

        selections = new Vector();
        selectionList = new JList();
        selectionList.addMouseListener(new MouseHandler());
        selectionList.setBackground(getBackground());

        JScrollPane scrollPane = new JScrollPane(selectionList);
        add(scrollPane, BorderLayout.CENTER);

        // this forces the minimal width of the selection panel
        add(Box.createHorizontalStrut(200), BorderLayout.SOUTH);
    }

    /**
     * DOCUMENT ME!
     */
    public void updateLocale() {
        border.setTitle(ElanLocale.getString("Recognizer.SelectionsPanel.Title"));
        addSelection.setText(ElanLocale.getString(
                "Recognizer.SelectionsPanel.Add"));
        addSelection1.setText(ElanLocale.getString(
                "Recognizer.SelectionsPanel.Add.Channel1"));
        addSelection2.setText(ElanLocale.getString(
                "Recognizer.SelectionsPanel.Add.Channel2"));
        removeSelection.setText(ElanLocale.getString(
                "Recognizer.SelectionsPanel.Remove"));
    }

    /**
     * DOCUMENT ME!
     *
     * @param stereoMode DOCUMENT ME!
     */
    public void setStereoMode(boolean stereoMode) {
        if (stereoMode) {
            buttonPanel.removeAll();
            buttonPanel.add(addSelection1);
            buttonPanel.add(addSelection2);
            buttonPanel.add(removeSelection);
        } else {
            buttonPanel.removeAll();
            buttonPanel.add(addSelection);
            buttonPanel.add(removeSelection);
        }

        buttonPanel.getParent().getParent().validate();

        // remove old selections
        selections.clear();
        selectionList.removeAll();
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void actionPerformed(ActionEvent e) {
        Object source = e.getSource();

        if (source.equals(addSelection)) {
            if (selection.getBeginTime() < selection.getEndTime()) {
                String newSelection = " ";
                newSelection += TimeFormatter.toString(selection.getBeginTime());
                newSelection += " - ";
                newSelection += TimeFormatter.toString(selection.getEndTime());
                updateSelections(newSelection);
            }
        } else if (source.equals(addSelection1)) {
            if (selection.getBeginTime() < selection.getEndTime()) {
                String newSelection = " 1  ";
                newSelection += TimeFormatter.toString(selection.getBeginTime());
                newSelection += " - ";
                newSelection += TimeFormatter.toString(selection.getEndTime());
                updateSelections(newSelection);
            }
        } else if (source.equals(addSelection2)) {
            if (selection.getBeginTime() < selection.getEndTime()) {
                String newSelection = " 2  ";
                newSelection += TimeFormatter.toString(selection.getBeginTime());
                newSelection += " - ";
                newSelection += TimeFormatter.toString(selection.getEndTime());
                updateSelections(newSelection);
            }
        } else if (source.equals(removeSelection)) {
            if (selectionList.getSelectedIndex() >= 0) {
                selections.removeElementAt(selectionList.getSelectedIndex());
                selectionList.setListData(selections);
            }
        }
    }

    /**
 *  Makes sure the selections are ordered and refuse double entries
 *  
 * @param selection new selection String
 */
    private void updateSelections(String selection) {
        boolean inserted = false;

        for (int i = 0; i < selections.size(); i++) {
            if (selection.equals((String) selections.get(i))) {
                inserted = true; // nothing to do, this selection is already in the list

                break;
            } else if (selection.compareTo((String) selections.get(i)) < 0) { // ordering based on begin time string
                selections.insertElementAt(selection, i);
                inserted = true;

                break;
            }
        }

        if (!inserted) { // add selection to the end of the list
            selections.add(selection);
        }

        // update the GUI list
        selectionList.setListData(selections);
    }

    /**
 * 
 * @return an ArrayList with recognizer.data.Selection Objects
 */
    public ArrayList getSelections() {
        ArrayList selectionObjects = new ArrayList();

        for (int i = 0; i < selections.size(); i++) {
            mpi.eudico.client.annotator.recognizer.data.Selection selection = new mpi.eudico.client.annotator.recognizer.data.Selection();
            String selectionString = (String) selections.get(i);
            String[] parts = selectionString.split("-");

            if (parts[0].trim().split("\\s+").length > 1) {
                String[] extraParts = parts[0].trim().split("\\s+");
                selection.channel = Integer.parseInt(extraParts[0].trim());
                selection.beginTime = TimeFormatter.toMilliSeconds(extraParts[1].trim());
                selection.endTime = TimeFormatter.toMilliSeconds(parts[1].trim());
            } else {
                selection.channel = 1; // mono signal is in channel1
                selection.beginTime = TimeFormatter.toMilliSeconds(parts[0].trim());
                selection.endTime = TimeFormatter.toMilliSeconds(parts[1].trim());
            }

            selectionObjects.add(selection);
        }

        return selectionObjects;
    }

    /**
     * DOCUMENT ME!
     * $Id: jalopy_gnu_src_dist.xml 16145 2009-08-24 11:55:06Z hasloe $
     * @author $Author$
     * @version $Revision$
     */
    public class MouseHandler extends MouseAdapter {
        /**
         * DOCUMENT ME!
         *
         * @param event DOCUMENT ME!
         */
        public void mouseClicked(MouseEvent event) {
            // set the selection to the double clicked entry in the JList
            if (event.getClickCount() > 1) {
                String targetSelection = ((String) (selections.get(selectionList.getSelectedIndex()))).trim();

                if (targetSelection.startsWith("1") ||
                        targetSelection.startsWith("2")) {
                    targetSelection = targetSelection.substring(1);
                }

                String[] parts = targetSelection.split("-");
                long beginTime = TimeFormatter.toMilliSeconds(parts[0].trim());
                long endTime = TimeFormatter.toMilliSeconds(parts[1].trim());
                selection.setSelection(beginTime, endTime);
                viewerManager.getMasterMediaPlayer().setMediaTime(beginTime);
            }
        }
    }
}
