/*
 * File:     SilenceRecognizer.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.recognizer.silence;

import mpi.eudico.client.annotator.recognizer.api.Recognizer;
import mpi.eudico.client.annotator.recognizer.api.RecognizerHost;
import mpi.eudico.client.annotator.recognizer.data.*;

import mpi.eudico.client.util.WAVHeader;
import mpi.eudico.client.util.WAVSampler;

import java.util.*;

import javax.swing.JPanel;


/**
 * 
 * @author albertr
 *
 */
public class SilenceRecognizer implements Recognizer {
    /** Holds value of property DOCUMENT ME! */
    private final static int SILENCE = -1;

    /** Holds value of property DOCUMENT ME! */
    private final static int NON_SILENCE = 1;

    /** Holds value of property DOCUMENT ME! */
    private final static String SILENCE_LABEL = "s";

    /** Holds value of property DOCUMENT ME! */
    private final static String NON_SILENCE_LABEL = "x";

    /** Holds value of property DOCUMENT ME! */
    private final static int stepDuration = 20;
    private int nSteps;
    private RecognizerHost host;
    private SilenceRecognizerPanel controlPanel;
    private String currentMediaFilePath;
    private WAVSampler sampler;
    private int nrOfChannels;
    private int sampleFrequency;
    private long nrOfSamples;
    private float duration;

    /** Holds value of property DOCUMENT ME! */
    boolean canHandleMedia;
    private long sampleBufferBeginTime;
    private int sampleBufferDuration;
    private boolean keepRunning;
    private float[] averageEnergy1;
    private int[] samples1;
    private float[] averageEnergy2;
    private int[] samples2;

    /**
 * Lightweight constructor, try to do as little as possible here
 *
 */
    public SilenceRecognizer() {
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public boolean needsExamples() {
        return true;
    }

    /**
 * Called by RecognizerHost to get a name for this recognizer in the ComboBox with available recognizers
 * 
 * @return the name of this recognizer
 */
    public String getName() {
        return "Silence Recognizer";
    }

    /**
 * Called by RecognizerHost to get a control panel for this recognizers parameters
 * 
 * @return a JPanel with the recognizers GUI controls or null if there are no controls
 */
    public JPanel getControlPanel() {
        if (controlPanel == null) {
            controlPanel = new SilenceRecognizerPanel();
        }

        return controlPanel;
    }

    /**
 * Called by RecognizerHost to set the media for this recognizer and to see if this recognizer can handle the media
 * 
 * @param mediaFilePath String that contains the full path to the media file
 * @return true if this recognizer can handle the media, false otherwise
 */
    public boolean setMedia(ArrayList mediaFilePaths) {
        if (mediaFilePaths == null) {
            return false;
        }

        currentMediaFilePath = (String) mediaFilePaths.get(0); // only uses one file, the visible one
        canHandleMedia = false;
        averageEnergy1 = null;
        averageEnergy2 = null;

        try {
            sampler = new WAVSampler(currentMediaFilePath);
            nrOfChannels = sampler.getWavHeader().getNumberOfChannels();
            System.out.println("Nr. of channels: " + nrOfChannels);
            sampleFrequency = sampler.getSampleFrequency();
            nrOfSamples = sampler.getNrOfSamples();
            duration = sampler.getDuration();
            nSteps = (int) (duration / stepDuration);
            sampleBufferBeginTime = -1;
            sampleBufferDuration = 0;
            canHandleMedia = true;
        } catch (Exception e) {
            //e.printStackTrace();
        }

        return canHandleMedia;
    }

    /**
 * Called by RecognizerHost to give this recognizer an object for callbacks
 * 
 * @param host the RecognizerHost that talks with this recognizer
 */
    public void setRecognizerHost(RecognizerHost host) {
        this.host = host;
    }

    /**
 * Fills an array with sample values for a certain time interval and a certain channel.
 * This method was needed to optimize the WAVReaderthat is rather slow for small time steps
 * 
 * Pad with zeros if you read more samples than available in the media file
 * 
 * @param from interval begin time in milliseconds
 * @param to interval end time in milliseconds
 * @param channel the audio channel from which the samples must be read
 * @param samples the int[] array that must be filled with the samples
 */
    private void getSamples(long from, long to, int channel, int[] samples) {
        try {
            // check if the requested samples are in the buffer
            long sampleBufferEndTime = sampleBufferBeginTime +
                sampleBufferDuration;

            if ((from < sampleBufferBeginTime) ||
                    (from >= sampleBufferEndTime) ||
                    (to < sampleBufferBeginTime) ||
                    (to >= sampleBufferEndTime)) {
                sampleBufferDuration = 10000;

                while ((to - from) > sampleBufferDuration) {
                    sampleBufferDuration += 1000;
                }

                int nSamples = (sampleBufferDuration * sampleFrequency) / 1000;
                sampleBufferBeginTime = from;
                sampler.seekTime(sampleBufferBeginTime);
                sampler.readInterval(nSamples, nrOfChannels);
            }

            Arrays.fill(samples, 0);

            int srcPos = (int) (((from - sampleBufferBeginTime) * sampleFrequency) / 1000);
            int length = (int) (((to - from) * sampleFrequency) / 1000);

            if (channel == 1) {
                System.arraycopy(sampler.getFirstChannelArray(), srcPos,
                    samples, 0, length);
            } else {
                System.arraycopy(sampler.getSecondChannelArray(), srcPos,
                    samples, 0, length);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
 * 
 * @param samples
 * @return
 */
    private float max(int[] samples) {
        int max = Integer.MIN_VALUE;

        for (int i = 0; i < samples.length; i++) {
            if (max < samples[i]) {
                max = samples[i];
            }
        }

        return max;
    }

    /**
 * 
 * @param samples
 * @return
 */
    private int zeroCrossings(int[] samples) {
        int crossings = 0;

        for (int i = 0; (i + 1) < samples.length; i++) {
            if ((samples[i] * samples[i + 1]) < 0) {
                crossings++;
            }
        }

        return crossings;
    }

    /**
 * 
 * @param samples
 * @return
 */
    private float averageEnergy(int[] samples) {
        if (samples.length == 0) {
            return 0;
        }

        double average = 0;

        for (int i = 0; i < samples.length; i++) {
            int sample = samples[i];
            average += (sample * sample);
        }

        return (float) Math.sqrt(average / samples.length);
    }

    private float averageEnergy(int channel, int step) {
        if (channel == 1) {
            if (averageEnergy1 == null) {
                averageEnergy1 = new float[nSteps];

                for (int i = 0; i < nSteps; i++) {
                    averageEnergy1[i] = -1;
                }

                samples1 = new int[(sampleFrequency * stepDuration) / 1000];
            }

            if (averageEnergy1[step] < 0) {
                long time = step * stepDuration;
                getSamples(time, time + stepDuration, 1, samples1);
                averageEnergy1[step] = averageEnergy(samples1);
            }

            return averageEnergy1[step];
        } else if (channel == 2) {
            if (averageEnergy2 == null) {
                averageEnergy2 = new float[nSteps];

                for (int i = 0; i < nSteps; i++) {
                    averageEnergy2[i] = -1;
                }

                samples2 = new int[(sampleFrequency * stepDuration) / 1000];
            }

            if (averageEnergy2[step] < 0) {
                long time = step * stepDuration;
                getSamples(time, time + stepDuration, 2, samples2);
                averageEnergy2[step] = averageEnergy(samples2);
            }

            return averageEnergy2[step];
        } else {
            return 0;
        }
    }

    /**
 * Called by RecognizerHost to start the recognizer
 *
 */
    public void start() {
        keepRunning = true;
        recog();
    }

    /**
 * Called by RecognizerHost to stop the recognizer, MUST BE OBEYED AT ALL TIMES
 *
 */
    public void stop() {
        keepRunning = false;
    }

    /**
 * 
 *
 */
    public void recog() {
        float maxEnergyChannel1 = Integer.MIN_VALUE;
        float maxEnergyChannel2 = Integer.MIN_VALUE;
        int[] samples = new int[(sampleFrequency * stepDuration) / 1000];
        ArrayList selections = host.getSelections();

        for (int i = 0; i < selections.size(); i++) {
            Selection selection = (Selection) selections.get(i);

            for (long time = selection.beginTime; time < selection.endTime;
                    time += stepDuration) {
                long to = time + stepDuration;

                if (to > selection.endTime) {
                    to = selection.endTime;
                }

                if (selection.channel == 1) {
                    getSamples(time, to, 1, samples);

                    float energy = averageEnergy(samples);

                    if (energy > maxEnergyChannel1) {
                        maxEnergyChannel1 = energy;
                    }
                } else {
                    getSamples(time, to, 2, samples);

                    float energy = averageEnergy(samples);

                    if (energy > maxEnergyChannel2) {
                        maxEnergyChannel2 = energy;
                    }
                }
            }
        }

        boolean lookAtChannel1 = maxEnergyChannel1 > Integer.MIN_VALUE;
        boolean lookAtChannel2 = maxEnergyChannel2 > Integer.MIN_VALUE;

        int nSteps = (int) (duration / stepDuration);
        int[] steps1 = new int[nSteps];
        int[] steps2 = new int[nSteps];

        for (int step = 0; step < nSteps; step++) {
            long time = step * stepDuration;
            host.setProgress(time / duration);

            if (lookAtChannel1) {
                //getSamples(time, time + stepDuration, 1, samples);
                //steps1[step] = averageEnergy(samples) < maxEnergyChannel1 ? SILENCE : NON_SILENCE;
                steps1[step] = (averageEnergy(1, step) < maxEnergyChannel1)
                    ? SILENCE : NON_SILENCE;
            }

            if (lookAtChannel2) {
                //getSamples(time, time + stepDuration, 2, samples);
                //steps2[step] = averageEnergy(samples) < maxEnergyChannel2 ? SILENCE : NON_SILENCE;
                steps2[step] = (averageEnergy(2, step) < maxEnergyChannel2)
                    ? SILENCE : NON_SILENCE;
            }

            if (!keepRunning) {
                break;
            }
        }

        // if keepRunning is still true make sure the progress is set to 1
        if (keepRunning) {
            host.setProgress(1);
        }

        // prune 
        if (lookAtChannel1) {
            prune(steps1, stepDuration);
        }

        if (lookAtChannel2) {
            prune(steps2, stepDuration);
        }

        // create the segments
        if (lookAtChannel1) {
            ArrayList segments = createSegmentation(steps1, stepDuration);
            MediaDescriptor descriptor = new MediaDescriptor(currentMediaFilePath,
                    1);
            Segmentation seg = new Segmentation("Channel1", segments, descriptor);
            host.addSegmentation(seg);
        }

        if (lookAtChannel2) {
            ArrayList segments = createSegmentation(steps2, stepDuration);
            MediaDescriptor descriptor = new MediaDescriptor(currentMediaFilePath,
                    2);
            Segmentation seg = new Segmentation("Channel2", segments, descriptor);
            host.addSegmentation(seg);
        }
    }

    private void prune(int[] steps, int stepDuration) {
        //     S  NS  S  ->  S  S  S
        for (int step = 1; (step + 1) < steps.length; step++) {
            if ((steps[step] > 0) && (steps[step - 1] < 0) &&
                    (steps[step + 1] < 0)) {
                steps[step] = -1;
            }
        }

        //    S  S  NS  NS  S  S  ->  S  S  S  S  S  S
        for (int step = 2; (step + 3) < steps.length; step++) {
            if ((steps[step] > 0) && (steps[step + 1] > 0) &&
                    (steps[step - 1] < 0) && (steps[step - 2] < 0) &&
                    (steps[step + 2] < 0) && (steps[step + 3] < 0)) {
                steps[step] = -1;
                steps[step + 1] = -1;
            }
        }

        //  NS  S  NS  ->  NS  NS  NS
        for (int step = 1; (step + 1) < steps.length; step++) {
            if ((steps[step] < 0) && (steps[step - 1] > 0) &&
                    (steps[step + 1] > 0)) {
                steps[step] = 1;
            }
        }

        //  NS  NS  S  S  NS  NS  ->  NS  NS  NS  NS  NS  NS
        for (int step = 2; (step + 3) < steps.length; step++) {
            if ((steps[step] < 0) && (steps[step + 1] < 0) &&
                    (steps[step - 1] > 0) && (steps[step - 2] > 0) &&
                    (steps[step + 2] > 0) && (steps[step + 3] > 0)) {
                steps[step] = 1;
                steps[step + 1] = 1;
            }
        }

        // remove NON_SILENCE patterns that are too short
        int minimalNonSilenceSteps = 1 +
            (controlPanel.getMinimalNonSilenceDuration() / stepDuration);

        for (int step = 0; step < steps.length; step++) {
            if (steps[step] >= NON_SILENCE) {
                int to = step + 1;

                while ((to < steps.length) && (steps[to] >= NON_SILENCE)) {
                    to++;
                }

                if ((to - step) < minimalNonSilenceSteps) {
                    for (int j = step; j < to; j++) {
                        steps[j] = SILENCE;
                    }
                }

                step = to - 1;
            }
        }

        // remove SILENCE patterns that are too short
        int minimalSilenceSteps = 1 +
            (controlPanel.getMinimalSilenceDuration() / stepDuration);

        for (int step = 0; step < steps.length; step++) {
            if (steps[step] <= SILENCE) {
                int to = step + 1;

                while ((to < steps.length) && (steps[to] <= SILENCE)) {
                    to++;
                }

                if ((to - step) < minimalSilenceSteps) {
                    for (int j = step; j < to; j++) {
                        steps[j] = NON_SILENCE;
                    }
                }

                step = to - 1;
            }
        }
    }

    private ArrayList createSegmentation(int[] steps, int stepDuration) {
        ArrayList segments = new ArrayList();
        Segment segment = new Segment();
        segment.beginTime = 0;

        int current = steps[0];

        if (current <= SILENCE) {
            segment.label = SILENCE_LABEL;
        } else {
            segment.label = NON_SILENCE_LABEL;
        }

        for (int step = 1; step < steps.length; step++) {
            if (steps[step] != current) {
                segment.endTime = step * stepDuration;
                segments.add(segment);
                segment = new Segment();
                segment.beginTime = step * stepDuration;
                current = steps[step];

                if (current <= SILENCE) {
                    segment.label = SILENCE_LABEL;
                } else {
                    segment.label = NON_SILENCE_LABEL;
                }
            }
        }

        segment.endTime = (long) duration;
        segments.add(segment);

        return segments;
    }

    /**
     * DOCUMENT ME!
     *
     * @param locale DOCUMENT ME!
     */
    public void updateLocale(Locale locale) {
        if (controlPanel != null) {
            controlPanel.updateLocale();
        }
    }
}
