/*
 * File:     ExportResultDialog.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.search.viewer;

import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.export.AbstractBasicExportDialog;

import mpi.eudico.client.annotator.search.result.viewer.ContentMatch2TabDelimitedText;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileExtension;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.search.content.query.model.ContentQuery;

import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.event.ActionListener;

import java.io.File;
import java.io.IOException;

import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSeparator;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;


/**
 * Dialog to chose file type and parameters for the export of a result
 *
 * $Id: ExportResultDialog.java 16974 2009-11-17 08:36:48Z hasloe $
 *
 * @author $author$
 * @version $Revision$
 */
public class ExportResultDialog extends AbstractBasicExportDialog
    implements ActionListener {
    /** Holds value of property DOCUMENT ME! */
    private final ButtonGroup fileTypeGroup = new ButtonGroup();

    /** Holds value of property DOCUMENT ME! */
    private final ButtonGroup formatTypeGroup = new ButtonGroup();

    /** Holds value of property DOCUMENT ME! */
    private final ContentQuery query;

    /** insets for ui components */
    private final JLabel dataFormatLabel = new JLabel();

    /** Holds value of property DOCUMENT ME! */
    private final JLabel fileFormatLabel = new JLabel();

    /** Holds value of property DOCUMENT ME! */
    private final JRadioButton asTableButton = new JRadioButton();

    /** Holds value of property DOCUMENT ME! */
    private final JRadioButton asTreeButton = new JRadioButton();

    /** Holds value of property DOCUMENT ME! */
    private final JRadioButton htmlButton = new JRadioButton();

    /** Holds value of property DOCUMENT ME! */
    private final JRadioButton tabButton = new JRadioButton();

    /**
     * Creates a new ExportElanMatchDialog object.
     *
     * @param parent parent window
     * @param modal should be always true
     * @param transcription transcription
     * @param query query (containing result)
     */
    public ExportResultDialog(Frame parent, boolean modal,
        Transcription transcription, ContentQuery query) {
        super(parent, modal, transcription);
        this.query = query;
        makeLayout();
        updateLocale();
        postInit();
    }

    /**
     * DOCUMENT ME!
     */
    protected void makeLayout() {
        super.makeLayout();
        getContentPane().setLayout(new GridBagLayout());
        fileTypeGroup.add(tabButton);
        fileTypeGroup.add(htmlButton);
        formatTypeGroup.add(asTableButton);
        formatTypeGroup.add(asTreeButton);

        JPanel fileFormatPanel = new JPanel(new GridLayout(0, 1));
        fileFormatPanel.add(tabButton);
        fileFormatPanel.add(htmlButton);

        JPanel dataFormatPanel = new JPanel(new GridLayout(0, 1));
        dataFormatPanel.add(asTableButton);
        dataFormatPanel.add(asTreeButton);

        optionsPanel.setLayout(new GridBagLayout());

        GridBagConstraints c = new GridBagConstraints();
        c.gridx = GridBagConstraints.REMAINDER;
        c.weightx = 1.0;
        c.weighty = 0.25;
        c.fill = GridBagConstraints.BOTH;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;

        optionsPanel.add(fileFormatLabel, c);
        c.weighty = 0.5;
        optionsPanel.add(fileFormatPanel, c);
        c.weighty = 0;
        optionsPanel.add(new JSeparator(), c);
        c.weighty = 0.25;
        optionsPanel.add(dataFormatLabel, c);
        c.weighty = 0.5;
        optionsPanel.add(dataFormatPanel, c);

        c.anchor = GridBagConstraints.CENTER;
        c.weighty = 0.0;
        getContentPane().add(titleLabel, c);
        c.fill = GridBagConstraints.BOTH;
        c.weighty = 1.0;
        getContentPane().add(optionsPanel, c);
        c.weighty = 0.0;
        c.fill = GridBagConstraints.NONE;
        getContentPane().add(buttonPanel, c);

        asTreeButton.setVisible(query.getAnchorConstraint().getChildCount() > 0);
        asTreeButton.setEnabled(false);

        htmlButton.addChangeListener(new ChangeListener() {
                public void stateChanged(ChangeEvent e) {
                    if (tabButton.isSelected()) {
                        asTableButton.setSelected(true);
                    }

                    asTreeButton.setEnabled(htmlButton.isSelected());
                }
            });
        tabButton.setSelected(true);
        asTableButton.setSelected(true);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     */
    protected boolean startExport() throws IOException {
        File exportFile = null;

        if (tabButton.isSelected()) {
            exportFile = promptForFile(ElanLocale.getString(
                        "ExportResultDialog.Title"), FileExtension.TEXT_EXT,
                    ElanFileFilter.createFileFilter(ElanFileFilter.TEXT_TYPE),
                    true);
        } else {
            exportFile = promptForFile(ElanLocale.getString(
                        "ExportResultDialog.Title"), FileExtension.HTML_EXT,
                    ElanFileFilter.createFileFilter(ElanFileFilter.HTML_TYPE),
                    true);
        }

        if (exportFile == null) {
            return false;
        }

        if (tabButton.isSelected()) {
            ContentMatch2TabDelimitedText.exportMatches(query.getResult()
                                                             .getMatches(),
                exportFile, encoding);
        } else {
            ElanQuery2HTML.exportQuery(query, exportFile,
                asTableButton.isSelected(), transcription.getName(), encoding);
        }

        return true;
    }

    /**
     * Set the localized text on ui elements
     */
    protected void updateLocale() {
        super.updateLocale();
        setTitle(ElanLocale.getString("ExportResultDialog.Title"));
        titleLabel.setText(ElanLocale.getString("ExportResultDialog.Title"));
        asTableButton.setText(ElanLocale.getString("ExportResultDialog.AsTable"));
        asTreeButton.setText(ElanLocale.getString("ExportResultDialog.AsTree"));
        htmlButton.setText(ElanLocale.getString(
                "ExportDialog.FileDescription.Html"));
        tabButton.setText(ElanLocale.getString(
                "ExportResultDialog.TabDelimitedText"));

        optionsPanel.setBorder(new TitledBorder(ElanLocale.getString(
                    "ExportDialog.Label.Options")));
        fileFormatLabel.setText(ElanLocale.getString(
                "ExportResultDialog.FileFormat"));
        dataFormatLabel.setText(ElanLocale.getString(
                "ExportResultDialog.ExportMatches"));
    }
}
