/*
 * File:     AnnotationsFromGapsDlg.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.tier;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.commands.Command;
import mpi.eudico.client.annotator.commands.ELANCommandFactory;

import mpi.eudico.client.annotator.gui.ClosableDialog;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.HeadlessException;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;


/**
 * A dialog for customizing the annotations from gaps action.
 *
 * @author Han Sloetjes
 */
public class AnnotationsFromGapsDlg extends ClosableDialog
    implements ActionListener, ChangeListener {
    private TranscriptionImpl transcription;
    private long mediaDuration;

    // ui elements
    //private JPanel titlePanel;
    private JPanel tierPanel;
    private JPanel optionsPanel;
    private JPanel buttonPanel;
    private JButton closeButton;
    private JButton startButton;
    private JLabel titleLabel;
    private JList tierList;
    private DefaultListModel model;
    private JRadioButton onSameTierRB;
    private JRadioButton onNewTierRB;
    private JLabel tierNameLabel;
    private JTextField tierNameTF;
    private JRadioButton emptyRB;
    private JRadioButton durationRB;
    private JRadioButton valueRB;
    private JRadioButton msRB;
    private JRadioButton secRB;
    private JRadioButton hourRB;
    private JTextField annValueTF;

    /**
     * Creates a new AnnotationsFromGapsDlg instance
     *
     * @param owner the parent frame
     * @param transcription the transcription
     * @param mediaDuration the master media duration
     *
     * @throws HeadlessException
     */
    public AnnotationsFromGapsDlg(Frame owner, TranscriptionImpl transcription,
        long mediaDuration) throws HeadlessException {
        super(owner);
        this.transcription = transcription;
        this.mediaDuration = mediaDuration;
        initComponents();
        extractTiers();
        postInit();
    }

    /**
     * Pack, size and set location.
     */
    private void postInit() {
        pack();

        //setSize((getSize().width < minimalWidth) ? minimalWidth : getSize().width,
        //    (getSize().height < minimalHeight) ? minimalHeight : getSize().height);
        setLocationRelativeTo(getParent());

        //setResizable(false);
    }

    private void initComponents() {
        //titlePanel = new JPanel();
        tierPanel = new JPanel();
        optionsPanel = new JPanel();
        buttonPanel = new JPanel();
        startButton = new JButton();
        closeButton = new JButton();
        titleLabel = new JLabel();
        model = new DefaultListModel();
        tierList = new JList(model);
        tierList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        onSameTierRB = new JRadioButton();
        onSameTierRB.setSelected(true);
        onSameTierRB.addChangeListener(this);
        onNewTierRB = new JRadioButton();
        onNewTierRB.addChangeListener(this);
        tierNameLabel = new JLabel();
        tierNameTF = new JTextField();
        tierNameTF.setEnabled(false);
        emptyRB = new JRadioButton();
        emptyRB.setSelected(true);
        emptyRB.addChangeListener(this);
        durationRB = new JRadioButton();
        durationRB.addChangeListener(this);
        valueRB = new JRadioButton();
        valueRB.addChangeListener(this);
        msRB = new JRadioButton(ElanLocale.getString("TimeCodeFormat.MilliSec"));
        msRB.setSelected(true);
        msRB.setEnabled(false);
        secRB = new JRadioButton(ElanLocale.getString("TimeCodeFormat.Seconds"));
        secRB.setEnabled(false);
        hourRB = new JRadioButton(ElanLocale.getString(
                    "TimeCodeFormat.TimeCode"));
        hourRB.setEnabled(false);
        annValueTF = new JTextField(); // should enable input methods!?
        annValueTF.setEnabled(false);

        ButtonGroup bg = new ButtonGroup();
        bg.add(msRB);
        bg.add(secRB);
        bg.add(hourRB);

        ButtonGroup bg2 = new ButtonGroup();
        bg2.add(onSameTierRB);
        bg2.add(onNewTierRB);

        ButtonGroup bg3 = new ButtonGroup();
        bg3.add(emptyRB);
        bg3.add(durationRB);
        bg3.add(valueRB);

        getContentPane().setLayout(new GridBagLayout());

        Insets insets = new Insets(2, 6, 2, 6);

        titleLabel = new JLabel();
        titleLabel.setFont(titleLabel.getFont().deriveFont((float) 16));
        titleLabel.setHorizontalAlignment(SwingConstants.CENTER);

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.weightx = 1.0;
        getContentPane().add(titleLabel, gbc);

        tierPanel.setLayout(new GridBagLayout());

        JScrollPane tierScroll = new JScrollPane(tierList);
        tierScroll.setPreferredSize(new Dimension(450, 100));

        gbc = new GridBagConstraints();
        gbc.fill = GridBagConstraints.BOTH;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        tierPanel.add(tierScroll, gbc);

        gbc.gridy = 1;
        getContentPane().add(tierPanel, gbc);

        optionsPanel.setLayout(new GridBagLayout());

        //insets.bottom = 3;
        JLabel destLabel = new JLabel(ElanLocale.getString(
                    "FillGapsDialog.Label.Destination"));

        gbc = new GridBagConstraints();
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        gbc.weightx = 1.0;
        gbc.gridwidth = 3;
        optionsPanel.add(destLabel, gbc);

        gbc.gridy = 1;
        gbc.insets = new Insets(2, 16, 2, 6);
        optionsPanel.add(onSameTierRB, gbc);

        gbc.gridy = 2;
        gbc.gridwidth = 1;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 0.0;
        optionsPanel.add(onNewTierRB, gbc);

        gbc.gridx = 1;
        gbc.gridwidth = 2;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        optionsPanel.add(tierNameTF, gbc);

        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        gbc.insets = new Insets(24, 6, 2, 6);
        optionsPanel.add(new JLabel(ElanLocale.getString(
                    "FillGapsDialog.Label.AnnotationValue")), gbc);

        gbc.gridx = 0;
        gbc.gridy = 4;
        gbc.gridwidth = 1;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 0.0;
        gbc.insets = new Insets(2, 16, 2, 6);
        optionsPanel.add(valueRB, gbc);

        gbc.gridx = 1;
        gbc.gridwidth = 2;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;

        //gbc.insets = insets;
        optionsPanel.add(annValueTF, gbc);

        gbc.gridx = 0;
        gbc.gridy = 5;
        gbc.gridwidth = 3;

        //gbc.insets = insets;
        optionsPanel.add(durationRB, gbc);

        gbc.gridy = 6;
        gbc.insets = new Insets(2, 32, 2, 6);
        optionsPanel.add(msRB, gbc);
        gbc.gridy = 7;
        optionsPanel.add(secRB, gbc);
        gbc.gridy = 8;
        optionsPanel.add(hourRB, gbc);

        gbc.gridy = 9;
        gbc.insets = new Insets(2, 16, 2, 6);
        optionsPanel.add(emptyRB, gbc);

        gbc = new GridBagConstraints();
        gbc.fill = GridBagConstraints.BOTH;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        gbc.gridy = 2;
        getContentPane().add(optionsPanel, gbc);

        buttonPanel.setLayout(new GridLayout(1, 2, 6, 0));

        startButton.addActionListener(this);
        buttonPanel.add(startButton);

        closeButton.addActionListener(this);
        buttonPanel.add(closeButton);

        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.insets = insets;
        getContentPane().add(buttonPanel, gbc);

        updateLocale();
    }

    private void updateLocale() {
        setTitle(ElanLocale.getString("FillGapsDialog.Title"));
        titleLabel.setText(ElanLocale.getString("FillGapsDialog.Title"));
        tierPanel.setBorder(new TitledBorder(ElanLocale.getString(
                    "EditTierDialog.Label.ChangeTier")));
        optionsPanel.setBorder(new TitledBorder(ElanLocale.getString(
                    "LabelAndNumberDialog.Label.Options")));
        onSameTierRB.setText(ElanLocale.getString(
                "FillGapsDialog.Label.SameTier"));
        onNewTierRB.setText(ElanLocale.getString("FillGapsDialog.Label.NewTier"));
        tierNameLabel.setText(ElanLocale.getString(
                "FillGapsDialog.Label.NewTierName"));
        emptyRB.setText(ElanLocale.getString("FillGapsDialog.Label.Empty"));
        durationRB.setText(ElanLocale.getString("FillGapsDialog.Label.Duration"));
        valueRB.setText(ElanLocale.getString(
                "FillGapsDialog.Label.SpecifyValue"));
        startButton.setText(ElanLocale.getString("Button.OK"));
        closeButton.setText(ElanLocale.getString("Button.Close"));
    }

    /**
     * Extract all top-level tiers and fill the list.
     */
    private void extractTiers() {
        if (transcription != null) {
            List v = transcription.getTiers();
            TierImpl t;

            for (int i = 0; i < v.size(); i++) {
                t = (TierImpl) v.get(i);

                if ((t.getParentTier() == null) && t.isTimeAlignable()) { //currently superfluous check
                    model.addElement(t.getName());
                }
            }

            if (model.getSize() > 0) {
                tierList.setSelectedIndex(0);
            } else {
                startButton.setEnabled(false);
            }
        }
    }

    /**
     * The action performed handling.
     *
     * @param e the event
     */
    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == closeButton) {
            setVisible(false);
            dispose();
        } else if (e.getSource() == startButton) {
            String tierName = (String) tierList.getSelectedValue();

            if (tierName == null) {
                JOptionPane.showMessageDialog(this,
                    ElanLocale.getString("LabelAndNumberDialog.Warning.NoTier"),
                    ElanLocale.getString("Message.Error"),
                    JOptionPane.WARNING_MESSAGE);

                return;
            }

            String nTierName = tierNameTF.getText();

            if (onNewTierRB.isSelected()) {
                if ((nTierName == null) || (nTierName.length() == 0)) {
                    JOptionPane.showMessageDialog(this,
                        ElanLocale.getString("EditTierDialog.Message.TierName"),
                        ElanLocale.getString("Message.Error"),
                        JOptionPane.WARNING_MESSAGE);

                    return;
                } else {
                    if (transcription.getTierWithId(nTierName) != null) {
                        JOptionPane.showMessageDialog(this,
                            ElanLocale.getString(
                                "EditTierDialog.Message.Exists"),
                            ElanLocale.getString("Message.Error"),
                            JOptionPane.WARNING_MESSAGE);

                        return;
                    }
                }
            } else {
                nTierName = null;
            }

            String annValue = null;
            String format = null;

            if (valueRB.isSelected()) {
                annValue = annValueTF.getText();
            } else if (durationRB.isSelected()) {
                if (msRB.isSelected()) {
                    format = Constants.MS_STRING;
                } else if (secRB.isSelected()) {
                    format = Constants.SSMS_STRING;
                } else if (hourRB.isSelected()) {
                    format = Constants.HHMMSSMS_STRING;
                }
            }
             // else value is empty
 // create command

            Object[] arg = new Object[] {
                    tierName, nTierName, annValue, format,
                    new Long(mediaDuration)
                }; // if nTierName == null, the annotations are inserted on the source tier
            Command com = ELANCommandFactory.createCommand(transcription,
                    ELANCommandFactory.ANN_FROM_GAPS);
            com.execute(transcription, arg);
        }
    }

    /**
     * The handling of changes in radio button selections
     *
     * @param e the event
     */
    public void stateChanged(ChangeEvent e) {
        if ((e.getSource() == onSameTierRB) || (e.getSource() == onNewTierRB)) {
            tierNameTF.setEnabled(onNewTierRB.isSelected());
        } else if (e.getSource() == valueRB) {
            boolean b = valueRB.isSelected();
            annValueTF.setEnabled(b);
            msRB.setEnabled(!b);
            secRB.setEnabled(!b);
            hourRB.setEnabled(!b);
        } else if (e.getSource() == durationRB) {
            boolean b = durationRB.isSelected();
            annValueTF.setEnabled(!b);
            msRB.setEnabled(b);
            secRB.setEnabled(b);
            hourRB.setEnabled(b);
        } else if (e.getSource() == emptyRB) {
            boolean b = emptyRB.isSelected();
            annValueTF.setEnabled(!b);
            msRB.setEnabled(!b);
            secRB.setEnabled(!b);
            hourRB.setEnabled(!b);
        }
    }
}
