/*
 * File:     TransferableAnnotation.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.util;

import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;

import java.io.IOException;


/**
 * A transerable annotation, using an AnnotationdataRecord as the transferable object.
 *
 * Note June 2006 transferal of application/x-java-serialized-object objects between jvm's
 * doesn't seem to work on Mac OS X. The Transferable sun.awt.datatransfer.ClipboardTransferable@8b4cb8
 * doesn't support any flavor with human presentable name "application/x-java-serialized-object"
 */
public class TransferableAnnotation implements Transferable, ClipboardOwner {
    /** Holds value of property DOCUMENT ME! */
    private final static DataFlavor[] flavors;

    /** Holds value of property DOCUMENT ME! */
    private static final int STRING = 0;

    /** Holds value of property DOCUMENT ME! */
    private static final int ANNOTATION = 1;

    static {
        DataFlavor flav = AnnotationDataFlavor.getInstance();

        if (flav == null) {
            flavors = new DataFlavor[] { DataFlavor.stringFlavor };
        } else {
            flavors = new DataFlavor[] { DataFlavor.stringFlavor, flav };
        }
    }

    private AnnotationDataRecord record;

    /**
     * Creates a new TransferableAnnotation
     * @param record the transferable object
     */
    public TransferableAnnotation(AnnotationDataRecord record) {
        if (record == null) {
            throw new NullPointerException("AnnotationDataRecord is null.");
        }

        this.record = record;
    }

    /**
     * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
     */
    public DataFlavor[] getTransferDataFlavors() {
        return flavors;
    }

    /**
     * @see java.awt.datatransfer.Transferable#isDataFlavorSupported(java.awt.datatransfer.DataFlavor)
     */
    public boolean isDataFlavorSupported(DataFlavor flavor) {
        if (flavor == null) {
            return false; // could throw NullPointer Exc.
        }

        for (int i = 0; i < flavors.length; i++) {
            if (flavor.equals(flavors[i])) {
                return true;
            }
        }

        return false;
    }

    /**
     * @see java.awt.datatransfer.Transferable#getTransferData(java.awt.datatransfer.DataFlavor)
     */
    public Object getTransferData(DataFlavor flavor)
        throws UnsupportedFlavorException, IOException {
        if (!isDataFlavorSupported(flavor)) {
            throw new UnsupportedFlavorException(flavor);
        }

        if ((flavors.length > 1) && flavor.equals(flavors[ANNOTATION])) {
            return record; // clone?? once on the clipboard the record don't seem to change anymore
        } else if (flavor.equals(flavors[STRING])) {
            return recordParamString();
        }

        return null;
    }

    /**
     * Returns a string representation of the annotation data record.
     *
     * @return a string representation of the annotation data record
     */
    private String recordParamString() {
        if (record != null) {
            return record.getValue() + ",T=" + record.getTierName() + ",B=" +
            record.getBeginTime() + ",E=" + record.getEndTime();
        } else {
            return "null";
        }
    }

    /**
     * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard, java.awt.datatransfer.Transferable)
     */
    public void lostOwnership(Clipboard clipboard, Transferable contents) {
        record = null;
    }
}
