/*
 * File:     TransferableAnnotationTree.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.util;

import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;

import java.io.IOException;

import javax.swing.tree.DefaultMutableTreeNode;


/**
 * A transerable annotation tree, using a DefaultMutableTreeNode as the transferable object.
 * An annotation tree is an annotation with its depending or descendant annotations.
 *
 * Note June 2006 transferal of application/x-java-serialized-object objects between jvm's
 * doesn't seem to work on Mac OS X. The Transferable sun.awt.datatransfer.ClipboardTransferable
 * doesn't support any flavor with human presentable name "application/x-java-serialized-object"
 */
public class TransferableAnnotationTree implements Transferable, ClipboardOwner {
    /** Holds value of property DOCUMENT ME! */
    private final static DataFlavor[] flavors = new DataFlavor[] {
            AnnotationTreeDataFlavor.getInstance()
        };
    private DefaultMutableTreeNode node;

    /**
     * Creates a new TransferableAnnotationTree
     *
     * @param node the object for transfer
     */
    public TransferableAnnotationTree(DefaultMutableTreeNode node) {
        if (node == null) {
            throw new NullPointerException("Annotation TreeNode is null.");
        }

        this.node = node;
    }

    /**
     * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
     */
    public DataFlavor[] getTransferDataFlavors() {
        return flavors;
    }

    /**
     * @see java.awt.datatransfer.Transferable#isDataFlavorSupported(java.awt.datatransfer.DataFlavor)
     */
    public boolean isDataFlavorSupported(DataFlavor flavor) {
        if (flavor == null) {
            return false; // could throw NullPointer Exc.
        }

        return flavor.equals(flavors[0]);
    }

    /**
     * @see java.awt.datatransfer.Transferable#getTransferData(java.awt.datatransfer.DataFlavor)
     */
    public Object getTransferData(DataFlavor flavor)
        throws UnsupportedFlavorException, IOException {
        if (!isDataFlavorSupported(flavor)) {
            throw new UnsupportedFlavorException(flavor);
        }

        /* as long as there is one supported flavor the test below is always true if no exception
         has been thrown above
        if (flavor.equals(flavors[0])) {
            return node;
        }
        */
        return node;
    }

    /**
     * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard, java.awt.datatransfer.Transferable)
     */
    public void lostOwnership(Clipboard clipboard, Transferable contents) {
        node = null;
    }
}
