/*
 * File:     EventPostingBase.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.mediacontrol;

import java.util.Enumeration;
import java.util.Vector;


/**
 * Takes care of administrating ControllerListeners and sending events to them.
 * Inspired by EventPostingBase as described in Appendix C of the JMF
 * documentation.
 *
 * @author Hennie Brugman
 * @author Albert Russel
 * @version 09-June-1998
 */
public class EventPostingBase {
    private Vector listeners;

    /**
     * Creates a new posting base for Controller events.
     */
    public EventPostingBase() {
        listeners = new Vector();
    }

    /**
     * Returns the number of connected listeners
     *
     * @return DOCUMENT ME!
     */
    public int getNrOfConnectedListeners() {
        return listeners.size();
    }

    /**
     * Adds a ControllerListener only if it was not added before.
     *
     * @param listener the ControllerListener that wants to receive Controller
     *        events.
     */
    public synchronized void addControllerListener(ControllerListener listener) {
        // Check if the newListener is realy new, if so add it to the listeners.
        if (!listeners.contains(listener)) {
            listeners.addElement(listener);
        }
    }

    /**
     * Removes a ControllerListener.
     *
     * @param listener the ControllerListener that no longer wants to receive
     *        Controller events.
     */
    public synchronized void removeControllerListener(
        ControllerListener listener) {
        listeners.removeElement(listener);
    }

    /**
     * Posts a ControllerEvent to all registered listeners.
     *
     * @param event the Controller event that has to be posted to all
     *        registered listeners.
     */
    public synchronized void postEvent(ControllerEvent event) {
        Enumeration en = listeners.elements();

        while (en.hasMoreElements()) {
            new EventPoster((ControllerListener) en.nextElement(), event);
        }
    }
}
