/*
 * File:     SubtitleSequencer.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.util;

import mpi.eudico.client.annotator.util.ClientLogger;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.AbstractAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;

import mpi.eudico.util.TimeRelation;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;


/**
 * Class that creates a sequence of data units for subtitles. The units consist
 * of a text value and a begin and end time, where the  end time can be either
 * the real end time or a calculated end time based on a minimum duration.<br>
 * Overlaps are handled by splitting into multiple units.
 *
 * @author Han Sloetjes
 */
public class SubtitleSequencer implements ClientLogger {
    /**
     * Creates a new SubtitleSequencer instance
     */
    public SubtitleSequencer() {
        super();
    }

    /**
     * Creates a list of subtitle objects, including all annotations of the
     * specified  tiers and applying a minimal duration.
     *
     * @param transcription the transcription document
     * @param tierNames the tiers to include
     * @param intervalBegin the selection begintime
     * @param intervalEnd the selection end time
     * @param minimalDuration the minimal duration per subtitle
     * @param offset the number of ms. to add to all time values
     * @param resolveOverlaps detects overlapping units and creates new,
     *        merging units for the overlaps
     *
     * @return a list of subtitle objects
     *
     * @throws NullPointerException if the {@link Transcription} or the list of
     *         tiernames is null
     * @throws IllegalArgumentException if the size of the list of tier names
     *         is 0
     */
    public List createSequence(Transcription transcription, List tierNames,
        long intervalBegin, long intervalEnd, int minimalDuration, long offset,
        boolean resolveOverlaps) {
        if (transcription == null) {
            throw new NullPointerException("The transcription is null");
        }

        if (tierNames == null) {
            throw new NullPointerException("The list of tier names is null");
        }

        if (tierNames.size() == 0) {
            throw new IllegalArgumentException("No tiers have been specified");
        }

        ArrayList units = new ArrayList();
        String name;
        TierImpl tier;
        AbstractAnnotation ann;
        List annotations;

        for (int i = 0; i < tierNames.size(); i++) {
            name = (String) tierNames.get(i);
            tier = (TierImpl) transcription.getTierWithId(name);

            if (tier == null) {
                LOG.warning("The tier does not exist: " + name);

                continue;
            }

            annotations = tier.getAnnotations();

            for (int j = 0; j < annotations.size(); j++) {
                ann = (AbstractAnnotation) annotations.get(j);

                if ((ann != null) &&
                        TimeRelation.overlaps(ann, intervalBegin, intervalEnd)) {
                    // correct annotation times to fit in the interval??

                    /*
                       units.add(new SubtitleUnit(Math.max(intervalBegin,
                                   ann.getBeginTimeBoundary()),
                               Math.min(intervalEnd, ann.getEndTimeBoundary()),
                               ann.getValue()));
                     */
                    units.add(new SubtitleUnit(ann.getBeginTimeBoundary() +
                            offset, ann.getEndTimeBoundary() + offset,
                            ann.getValue()));
                }
            }
        }

        // all units have been added, sort first
        Collections.sort(units);

        // apply minimal duration if possible
        if (minimalDuration > 0) {
            SubtitleUnit unit1 = null;
            SubtitleUnit unit2 = null;
            long calcEnd;

            for (int i = 0; i < (units.size() - 1); i++) {
                unit2 = (SubtitleUnit) units.get(i);

                if (i == 0) {
                    unit1 = unit2;

                    continue;
                } else {
                    // unit1 is the previous unit
                    if ((unit1.getRealEnd() - unit1.getBegin()) < minimalDuration) {
                        calcEnd = unit1.getBegin() + minimalDuration;

                        if ((unit2.getBegin() < calcEnd) &&
                                (unit2.getBegin() > unit1.getRealEnd())) {
                            calcEnd = unit2.getBegin();
                        }

                        unit1.setCalcEnd(calcEnd);
                    }

                    // swap
                    unit1 = unit2;
                }
            }
        }

        if (!resolveOverlaps) {
            return units;
        }

        // create new units for overlaps
        SubtitleUnit unit1 = null;
        SubtitleUnit unit2 = null;
        long overlap;
        int overRatio;
        long unitDur;

        // some arbitrary minimal values
        // the minimal overlap before considering to insert a unit for the overlap
        long minOverDur = 100;

        // the minimal remainder of a unit if a bit of the end is to be cut of
        long minRemainder = 250;

        // the minimal percentage of overlap before considering to split a unit
        int minOverRatio = 25;

        if (minimalDuration > 0) {
            minRemainder = minimalDuration;
        }

        for (int i = (units.size() - 1); i >= 0; i--) {
            unit1 = (SubtitleUnit) units.get(i);

            if (i == (units.size() - 1)) {
                unit2 = unit1;

                continue;
            }

            overlap = unit1.getCalcEnd() - unit2.getBegin();
            unitDur = unit1.getCalcEnd() - unit1.getBegin();
            overRatio = (int) (100 * ((float) overlap / unitDur));

            // algorithm: 
            // - if overlap > minOverDur and 
            // (overlap > 25% (??) of the duration of unit 1 or 
            // duration of unit 1 - overlap < minRemainder
            // create and insert a new unit
            if ((overlap > minOverDur) &&
                    ((overRatio >= minOverRatio) ||
                    ((unitDur - overlap) < minRemainder))) {
                // insert new unit
                int numLines = 1;

                if (unit1.getValues() != null) {
                    numLines = unit1.getValues().length;
                }

                if (unit2.getValues() != null) {
                    numLines += unit2.getValues().length;
                } else {
                    numLines++;
                }

                String[] nextVal = new String[numLines];
                int k = 0;

                if (unit1.getValues() != null) {
                    for (int j = 0;
                            (j < unit1.getValues().length) &&
                            (k < nextVal.length); j++, k++) {
                        nextVal[k] = unit1.getValues()[j];
                    }
                } else {
                    nextVal[k] = unit1.getValue();
                    k++;
                }

                if (unit2.getValues() != null) {
                    for (int j = 0;
                            (j < unit2.getValues().length) &&
                            (k < nextVal.length); j++, k++) {
                        nextVal[k] = unit2.getValues()[j];
                    }
                } else {
                    nextVal[k] = unit2.getValue();
                }

                SubtitleUnit nextUnit = new SubtitleUnit(unit2.getBegin(),
                        unit1.getCalcEnd(), null);
                nextUnit.setValues(nextVal);
                unit1.setCalcEnd(nextUnit.getBegin());
                unit2.setBegin(nextUnit.getCalcEnd());

                // Oct 2008; check the remaining duration of unit 1 and 2
                // < minOverDuration is questionable, the goal is to prevent flickering
                if ((unit1.getCalcEnd() - unit1.getBegin()) < minOverDur) {
                    nextUnit.setBegin(unit1.getBegin());

                    // replace
                    units.set(i, nextUnit);
                } else {
                    //insert after unit 1
                    units.add(i + 1, nextUnit);
                }

                if ((unit2.getCalcEnd() - unit2.getBegin()) < minOverDur) {
                    nextUnit.setCalcEnd(unit2.getCalcEnd());
                    units.remove(unit2);
                }
            } else if (overlap > 0) {
                // adjust the endtime of unit 1
                unit1.setCalcEnd(unit2.getBegin());
            }

            unit2 = unit1;
        }

        return units;
    }

    /**
     * Creates a list of subtitle objects, including all annotations of the
     * specified  tiers and applying a minimal duration.
     *
     * @param transcription the transcription document
     * @param tierNames the tiers to include
     * @param intervalBegin the selection begintime
     * @param intervalEnd the selection end time
     * @param minimalDuration the minimal duration per subtitle
     *
     * @return a list of subtitle objects
     *
     * @see #createSequence(Transcription, List, long, long, int, boolean)
     */
    public List createSequence(Transcription transcription, List tierNames,
        long intervalBegin, long intervalEnd, int minimalDuration) {
        return createSequence(transcription, tierNames, intervalBegin,
            intervalEnd, minimalDuration, 0L, false);
    }

    /**
     * Creates a list of subtitle objects, including all annotations of the
     * specified  tiers and applying a minimal duration.
     *
     * @param transcription the transcription document
     * @param tierNames the tiers to include
     * @param intervalBegin the selection begintime
     * @param intervalEnd the selection end time
     * @param minimalDuration the minimal duration per subtitle
     * @param offset the number of ms. to add to all time values
     * @param resolveOverlaps detects overlapping units and creates new,
     *        merging units for the overlaps
     *
     * @return a list of subtitle objects
     *
     * @see #createSequence(Transcription, List, long, long, int, boolean)
     */
    public List createSequence(Transcription transcription, String[] tierNames,
        long intervalBegin, long intervalEnd, int minimalDuration, long offset,
        boolean resolveOverlaps) {
        ArrayList names = null;

        if (tierNames != null) {
            names = new ArrayList(tierNames.length);

            for (int i = 0; i < tierNames.length; i++) {
                names.add(tierNames[i]);
            }
        }

        return createSequence(transcription, names, intervalBegin, intervalEnd,
            minimalDuration, offset, resolveOverlaps);
    }

    /**
     * Creates a list of subtitle objects, including all annotations of the
     * specified  tiers and applying a minimal duration.
     *
     * @param transcription the transcription document
     * @param tierNames the tiers to include
     * @param intervalBegin the selection begintime
     * @param intervalEnd the selection end time
     * @param minimalDuration the minimal duration per subtitle
     *
     * @return a list of subtitle objects
     *
     * @see #createSequence(Transcription, String[], long, long, int, boolean)
     */
    public List createSequence(Transcription transcription, String[] tierNames,
        long intervalBegin, long intervalEnd, int minimalDuration) {
        return createSequence(transcription, tierNames, intervalBegin,
            intervalEnd, minimalDuration, 0L, false);
    }
}
