/*
 * File:     Transcription2SubtitleText.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.util;

import mpi.eudico.client.annotator.util.ClientLogger;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.util.TimeFormatter;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;

import java.util.List;


/**
 * Exports annotations of a selection of tiers to a DVD subtitle textformat.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class Transcription2SubtitleText implements ClientLogger {
    /** the Windows CR+LF line break for SRT */
    final private String WIN_NEWLINE = "\r\n";

    /**
     * Constructor
     */
    public Transcription2SubtitleText() {
        super();
    }

    /**
     * Exports to .srt format file.  Provides support for a minimum duration
     * tries to solve overlaps.
     *
     * @param transcription the transcription
     * @param tierNames the tiers to include
     * @param exportFile the file to export to
     * @param encoding the file encoding, defaults to UTF-8
     * @param beginTime the (selection) begintime
     * @param endTime the (selection) end time
     * @param minimalDuration the minimal duration per subtitle
     *
     * @throws IOException any io exception
     */
    public void exportTiersSRT(Transcription transcription, String[] tierNames,
        File exportFile, String encoding, long beginTime, long endTime,
        int minimalDuration, long offset) throws IOException {
        if (exportFile == null) {
            LOG.severe("No export file specified");

            return;
        }

        SubtitleSequencer sequencer = new SubtitleSequencer();

        List allUnits = sequencer.createSequence(transcription, tierNames,
                beginTime, endTime, minimalDuration, offset, true);

        SubtitleUnit unit;
        FileOutputStream out = null;
        BufferedWriter writer = null;

        out = new FileOutputStream(exportFile);

        if (encoding != null) {
            try {
                writer = new BufferedWriter(new OutputStreamWriter(out, encoding));
            } catch (UnsupportedEncodingException uee) {
                LOG.warning("Encoding not supported: " + encoding);
                writer = new BufferedWriter(new OutputStreamWriter(out, "UTF-8"));
            }
        } else {
            writer = new BufferedWriter(new OutputStreamWriter(out, "UTF-8"));
        }

        for (int i = 0; i < allUnits.size(); i++) {
            unit = (SubtitleUnit) allUnits.get(i);
            writer.write(String.valueOf(i + 1)); // some apps don't accept index 0
            writer.write(WIN_NEWLINE);
            writer.write(TimeFormatter.toString(unit.getBegin()).replace('.',
                    ','));
            writer.write(" --> ");
            writer.write(TimeFormatter.toString(unit.getCalcEnd()).replace('.',
                    ','));
            writer.write(WIN_NEWLINE);

            if (unit.getValue() != null) {
                writer.write(unit.getValue().replace('\n', ' '));
            } else {
                for (int j = 0; j < unit.getValues().length; j++) {
                    writer.write(unit.getValues()[j].replace('\n', ' '));

                    if (j != (unit.getValues().length - 1)) {
                        writer.write(WIN_NEWLINE);
                    }
                }
            }

            writer.write(WIN_NEWLINE);
            writer.write(WIN_NEWLINE);
        }

        writer.close();
    }

    /**
     * Exports to .srt format file using the default file encoding, UTF-8.
     *
     * @param transcription the transcription
     * @param tierNames the tiers to include
     * @param exportFile the file to export to
     * @param beginTime the (selection) begintime
     * @param endTime the (selection) end time
     * @param minimalDuration the minimal duration per subtitle
     *
     * @throws IOException any io exception
     */
    public void exportTiersSRT(Transcription transcription, String[] tierNames,
        File exportFile, long beginTime, long endTime, int minimalDuration)
        throws IOException {
        exportTiersSRT(transcription, tierNames, exportFile, null, beginTime,
            endTime, minimalDuration, 0L);
    }

    /**
     * Exports to .srt format file, using default file encoding (UTF-8),
     * interval begin (0), interval end (Long.MAX_VALUE) and minimal duration
     * (0).
     *
     * @param transcription the transcription
     * @param tierNames the tiers to include
     * @param exportFile the file to export to
     *
     * @throws IOException any io exception
     */
    public void exportTiersSRT(Transcription transcription, String[] tierNames,
        File exportFile) throws IOException {
        exportTiersSRT(transcription, tierNames, exportFile, 0, Long.MAX_VALUE,
            0);
    }
}
