/*
 * File:     TranscriptionScrubber.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.util;

import mpi.eudico.client.annotator.util.ClientLogger;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.TranscriptionStore;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;
import mpi.eudico.server.corpora.clomimpl.dobes.ACMTranscriptionStore;

import java.io.File;
import java.io.IOException;

import java.util.List;

import javax.swing.ProgressMonitor;


/**
 * A class that removes leading and/or trailing space and ASCII control
 * characters from all annotations in one or more transcriptions.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class TranscriptionScrubber implements ClientLogger {
    /** the space character! */
    public final char SP = '\u0020';

    /**
     * Creates a new TranscriptionScrubber instance
     */
    public TranscriptionScrubber() {
        super();
    }

    /**
     * Iterates over the list of files scrubs the transcription and (if
     * anything changed)  saves the file. Overwrites.
     *
     * @param files the eaf files
     */
    public void scrubAndSave(List files) {
        if ((files == null) || (files.size() == 0)) {
            LOG.info("No (valid) files supplied.");

            return;
        }

        new MultiFileScrubber(files).start();
    }

    //###############################################

    /**
     * The actual scrubbing in a seperate thread. Customization options could
     * be added, like save to a new file instead of overwriting, only scrub at
     * the end, also remove newline characters within annotations etc.
     *
     * @author Han Sloetjes
     * @version 1.0
     */
    class MultiFileScrubber extends Thread {
        private List files;

        /**
         * Creates a new MultifileScrubber instance
         *
         * @param files the eaf files
         */
        public MultiFileScrubber(List files) {
            super();
            this.files = files;
        }

        /**
         * Creates a new MultifileScrubber instance
         *
         * @param name the thread name
         * @param files the eaf files
         */
        public MultiFileScrubber(String name, List files) {
            super(name);
            this.files = files;
        }

        /**
         * The actual work
         */
        public void run() {
            if ((files == null) || (files.size() == 0)) {
                LOG.info("No files supplied to the thread.");

                return;
            }

            int numFiles = files.size();

            final ProgressMonitor monitor = new ProgressMonitor(null,
                    "Scrubbing annotations...", "", 0, numFiles);
            monitor.setMillisToDecideToPopup(100);
            monitor.setMillisToPopup(100);

            TranscriptionStore eafTranscriptionStore = ACMTranscriptionStore.getCurrentTranscriptionStore();
            File file;
            TranscriptionImpl trans;
            TierImpl tier;
            List annotations;
            Annotation ann;
            String val;
            char b;
            char e;

            for (int i = 0; i < numFiles; i++) {
                file = (File) files.get(i);

                if (file == null) {
                    LOG.warning("File is null (index = " + i + ")");

                    continue;
                }

                monitor.setNote(file.getName());

                try {
                    trans = new TranscriptionImpl(file.getAbsolutePath());
                    trans.setNotifying(false);

                    boolean changed = false;

                    List tiers = trans.getTiers();

                    for (int j = 0; j < tiers.size(); j++) {
                        tier = (TierImpl) tiers.get(j);

                        if (tier == null) {
                            LOG.warning("Tier is null (index = " + j + ")");

                            continue;
                        }

                        annotations = tier.getAnnotations();

                        int numAnnos = tier.getNumberOfAnnotations();

                        for (int k = 0; k < numAnnos; k++) {
                            ann = (Annotation) annotations.get(k);
                            val = ann.getValue();

                            if ((val != null) && (val.length() > 0)) {
                                b = val.charAt(0);
                                e = val.charAt(val.length() - 1);

                                if ((b <= SP) || (e <= SP)) {
                                    val = val.trim();
                                    ann.setValue(val);
                                    changed = true;
                                }
                            }
                        }
                    }

                    if (changed) {
                        // store
                        try {
                            eafTranscriptionStore.storeTranscription(trans,
                                null, trans.getTiers(), TranscriptionStore.EAF);
                            LOG.info("Scrubbed and saved file: " +
                                file.getName());
                        } catch (IOException ioe) {
                            LOG.severe(
                                "Error while saving eaf file! The file could have been damaged!");
                        }
                    }
                } catch (Exception ex) {
                    // catch any exception that could occur and continue
                    LOG.warning("Could not handle file: " +
                        file.getAbsolutePath());
                }

                monitor.setProgress(i + 1);

                if (monitor.isCanceled()) {
                    LOG.warning("Scrubbing canceled after " + i + " files.");

                    break;
                }
            }
        }
    }
}
