/*
 * File:     SearchLocale.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.search;

import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;


/**
 * Created on Jun 10, 2004
 *
 * @author Alexander Klassmann
 * @version July 2004
 */
public class SearchLocale {
    private static Locale locale;
    private static ResourceBundle resourcebundle;

    /**
     * Constructor
     */
    static {
        setLocale(Locale.getDefault());
    }

    /**
     * Gets the current locale
     *
     * @return The current locale
     */
    final public static Locale getLocale() {
        return locale;
    }

    /**
     * Sets the current locale
     *
     * @param locale_in
     *            The new locale
     */
    final public static void setLocale(Locale locale_in) {
        if ((locale != null) && locale.equals(locale_in)) {
            return;
        }

        //if (locale != null && locale.getCountry().equals(locale_in.getCountry())) {
        //    return;
        //}
        locale = locale_in;

        try {
            resourcebundle = ResourceBundle.getBundle("mpi.search.resources.SearchLanguage",
                    locale);
        } catch (MissingResourceException e) {
        }

        if (resourcebundle == null) {
            System.out.println("WARNING: no language resources for " +
                locale.getDisplayLanguage());
        }
    }

    /**
     * Alternative for {@link #setLocale(Locale)}. In case a bundle has to be loaded from a different
     * location than the standard location.
     *
     * @param resBundle the resource bundle, loaded from an alternative location
     */
    public static final void setResourceBundle(ResourceBundle resBundle) {
        if (resBundle != null) {
            resourcebundle = resBundle;
        }
    }

    /**
     * Gets the string in the right language from the right resource file
     *
     * @param str
     *            The string which has to be mapped to the right language
     *
     * @return The string in the right language
     */
    final public static String getString(String str) {
        if (locale == null) {
            setLocale(Locale.getDefault());
        }

        if (resourcebundle != null) {
            try {
                return resourcebundle.getString(str);
            } catch (MissingResourceException ex) {
                System.out.println("Warning: no localization for " + str +
                    " found in language " + locale.getDisplayCountry());
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }

        return "";
    }
}
