/*
 * File:     PagingListModel.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.util;

import java.util.List;

import javax.swing.AbstractListModel;


/**
 * model that allows paging through list data
 * corresponds to the Paging(Table)Model of O'Reilly book JavaSwing
 * $Id: PagingListModel.java 7998 2007-01-30 15:04:34Z klasal $
 *
 * @author $author$
 * @version $Revision$
 */
public class PagingListModel extends AbstractListModel {
    /** Holds value of property DOCUMENT ME! */
    public static final int DEFAULT_PAGE_SIZE = 50;

    /** Holds value of property DOCUMENT ME! */
    protected List data;

    /** Holds value of property DOCUMENT ME! */
    protected int pageOffset = 0;

    /** Holds value of property DOCUMENT ME! */
    protected int pageSize = DEFAULT_PAGE_SIZE;

    // Work only on the visible part of the table.
    public Object getElementAt(int index) {
        //dummy exception if attempt to get data beyond viewable part
        if (index >= pageSize) {
            throw new IndexOutOfBoundsException("Index " + index +
                ", viewable Size " + pageSize);
        }

        int realIndex = index + (pageOffset * pageSize);

        return data.get(realIndex);
    }

    /**
     *
     * @return real index of first entry in viewable page
     */
    public int getFirstShownRealIndex() {
        return (data.size() == 0) ? (-1) : (pageOffset * pageSize);
    }

    /**
     *
     * @return real index of last entry in viewable page
     */
    public int getLastShownRealIndex() {
        return ((pageOffset * pageSize) + getSize()) - 1;
    }

    /**
     *
     * @return number of pages needed to cover the data
     */
    public int getPageCount() {
        return (int) Math.ceil((double) getRealSize() / pageSize);
    }

    /**
     * @return actual page number that is accesible
     */
    public int getPageOffset() {
        return pageOffset;
    }

    /**
     * sets the size of the viewable part
     *
     * @param s page size
     */
    public void setPageSize(int s) {
        if (s == pageSize) {
            return;
        }

        int oldPageSize = pageSize;
        pageSize = s;
        pageOffset = (oldPageSize * pageOffset) / pageSize;

        fireContentsChanged(this, 0, getSize());
    }

    /**
     *
     *
     * @return page size (maximum size of viewable data)
     */
    public int getPageSize() {
        return pageSize;
    }

    /**
     *
     * @return size of total data
     */

    // Use this method if you want to know how big the real list is . . . we
    // could also write "getRealValueAt()" if needed.
    public int getRealSize() {
        return data.size();
    }

    /**
     * @return size of viewable part
     */
    public int getSize() {
        return Math.min(pageSize, data.size() - (pageOffset * pageSize));
    }

    /**
     * Update the page offset and fire a data changed.
     */
    public void pageDown() {
        if (pageOffset > 0) {
            pageOffset--;
            fireContentsChanged(this, 0, getSize());
        }
    }

    /**
     * Update the page offset and fire a data changed.
     */
    public void pageUp() {
        if (pageOffset < (getPageCount() - 1)) {
            pageOffset++;
            fireContentsChanged(this, 0, getSize());
        }
    }
}
