/*
 * File:     TreeFrame.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.util.gui;

import mpi.util.Xml2Tree;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;

import java.net.URL;

import javax.swing.Icon;
import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.WindowConstants;
import javax.swing.plaf.basic.BasicTreeUI;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreeNode;


/**
 * presents a tree in a JFrame
 *
 * $Id: TreeFrame.java 8315 2007-03-06 12:53:22Z klasal $
 *
 * @author $Author$
 * @version $Revision$
 */
public class TreeFrame extends JFrame {
    /** Holds representation of tree */
    private final JTree tree;

    /** Holds value of property DOCUMENT ME! */
    private final DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();

    /**
     * Creates a new TreeFrame instance
     *
     * @param treeNode root node
     *
     */
    public TreeFrame(TreeNode treeNode) {
        this(treeNode, false);
    }

    /**
     * @param schemaFileName file name of imdi-schema
     */
    public TreeFrame(String schemaFileName) throws Exception {
        this(new URL("file://" + schemaFileName));
    }

    /**
     *
     * @param schemaURL
     * @throws Exception
     */
    public TreeFrame(URL schemaURL) throws Exception {
        if (schemaURL != null) {
            DefaultMutableTreeNode root = (DefaultMutableTreeNode) Xml2Tree.getTree(schemaURL);
            TreeNode sessionNode = root.getChildAt(0).getChildAt(0);
            tree = new JTree(sessionNode);
            makeTreeLayout();
            setTitle(schemaURL.getFile());
            pack();
            setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        } else {
            throw new Exception("URL is null.");
        }
    }

    /**
     * Creates a new TreeFrame instance
     *
     * @param treeNode root node
     * @param bDefaultCloseOperation window default close operation (WindowConstants)
     *
     */
    public TreeFrame(TreeNode treeNode, boolean bDefaultCloseOperation) {
        tree = new JTree(treeNode);
        makeTreeLayout();

        pack();

        setDefaultCloseOperation(bDefaultCloseOperation
            ? WindowConstants.DO_NOTHING_ON_CLOSE
            : WindowConstants.DISPOSE_ON_CLOSE);
    }

    /**
     * set background
     * @param color background color
     */
    public void setTreeBackground(Color color) {
        tree.setBackground(color);
        renderer.setBackgroundNonSelectionColor(color);
    }

    private void makeTreeLayout() {
        tree.putClientProperty("JTree.lineStyle", "Angled");
        renderer.setLeafIcon(null);
        renderer.setOpenIcon(null);
        renderer.setClosedIcon(null);
        tree.setCellRenderer(renderer);
        getContentPane().add(new JScrollPane(tree));

        for (int i = 0; i < tree.getRowCount(); i++)
            tree.expandRow(i);

        tree.setUI(new BasicTreeUI() {
                Icon emptyIcon = new Icon() {
                        public void paintIcon(Component c, Graphics g, int x,
                            int y) {
                        }

                        public int getIconWidth() {
                            return 16;
                        }

                        public int getIconHeight() {
                            return 16;
                        }
                    };

                /**
                 * DOCUMENT ME!
                 *
                 * @return DOCUMENT ME!
                 */
                public Icon getExpandedIcon() {
                    return emptyIcon;
                }

                /**
                 * DOCUMENT ME!
                 *
                 * @return DOCUMENT ME!
                 */
                public Icon getCollapsedIcon() {
                    return emptyIcon;
                }
            });
    }

    /**
     * test
     *
     * @param a no parameters used
     */
    public static void main(String[] a) {
        String fileName = "MPI/src/java/mpi/metadata/profiles/SESSION.Profile.xml";

        try {
            DefaultMutableTreeNode root = (DefaultMutableTreeNode) Xml2Tree.getTree(new URL(
                        "file://" + fileName));
            TreeNode sessionNode = root.getChildAt(0).getChildAt(0);
            JFrame frame = new TreeFrame(sessionNode);
            frame.setSize(400, 400);
            frame.pack();
            frame.setVisible(true);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns the tree.
     *
     * @return the tree
     */
    public JTree getTree() {
        return tree;
    }
}
