package mpi.dcr;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.ResourceBundle;

import javax.swing.DefaultListModel;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTree;
import javax.swing.ListSelectionModel;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreePath;

import mpi.dcr.isocat.Profile;


/**
 * A panel with 2 or 3 sub panels, one with a list of (pre) selected profiles,
 * one with the data categories of the selected profile and a description
 * panel for the selected data category.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public abstract class AbstractDCSelectPanel extends JPanel
    implements ActionListener, ItemListener, ListSelectionListener,
        TreeSelectionListener {
    /** the DCR connector */
    protected ILATDCRConnector connector = null;

    /** the profiles panel */
    protected JPanel profPanel;

    /** the list of profiles */
    protected JList profList;

    /** the profiles list model */
    protected DefaultListModel profModel;

    /** the data categories panel */
    protected JPanel catPanel;

    /** the sort mode label */
    protected JLabel sortModeLabel;

    /** the sort mode combo box */
    protected JComboBox sortModeCombo;

    /** the sort modes */
    protected String[] sortModes;

    /** the scroll pane for the data categories list or tree */
    protected JScrollPane catScroll;

    /** the data categories list */
    protected JList catList;

    /** the data categories list model */
    protected DefaultListModel catModel;

    /** the tree view of the data categories */
    protected JTree catTree;

    /** the root node of the tree */
    protected DefaultMutableTreeNode rootNode;

    //protected JButton refreshButton;
    /** data category description panel */
    protected JPanel descPanel;

    /** text area for (english) description of a data category */
    protected JTextArea descArea;

    /** the label for the key of the (camelcase) data category indentifier */
    protected JLabel identifierKeyLabel;

    /** the label for the value of the (camelcase) data category indentifier */
    protected JLabel identifierValLabel;

    /** the label for the key of the data category (numeric) id */
    protected JLabel idKeyLabel;

    /** the label for the value of the data category (numeric) id */
    protected JLabel idValLabel;

    /** the label for the key of the profiles the data category belongs to */
    protected JLabel profilesKeyLabel;

    /** the label for the value of the profiles the data category belongs to */
    protected JLabel profilesValLabel;

    /** alphabetical sort order */
    protected final int BY_ALPHA = 0;

    /** sorting by id */
    protected final int BY_ID = 1;

    /**
     * tree wise representation of the data categories, based on the broader
     * concept generic
     */
    protected final int BY_CONCEPT = 2;

    /** the current sort mode */
    protected int sortMode = BY_ALPHA;

    /** the current profile */
    protected Profile currentProfile = null;

    /** the empty string */
    protected final String EMPTY = "-";

    /** a resource bundle for localisation */
    protected ResourceBundle resBundle;

    /**
     * Creates a new AbstractDCSelectPanel instance
     *
     * @param connector the connector to connect to the dcr
     */
    public AbstractDCSelectPanel(ILATDCRConnector connector) {
        super();
        this.connector = connector;

        //initComponents();
    }

    /**
     * Creates a new AbstractDCSelectPanel instance
     *
     * @param connector the connector to connect to the dcr
     * @param resBundle the resource bundle
     */
    public AbstractDCSelectPanel(ILATDCRConnector connector,
        ResourceBundle resBundle) {
        super();
        this.connector = connector;
        this.resBundle = resBundle;

        //initComponents();
    }

    /**
     * Creates a new AbstractDCSelectPanel instance
     */
    public AbstractDCSelectPanel() {
        super();

        //initComponents();
    }

    /**
     * Initializes the ui components
     */
    protected void initComponents() {
        setLayout(new GridBagLayout());

        Insets insets = new Insets(2, 6, 2, 6);
        // profile panel
        profPanel = new JPanel(new GridBagLayout());
        profModel = new DefaultListModel();
        profList = new JList(profModel);

        profList.getSelectionModel()
                .setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        profList.addListSelectionListener(this);

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        profPanel.add(new JScrollPane(profList), gbc);

        // categories panel
        catPanel = new JPanel(new GridBagLayout());
        sortModeLabel = new JLabel();
        sortModeCombo = new JComboBox();
        sortModeCombo.addItemListener(this);
        catScroll = new JScrollPane();
        catModel = new DefaultListModel();
        catList = new JList(catModel);
        catList.addListSelectionListener(this);
        rootNode = new DefaultMutableTreeNode();
        catTree = new JTree(rootNode);
        catTree.setRootVisible(false);
        ((DefaultTreeCellRenderer) catTree.getCellRenderer()).setLeafIcon(null);
        ((DefaultTreeCellRenderer) catTree.getCellRenderer()).setOpenIcon(null);
        ((DefaultTreeCellRenderer) catTree.getCellRenderer()).setClosedIcon(null);
        catTree.setShowsRootHandles(true);
        catTree.addTreeSelectionListener(this);
        //refreshButton = new JButton();
        //refreshButton.setEnabled(false);
        gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        catPanel.add(sortModeLabel, gbc);

        gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        gbc.gridy = 1;
        catPanel.add(sortModeCombo, gbc);

        // default is list
        catScroll.setViewportView(catList);
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        gbc.gridy = 2;
        catPanel.add(catScroll, gbc);

        catScroll.setViewportView(catList);
        /*
           gbc.fill = GridBagConstraints.NONE;
           gbc.weightx = 0.0;
           gbc.weighty = 0.0;
           gbc.gridy = 3;
           catPanel.add(refreshButton, gbc);
         */

        // description panel
        identifierKeyLabel = new JLabel();
        identifierValLabel = new JLabel();
        idKeyLabel = new JLabel();
        idValLabel = new JLabel();
        profilesKeyLabel = new JLabel();
        profilesValLabel = new JLabel();

        Insets spacerInsets = new Insets(2, 6, 10, 6);

        descPanel = new JPanel(new GridBagLayout());

        gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.insets = insets;
        gbc.weightx = 1.0;
        descPanel.add(identifierKeyLabel, gbc);

        gbc.gridy = 1;
        gbc.insets = spacerInsets;
        descPanel.add(identifierValLabel, gbc);

        gbc.gridy = 2;
        gbc.insets = insets;
        descPanel.add(idKeyLabel, gbc);

        gbc.gridy = 3;
        gbc.insets = spacerInsets;
        descPanel.add(idValLabel, gbc);

        gbc.gridy = 4;
        gbc.insets = insets;
        descPanel.add(profilesKeyLabel, gbc);

        gbc.gridy = 5;
        gbc.insets = spacerInsets;
        descPanel.add(profilesValLabel, gbc);

        descArea = new JTextArea();
        descArea.setEditable(false);
        descArea.setLineWrap(true);
        descArea.setWrapStyleWord(true);

        Dimension dim = new Dimension(220, 300);
        gbc = new GridBagConstraints();
        gbc.gridy = 6;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        descPanel.add(new JScrollPane(descArea), gbc);

        profPanel.setMinimumSize(dim);
        profPanel.setPreferredSize(dim);
        gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.VERTICAL;
        gbc.weightx = 0.0;
        gbc.weighty = 1.0;
        gbc.gridheight = 1;
        add(profPanel, gbc);

        gbc.gridx = 1;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        gbc.gridheight = 1;
        add(catPanel, gbc);

        descPanel.setMinimumSize(dim);
        descPanel.setPreferredSize(dim);
        gbc.gridx = 2;
        gbc.fill = GridBagConstraints.VERTICAL;
        gbc.weightx = 0.0;
        gbc.weighty = 1.0;
        add(descPanel, gbc);

        Font valFont = identifierKeyLabel.getFont().deriveFont(Font.ITALIC, 10);
        identifierValLabel.setFont(valFont);
        idValLabel.setFont(valFont);
        profilesValLabel.setFont(valFont);

        updateLocale();
    }

    /**
     * Applies the textsof the ui elements.
     */
    protected void updateLocale() {
        String selProf = "Select Profile";
        String selCat = "Select Category";
        String sortCat = "Sort Categories:";
        String alpha = "Alphabetically";
        String byId = "By Id";
        String byBC = "By Broader Concept";
        String desc = "Category Description";
        String ident = "Identifier";
        String id = "Id";
        String profs = "Profiles";

        sortModes = new String[3];

        if (resBundle != null) {
            try {
                selProf = resBundle.getString("DCR.Label.SelectProfile");
            } catch (Exception ex) {
            }

            try {
                selCat = resBundle.getString("DCR.Label.SelectCategory");
            } catch (Exception ex) {
            }

            try {
                sortCat = resBundle.getString("DCR.Label.SortCategories");
            } catch (Exception ex) {
            }

            try {
                alpha = resBundle.getString("DCR.Sort.Alphabetical");
            } catch (Exception ex) {
            }

            try {
                byId = resBundle.getString("DCR.Sort.ById");
            } catch (Exception ex) {
            }

            try {
                byBC = resBundle.getString("DCR.Sort.ByBroaderConcept");
            } catch (Exception ex) {
            }

            try {
                desc = resBundle.getString("DCR.Label.CategoryDescription");
            } catch (Exception ex) {
            }

            try {
                ident = resBundle.getString("DCR.Label.Identifier");
            } catch (Exception ex) {
            }

            try {
                id = resBundle.getString("DCR.Label.Id");
            } catch (Exception ex) {
            }

            try {
                profs = resBundle.getString("DCR.Label.Profiles");
            } catch (Exception ex) {
            }
        }

        profPanel.setBorder(new TitledBorder(selProf));
        catPanel.setBorder(new TitledBorder(selCat));
        sortModeLabel.setText(sortCat);
        descPanel.setBorder(new TitledBorder(desc));
        identifierKeyLabel.setText(ident);
        idKeyLabel.setText(id);
        profilesKeyLabel.setText(profs);

        sortModes = new String[3];
        sortModes[0] = alpha;
        sortModes[1] = byId;
        sortModes[2] = byBC;

        sortModeCombo.removeItemListener(this);
        sortModeCombo.removeAllItems();

        for (int i = 0; i < sortModes.length; i++) {
            sortModeCombo.addItem(sortModes[i]);
        }

        sortModeCombo.addItemListener(this);
    }

    /**
     * Returns the selected categories.
     *
     * @return the selected categories
     */
    public List getSelectedCategories() {
        List selected = new ArrayList();

        if (sortMode != BY_CONCEPT) {
            // get from the jlist
            if (catList.getSelectedIndex() > -1) {
                Object[] vals = catList.getSelectedValues();

                for (int i = 0; i < vals.length; i++) {
                    selected.add(vals[i]);
                }
            }
        } else {
            // get from the jtree 
            if (catTree.getSelectionCount() > 0) {
                TreePath[] paths = catTree.getSelectionPaths();
                Object node;

                for (int i = 0; i < paths.length; i++) {
                    node = paths[i].getLastPathComponent();

                    if (node instanceof DefaultMutableTreeNode) {
                        selected.add(((DefaultMutableTreeNode) node).getUserObject());
                    }
                }
            }
        }

        return selected;
    }

    /**
     * Adds the profiles in the specified list to the list of selected profiles
     *
     * @param profiles a List containing profile names
     */
    public void addProfiles(List<Profile> profiles) {
        if (profiles != null) {
        	Profile profile;
profilesloop: 
            for (int i = 0; i < profiles.size(); i++) {
            	profile = profiles.get(i);

                if (profile == null) {
                    continue;
                }

                for (int j = 0; j < profList.getModel().getSize(); j++) {
                    if (profile.getId().equals(((Profile)profList.getModel().getElementAt(j)).getId())) {
                        continue profilesloop;
                    }
                }

                // not in the list already
                ((DefaultListModel) profList.getModel()).addElement(profile);
            }
        }
    }

    /**
     * Adds the data categories in the specified list to the list of selected
     * categories.
     *
     * @param datcats a list containing data category representations
     */
    protected void updateCategories(List datcats) {
        if (datcats == null) {
            //return; // or clear the list??
        	datcats = new ArrayList(0);
        }

        if (sortMode != BY_CONCEPT) {
            catList.removeListSelectionListener(this);
            descArea.setText("");
            catModel.clear();

            for (int i = 0; i < datcats.size(); i++) {
                catModel.addElement(datcats.get(i));
            }

            catList.addListSelectionListener(this);

            if (catScroll.getViewport().getView() != catList) {
                catScroll.setViewportView(catList);
            }
        } else {
            catTree.removeTreeSelectionListener(this);
            descArea.setText("");

            //rootNode.removeAllChildren();// create new root node?
            DCTree treeBuilder = new DCTree();
            rootNode = treeBuilder.getBroaderGenericConceptTree(datcats);

            // the tree should be complete, have to make a new tree?
            catTree = new JTree(rootNode);
            ((DefaultTreeCellRenderer) catTree.getCellRenderer()).setLeafIcon(null);
            ((DefaultTreeCellRenderer) catTree.getCellRenderer()).setOpenIcon(null);
            ((DefaultTreeCellRenderer) catTree.getCellRenderer()).setClosedIcon(null);
            catTree.setRootVisible(false);
            catTree.setShowsRootHandles(true);
            catTree.addTreeSelectionListener(this);

            if (catScroll.getViewport().getView() != catTree) {
                catScroll.setViewportView(catTree);
            }

            catTree.revalidate();
        }

        catScroll.revalidate();
    }

    /**
     * Returns a list of data categories belonging to the specified profile.
     *
     * @param profile the ID of the profile
     *
     * @return a list of data categories
     */
    protected List getDataCategories(Profile profile) {
        List datCats = null;

        try {
            datCats = connector.getDCSmallList(profile.getId(), null);

            if (datCats.size() == 0) {
            	String message;
            	if (resBundle != null) {
            		message = resBundle.getString("DCR.Message.NoCategories");
            	} else {
            		message = "No categories available in this profile";
            	}
                JOptionPane.showMessageDialog(this, 
                		message, "", 
                		JOptionPane.INFORMATION_MESSAGE);
            }
            
            if (sortMode == BY_ALPHA) {
                Collections.sort(datCats, new DCIdentifierComparator());
            } else if (sortMode == BY_ID) {
                Collections.sort(datCats, new DCIdComparator());
            }
        } catch (DCRConnectorException dce) {
        	String message;
        	if (resBundle != null) {
        		message = resBundle.getString("DCR.Message.NoConnection");
        	} else {
        		message = "Could not connect to the registry: ";
        	}
            JOptionPane.showMessageDialog(this, (message + " " + dce.getMessage()), "", JOptionPane.ERROR_MESSAGE);
        } catch (IllegalArgumentException iae) {
        	String message;
        	if (resBundle != null) {
        		message = resBundle.getString("DCR.Message.NoConnection");
        	} else {
        		message = "Could not connect to the registry: ";
        	}
            JOptionPane.showMessageDialog(this, (message + " " + iae.getMessage()), "", JOptionPane.ERROR_MESSAGE);
        }

        currentProfile = profile;

        return datCats;
    }

    /**
     * Resorts the categories.
     */
    protected void reorderCategories() {
        ArrayList cats = new ArrayList();

        if (catScroll.getViewport().getView() == catList) {
            //catList.removeListSelectionListener(this);
            int size = catModel.getSize();
            cats.ensureCapacity(size);

            for (int i = 0; i < size; i++) {
                cats.add(catModel.get(i));
            }
        } else {
            //catTree.removeTreeSelectionListener(this);
            int size = rootNode.getChildCount();
            cats.ensureCapacity(size);

            Enumeration en = rootNode.breadthFirstEnumeration();
            en.nextElement(); // skip root

            while (en.hasMoreElements()) {
                cats.add(((DefaultMutableTreeNode) en.nextElement()).getUserObject());
            }
        }

        if (sortMode == BY_ALPHA) {
            Collections.sort(cats, new DCIdentifierComparator());
        } else if ((sortMode == BY_ID) || (sortMode == BY_CONCEPT)) {
            // sort by id first before creating the tree?
            Collections.sort(cats, new DCIdComparator());
        }

        updateCategories(cats);
    }

    /**
     * Updates the information in the description panel.
     *
     * @param dc the summary of the selected data category
     */
    protected void updateDescription(DCSmall dc) {
        if (dc != null) {
            identifierValLabel.setText(dc.getIdentifier());
            idValLabel.setText(dc.getId());

            if (dc.getProfiles().length == 1) {
                profilesValLabel.setText(dc.getProfiles()[0].getName());
            } else {
                StringBuffer buf = new StringBuffer();

                for (int i = 0; i < dc.getProfiles().length; i++) {
                    buf.append(dc.getProfiles()[i].getName());

                    if (i != (dc.getProfiles().length - 1)) {
                        buf.append(", ");
                    }
                }

                profilesValLabel.setText(buf.toString());
            }

            descArea.setText(dc.getDesc());
        } else {
            identifierValLabel.setText(EMPTY);
            idValLabel.setText(EMPTY);
            profilesValLabel.setText(EMPTY);
            descArea.setText("");
        }
    }

    /**
     * Empty action event handling.
     *
     * @param e the action event
     */
    public void actionPerformed(ActionEvent e) {
    }

    /**
     * Handles a change in the sort mode.
     *
     * @param e the item event
     */
    public void itemStateChanged(ItemEvent e) {
        if (e.getSource() == sortModeCombo) {
            // the text can be locale dependent, use the index (the order of the values is known)
            // or store the actual values in a hashmap to determine which mode is selected
            int oldMode = sortMode;

            int i = sortModeCombo.getSelectedIndex();

            switch (i) {
            case 0:
                sortMode = BY_ALPHA;

                break;

            case 1:
                sortMode = BY_ID;

                break;

            case 2:
                sortMode = BY_CONCEPT;

                break;
            }

            if (sortMode != oldMode) {
                reorderCategories();
                updateDescription(null);
            }
        }
    }

    /**
     * Handles changes in the category selection
     *
     * @param e list selection event
     */
    public void valueChanged(ListSelectionEvent e) {
        if (e.getSource() == profList) {
            if (!e.getValueIsAdjusting()) {
                //System.out.println("List sel...");
            	updateDescription(null);
                if (profList.getSelectedIndex() > -1) {
                    Profile profile = (Profile) profList.getSelectedValue();
                    updateCategories(getDataCategories(profile));
 
                } else {
                    // empty the category list/tree
                    currentProfile = null;
                    updateCategories(new ArrayList());
                }
            }
        } else if (e.getSource() == catList) {
            if ((catList.getSelectedIndex() > -1) &&
                    (catList.getSelectedIndices().length == 1)) {
                Object sel = catList.getSelectedValue();

                if (sel instanceof DCSmall) {
                    updateDescription((DCSmall) sel);

                    //descArea.setText(((DCSmall) sel).getDesc());
                } else {
                    updateDescription(null);

                    //descArea.setText("");
                }
            } else {
                updateDescription(null);

                //descArea.setText("");	
            }
        }
    }

    /**
     * Handles a change in the selection of a data category in the tree view.
     *
     * @param e tree selection event
     */
    public void valueChanged(TreeSelectionEvent e) {
        if (catTree.getSelectionCount() == 1) {
            TreePath lead = e.getNewLeadSelectionPath();

            if (lead != null) {
                Object sel = lead.getLastPathComponent();

                if (sel instanceof DefaultMutableTreeNode) {
                    Object uo = ((DefaultMutableTreeNode) sel).getUserObject();

                    if (uo instanceof DCSmall) {
                        updateDescription((DCSmall) uo);

                        //descArea.setText(((DCSmall) uo).getDesc());
                    } else {
                        updateDescription(null);

                        //descArea.setText("");
                    }
                }
            }
        }
    }
}
