package mpi.dcr;

import java.awt.GridBagConstraints;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.ResourceBundle;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import mpi.dcr.isocat.Profile;


/**
 * A panel with user interface elements to interact with a locally stored  data
 * category selection.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class LocalDCSelectPanel extends AbstractDCSelectPanel {
    /** add categories button */
    protected JButton addCatsButton;

    /** remove categories button */
    protected JButton removeCatsButton;
    private List<DCSmall> allDatCats;
    private List<Profile> locProfList;
    private boolean singleSelection = true;

    /**
     * Creates a new LocalDCSelectPanel instance
     *
     * @param connector the (local) connector to use
     */
    public LocalDCSelectPanel(ILATDCRConnector connector) {
        super(connector);

        if (connector instanceof LocalDCRConnector) {
            this.connector = (LocalDCRConnector) connector;
        }

        initComponents();
    }

    /**
     * Creates a new LocalDCSelectPanel instance
     *
     * @param connector the connector
     * @param resBundle a resource bundle with localized strings
     */
    public LocalDCSelectPanel(ILATDCRConnector connector,
        ResourceBundle resBundle) {
        super(connector, resBundle);

        if (connector instanceof LocalDCRConnector) {
            this.connector = (LocalDCRConnector) connector;
        }

        initComponents();
    }

    /**
     * Creates a new LocalDCSelectPanel instance
     */
    public LocalDCSelectPanel() {
        super();
        initComponents();
    }

    /**
     * Initializes ui components.
     */
    protected void initComponents() {
        locProfList = new ArrayList<Profile>();

        try {
            allDatCats = connector.getDCSmallList(null, null);

            if (allDatCats == null) {
                allDatCats = new ArrayList<DCSmall>();
            }
        } catch (DCRConnectorException dce) {
            allDatCats = new ArrayList<DCSmall>();
        }

        // categories panel
        addCatsButton = new JButton();
        removeCatsButton = new JButton();

        super.initComponents();

        Insets insets = new Insets(2, 6, 2, 6);
        addCatsButton.addActionListener(this);
        removeCatsButton.addActionListener(this);

        JPanel buttonPanel = new JPanel(new GridLayout(2, 1, 6, 2));
        buttonPanel.add(addCatsButton);
        buttonPanel.add(removeCatsButton);

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        gbc.gridy = 3;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 0.0;
        gbc.weighty = 0.0;
        catPanel.add(buttonPanel, gbc);

        updateLocale();
        setSingleSelection(singleSelection);

        DCSmall dcsmall = null;
        Profile prof = null;

        for (int i = 0; i < allDatCats.size(); i++) {
            dcsmall = (DCSmall) allDatCats.get(i);

            for (int j = 0; j < dcsmall.getProfiles().length; j++) {
                prof = dcsmall.getProfiles()[j];

                if (!locProfList.contains(prof)) {
                    locProfList.add(prof);
                }
            }
        }

        addProfiles(locProfList);
        profList.setSelectedIndex(0);
    }

    /**
     * Applies localized texts to elements, if provided.
     */
    protected void updateLocale() {
        super.updateLocale();

        String ap = "All Profiles";
        String addCat = "Add Categories";
        String remCat = "Remove Category";

        if (resBundle != null) {
            try {
                ap = resBundle.getString("DCR.Label.AllProfiles");
            } catch (Exception ex) {
            }

            try {
                addCat = resBundle.getString("DCR.Label.AddCategories");
            } catch (Exception ex) {
            }

            try {
                remCat = resBundle.getString("DCR.Label.RemoveCategory");
            } catch (Exception ex) {
            }
        }

        locProfList.add(0, new Profile("0", ap));
        addCatsButton.setText(addCat);
        removeCatsButton.setText(remCat);
        /*
           sortModeCombo.removeItemListener(this);
           sortModeCombo.removeAllItems();
           for (int i = 0; i < sortModes.length; i++) {
               sortModeCombo.addItem(sortModes[i]);
           }
         */
        sortModeCombo.addItemListener(this);
    }

    /**
     * Returns whether only one category can be selected or more than one.
     *
     * @return Returns the singleSelection flag
     */
    public boolean isSingleSelection() {
        return singleSelection;
    }

    /**
     * Sets whether only one category can be selected or more than one.
     *
     * @param singleSelection the single selection flag
     */
    public void setSingleSelection(boolean singleSelection) {
        this.singleSelection = singleSelection;

        if (catTree != null) {
            if (singleSelection) {
                catTree.getSelectionModel()
                       .setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            } else {
                catTree.getSelectionModel()
                       .setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
            }
        }

        if (catList != null) {
            if (singleSelection) {
                catList.getSelectionModel()
                       .setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
            } else {
                catList.getSelectionModel()
                       .setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
            }
        }
    }

    /**
     * Creates a dialog to connect to a remote DCR and adds the selected
     * categories to the cache.
     */
    protected void selectAndAddCategories() {
        //String dcrLoc = "http://lux12.mpi.nl/isocat/rpc/syntax";
        //String dcrName = "SYNTAX ISO12620";
        // accept the default
        /*
           LATDCRConnector remConnector = new LATDCRConnector();
           //remConnector.setDCRLocation(dcrLoc);
           //remConnector.setName(dcrName);
           RemoteDCSelectPanel dcsp = new RemoteDCSelectPanel(remConnector);
           //dcsp.addProfiles(pp.getSelectedProfileNames());
           JDialog catFrame = new JDialog();
           catFrame.setModal(true);
           catFrame.setContentPane(dcsp);
           catFrame.pack();
           catFrame.setVisible(true);// blocks
           List dcsToAdd = dcsp.getSelectedCategories();
           addCategories(dcsToAdd);
         */

    }

    /**
     * Adds the categories from the specified list to the local list.
     *
     * @param dcsToAdd the data categories to add
     */
    protected void addCategories(List dcsToAdd) {
        if ((dcsToAdd != null) && (dcsToAdd.size() > 0)) {
            // add the new categories
            List<DCSmall> addedDcs = new ArrayList<DCSmall>(dcsToAdd.size());
            List<Profile> addedProfiles = new ArrayList<Profile>(dcsToAdd.size());
            DCSmall small1 = null;
            DCSmall small2 = null;
dcloop: 
            for (int i = 0; i < dcsToAdd.size(); i++) {
                small1 = (DCSmall) dcsToAdd.get(i);

                for (int k = 0; k < small1.getProfiles().length; k++) {
                    if (!locProfList.contains(small1.getProfiles()[k])) {
                        if (!addedProfiles.contains(small1.getProfiles()[k])) {
                            addedProfiles.add(small1.getProfiles()[k]);
                        }
                    }
                }

                for (int j = 0; j < allDatCats.size(); j++) {
                    small2 = (DCSmall) allDatCats.get(j);

                    if (small1.getIdentifier().equals(small2.getIdentifier())) {
                        // already in the list
                        continue dcloop;
                    }
                }

                // not in the list
                addedDcs.add(small1);
            }

            if (addedDcs.size() > 0) {
                allDatCats.addAll(addedDcs);

                if (connector instanceof LocalDCRConnector) {
                    try {
                        ((LocalDCRConnector) connector).addDataCategories(addedDcs);
                    } catch (DCRConnectorException dce) {
                        // message
                    }
                }

                if (sortMode == BY_ALPHA) {
                    Collections.sort(allDatCats, new DCIdentifierComparator());
                } else if (sortMode == BY_ID) {
                    Collections.sort(allDatCats, new DCIdComparator());
                }

                // if there is a tree we could add to the tree directly?
                if (profList.getSelectedIndex() == 0) {
                    updateCategories(allDatCats);
                } else {
                    updateCategories(getDataCategories(
                            (Profile) profList.getSelectedValue()));
                }
            }

            if (addedProfiles.size() > 0) {
                locProfList.addAll(addedProfiles);

                for (int i = 0; i < addedProfiles.size(); i++) {
                    profModel.addElement(addedProfiles.get(i));
                }
            }

            profList.setSelectedIndex(0);
        }
    }

    private void removeCategory() {
        Object sel = null;

        if (sortMode == BY_CONCEPT) { // tree

            TreePath selPath = catTree.getSelectionPath();

            if (selPath != null) {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) selPath.getLastPathComponent();

                if (node.isRoot()) {
                    return;
                }

                sel = node.getUserObject();

                int chCount = node.getChildCount();

                if (chCount > 0) { //reparent children??

                    List childNodes = new ArrayList(chCount);
                    TreeNode tn = node.getFirstChild();
                    childNodes.add(tn);

                    while ((tn = node.getChildAfter(tn)) != null) {
                        childNodes.add(tn);
                    }

                    DefaultMutableTreeNode parentNode = (DefaultMutableTreeNode) node.getParent();
                    node.removeAllChildren();
                    chCount = parentNode.getChildCount();

                    int index = parentNode.getIndex(node);
                    parentNode.remove(node);

                    for (int i = 0; i < childNodes.size(); i++) {
                        if (index == (chCount - 1)) {
                            parentNode.add((DefaultMutableTreeNode) childNodes.get(
                                    i));
                        } else {
                            parentNode.insert((DefaultMutableTreeNode) childNodes.get(
                                    i), index + i);
                        }
                    }

                    ((DefaultTreeModel) catTree.getModel()).reload(parentNode);
                } else {
                    ((DefaultTreeModel) catTree.getModel()).removeNodeFromParent(node);
                }
            }
        } else {
            sel = catList.getSelectedValue();

            if (sel instanceof DCSmall) {
                allDatCats.remove(sel);
                catModel.removeElement(sel);
            }
        }

        if ((sel != null) && connector instanceof LocalDCRConnector) {
            List remDatCats = new ArrayList(1);
            remDatCats.add(sel);

            try {
                ((LocalDCRConnector) connector).removeCategories(remDatCats);
            } catch (DCRConnectorException dce) {
                // message
            }
        }
    }

    /**
     * @see org.mpi.DCR.AbstractDCSelectPanel#updateCategories(java.util.List)
     */
    protected void updateCategories(List datcats) {
        super.updateCategories(datcats);

        if (sortMode == BY_CONCEPT) { // tree has been recreated

            if (singleSelection) {
                catTree.getSelectionModel()
                       .setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            } else {
                catTree.getSelectionModel()
                       .setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
            }
        }
    }

    /**
     * The action event handling.
     *
     * @param e the action event
     */
    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == addCatsButton) {
            selectAndAddCategories();
        } else if (e.getSource() == removeCatsButton) {
            removeCategory();
        }
    }

    /**
     * The list selection event handling.
     *
     * @param e list selection event
     */
    public void valueChanged(ListSelectionEvent e) {
        if (e.getSource() == profList) {
            if (!e.getValueIsAdjusting()) {
                //System.out.println("List sel...");
                if (profList.getSelectedIndex() == 0) { // all profs
                                                        // empty the category list/tree
                    currentProfile = null;
                    updateCategories(allDatCats);
                } else {
                    super.valueChanged(e);
                }
            }

            updateDescription(null);
        } else {
            super.valueChanged(e);
        }
    }
}
