package mpi.eudico.client.annotator.gui;

import java.awt.Dialog;
import java.awt.Frame;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;

import mpi.eudico.client.annotator.commands.global.WebMA;

/**
 * Utility class for creating a Frame or a Dialog with a simple html viewer in a scrollpane.
 * Based on the SimpleHTMLViewer in nl.mpi.util
 * 
 * @author Han Sloetjes
 */
public class HTMLViewer implements HyperlinkListener {
	private final String urlString;
	private URL url;
	private String title;
	private boolean modal;
	private final JEditorPane htmlPane;
	
	/**
	 * Constructor, throws an exception in case of an io error
	 * @param urlString the url as string
	 * @param modal modal flag
	 * @param title the title for from or dialog
	 * @throws IOException when the file cannot be found
	 */
	public HTMLViewer(String urlString, boolean modal, String title) throws IOException {
		super();
		this.urlString = urlString;
		// first try if this is a local resource
		url = HTMLViewer.class.getResource(urlString);
		if (url == null) {
			try {
				url = new URL(urlString);
			} catch (MalformedURLException mue) {
				throw new IOException("Cannot load file: " + mue.getMessage());
			}
		}
		
		this.title = title;
		this.modal = modal;
		
		htmlPane = new JEditorPane(url);
		htmlPane.setEditable(false);
		htmlPane.addHyperlinkListener(this);
	}

	/**
	 * Creates a dialog with the specified frame as parent.
	 * 
	 * @param parent the parent frame
	 * @return a dialog
	 */
	public JDialog createHTMLDialog(Frame parent) {
		JDialog dialog = new JDialog(parent, title, modal);
		JScrollPane scrollPane = new JScrollPane(htmlPane);
		dialog.getContentPane().add(scrollPane);
		
		return dialog;
	}
	
	
	/**
	 * Creates a dialog with the specified dialog as parent.
	 * 
	 * @param parent the parent dialog
	 * @return a dialog
	 */
	public JDialog createHTMLDialog(Dialog parent) {
		JDialog dialog = new JDialog(parent, title, modal);
		JScrollPane scrollPane = new JScrollPane(htmlPane);
		dialog.getContentPane().add(scrollPane);
		
		return dialog;
	}
	
	/**
	 * For a frame the modal flag is used to determine whether the frame 
	 * should always be on top.
	 * 
	 * @return a Frame for the html page
	 */
	public JFrame createHTMLFrame() {
		JFrame frame = new JFrame(title);
		JScrollPane scrollPane = new JScrollPane(htmlPane);
		frame.setAlwaysOnTop(modal);
		frame.getContentPane().add(scrollPane);
		
		return frame;
	}
	
	/**
	 * Handles hyperlink events. 
	 *
	 */
	public void hyperlinkUpdate(HyperlinkEvent e) {
		if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
			//only care if link is to same document
			if (e.getURL().sameFile(url)) {
				try {
					htmlPane.scrollToReference(e.getURL().getRef());
				} catch (Throwable t) {
					//t.printStackTrace();
				}
			} else {
				// try to load the external link?
				// Makes no sense if it is not possible to go back
				// Open in browser instead?
				/*
				try {
					htmlPane.setPage(e.getURL());
				} catch (IOException ioe) {
					// warn?
				}
				*/
				// in Java 1.6 > use Desktop to browse the file in a browser
				// abuse WebMA for the time being
				WebMA webma = new WebMA("", null, e.getURL().toExternalForm());
				webma.actionPerformed(null);
			}
		}
	}

}
