package mpi.eudico.client.annotator.lexicon.api;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import mpi.eudico.client.annotator.util.ClientLogger;
import mpi.eudico.server.corpora.lexicon.LexiconServiceClientFactory;
import mpi.eudico.util.ExtClassLoader;

import org.xml.sax.SAXException;

/**
 * This class looks for LexiconService extensions in the designated directory and loads them.
 * So far, only a combination of a CMDI file and a JAR file is supported.
 * @author Micha Hulsbosch
 *
 */
public class LexSrvcAvailabilityDetector {
	private static Map<String, LexSrvcClntBundle> lexSrvcClntBundles = new HashMap<String, LexSrvcClntBundle>();
	private static LexSrvcClntLoader lexSrvcClntLoader;
	
	static {
		init();
	}
	
	/**
	 * Private constructor
	 */
	private LexSrvcAvailabilityDetector() {
		super();
	}

	/**
	 * Tries to find extensions in the designated directory
	 */
	private static void init() {
		File extFile = new File (ExtClassLoader.EXTENSIONS_DIR);
		
		if (!extFile.exists()) {
			ClientLogger.LOG.warning("The extension folder could not be found (" + ExtClassLoader.EXTENSIONS_DIR + ").");
			return;
		}
		if (!extFile.isDirectory()) {
			ClientLogger.LOG.warning("The extension \'folder\' is not a folder (" + ExtClassLoader.EXTENSIONS_DIR + ").");
			return;
		}
		if (!extFile.canRead()) {
			ClientLogger.LOG.warning("The extension folder is not accessible (" + ExtClassLoader.EXTENSIONS_DIR + ").");
			return;
		}
		File[] files = extFile.listFiles();
		for (File f : files) {
			if (f.isDirectory()) {
				detectLexSrvcClnt(f);
			} else {
				//load from jar
				String name = f.getName().toLowerCase();
				if (name.endsWith("jar")) {
					detectLexSrvcClntFromJar(f);
				} 
//				else if (name.endsWith("zip")) {
//					detectRecognizerFromZip(f);
//				}
			}
		}
	}
	
	/**
	 * Creates and returns the Lexicon Service Client Factories that are found in the designated extensions
	 * directory.
	 * @return a HashMap containing the Lexicon Service Client Factories identified by their type (name).
	 */
	public static HashMap<String, LexiconServiceClientFactory> getLexiconServiceClientFactories() {
		HashMap<String, LexiconServiceClientFactory> lexSrvcClnts = new HashMap<String, LexiconServiceClientFactory>(6);
		
		Iterator<String> keyIt = lexSrvcClntBundles.keySet().iterator();
		String key;
		LexSrvcClntBundle bundle;
		LexiconServiceClientFactory clientFactory;
		while (keyIt.hasNext()) {
			key = keyIt.next();
			bundle = lexSrvcClntBundles.get(key);
			
			if (bundle.getJavaLibs() != null) {
				
				if (lexSrvcClntLoader == null) {
					lexSrvcClntLoader = new LexSrvcClntLoader(bundle.getJavaLibs(), bundle.getNativeLibs());
				} else {
					lexSrvcClntLoader.addLibs(bundle.getJavaLibs());
					lexSrvcClntLoader.addNativeLibs(bundle.getNativeLibs());
				}
				
				try {
					clientFactory = (LexiconServiceClientFactory) Class.forName(bundle.getLexSrvcClntClass(), true, lexSrvcClntLoader).newInstance();
					clientFactory.setType(bundle.getName());
					clientFactory.setDescription(bundle.getDescription());
					for(Param p : bundle.getParamList()) {
						if(p.getType().equals("defaultUrl")) {
							clientFactory.setDefaultUrl(p.getContent());
						} else if(p.getType().equals("searchConstraint")) {
							clientFactory.addSearchConstraint(p.getContent());
						}
					}
					lexSrvcClnts.put(key, clientFactory);
				} catch (ClassNotFoundException cnfe) {
					ClientLogger.LOG.severe("Cannot load the lexicon service client class: " + bundle.getLexSrvcClntClass() + " - Class not found");
				} catch (InstantiationException ie) {
					ClientLogger.LOG.severe("Cannot instantiate the lexicon service client class: " + bundle.getLexSrvcClntClass());
				} catch (IllegalAccessException iae) {
					ClientLogger.LOG.severe("Cannot access the lexicon service client class: " + bundle.getLexSrvcClntClass());
				} catch (Exception ex) {// any other exception
					ClientLogger.LOG.severe("Cannot load the lexicon service client: " + bundle.getLexSrvcClntClass() + " - " + ex.getMessage());
					ex.printStackTrace();
				}
			} else {
				ClientLogger.LOG.severe("Cannot load the lexicon service client: no Java library has been found ");
			}
		}
		
		return lexSrvcClnts;
		
	}
	
	/**
	 *  Returns the number of detected service client factories.
	 *  
	 * @return the number of detected service client factories
	 */
	public static int getNumberOfFactories() {
		return lexSrvcClntBundles.size();
	}
	
	/**
	 * Returns a list of names of detected lexicon service client factories.
	 * 
	 * @return a list of names of detected lexicon service client factories
	 */
	public static List<String> getFactoryNames() {
		if (lexSrvcClntBundles.size() > 0) {
			ArrayList<String> names = new ArrayList<String>(lexSrvcClntBundles.size());
			names.addAll(lexSrvcClntBundles.keySet());
			return names;
		}
		
		return null;
	}
	
	/**
	 * Creates and returns a service factory of a given name.
	 * 
	 * @param name the name of the factory
	 * @return a new instance of the factory or null
	 */
	public static LexiconServiceClientFactory getFactoryByName(String name) {

		LexSrvcClntBundle bundle = lexSrvcClntBundles.get(name);
		if (bundle == null) {
			return null;
		}
		
		if (bundle.getJavaLibs() != null) {
			
			if (lexSrvcClntLoader == null) {
				lexSrvcClntLoader = new LexSrvcClntLoader(bundle.getJavaLibs(), bundle.getNativeLibs());
			} else {
				lexSrvcClntLoader.addLibs(bundle.getJavaLibs());
				lexSrvcClntLoader.addNativeLibs(bundle.getNativeLibs());
			}
			
			try {
				LexiconServiceClientFactory clientFactory = (LexiconServiceClientFactory) Class.forName(bundle.getLexSrvcClntClass(), 
						true, lexSrvcClntLoader).newInstance();
				clientFactory.setType(bundle.getName());
				clientFactory.setDescription(bundle.getDescription());
				for(Param p : bundle.getParamList()) {
					if(p.getType().equals("defaultUrl")) {
						clientFactory.setDefaultUrl(p.getContent());
					} else if(p.getType().equals("searchConstraint")) {
						clientFactory.addSearchConstraint(p.getContent());
					}
				}
				return clientFactory;
			} catch (ClassNotFoundException cnfe) {
				ClientLogger.LOG.severe("Cannot load the lexicon service client class: " + bundle.getLexSrvcClntClass() + " - Class not found");
			} catch (InstantiationException ie) {
				ClientLogger.LOG.severe("Cannot instantiate the lexicon service client class: " + bundle.getLexSrvcClntClass());
			} catch (IllegalAccessException iae) {
				ClientLogger.LOG.severe("Cannot access the lexicon service client class: " + bundle.getLexSrvcClntClass());
			} catch (Exception ex) {// any other exception
				ClientLogger.LOG.severe("Cannot load the lexicon service client: " + bundle.getLexSrvcClntClass() + " - " + ex.getMessage());
				ex.printStackTrace();
			}
		} else {
			ClientLogger.LOG.severe("Cannot load the lexicon service client: no Java library has been found ");
		}

		return null;
	}

	/**
	 * Detects whether there is a Lexicon Service Client Factory in the found JAR.
	 * @param file
	 */
	private static void detectLexSrvcClntFromJar(File file) {
		try {
			JarFile jFile = new JarFile(file);
			JarEntry je = jFile.getJarEntry("lexiconServiceClient.cmdi");
			if (je == null) {
				ClientLogger.LOG.warning("No plug-in metadata file (lexiconServiceClient.cmdi) found in " + file.getName());
				return;
			}
			try {
				URL jarUrl = file.toURI().toURL();
				URL[] libUrls = new URL[]{jarUrl};
				InputStream mdStream = jFile.getInputStream(je);
				createBundle(mdStream, libUrls, null, file.getParentFile());
			} catch (MalformedURLException mue) {
				ClientLogger.LOG.severe("Cannot create URL for file: " + file.getName());
			} catch (IOException ioe) {
				ClientLogger.LOG.warning("Cannot read the cmdi file from the jar file: " + file.getName());
			}
			
		} catch (IOException ioe) {
			ClientLogger.LOG.warning("Cannot read the jar file: " + file.getName());
		}
	}

	/**
	 * Detects whether there is a Lexicon Service Client Factory in a directory.
	 * @param the directory
	 */
	private static void detectLexSrvcClnt(File file) {
		File[] files = file.listFiles();
		File mdFile = null;
		
		for (File f : files) {
			if (f.getName().toLowerCase().equals("lexiconserviceclient.cmdi")) {
				mdFile = f;
				break;
			}
		}
		
		if (mdFile != null) {
			List<URL> libs = new ArrayList<URL>();
			List<URL> natLibs = new ArrayList<URL>();
			addLibs(file, libs, natLibs);
			
			try {
				InputStream mdStream = new FileInputStream(mdFile);
				URL[] libUrls = null;
				if (libs.size() > 0) {
					libUrls = (URL[]) libs.toArray(new URL[]{});
				}
				URL[] natLibUrls = null;
				if (natLibs.size() > 0) {
					natLibUrls = (URL[]) natLibs.toArray(new URL[]{});
				}
				
				createBundle(mdStream, libUrls, natLibUrls, file);
			} catch (FileNotFoundException fnfe) {
				ClientLogger.LOG.severe("Cannot parse metadata file: " + fnfe.getMessage());
			}
		} else {
			ClientLogger.LOG.info("No lexiconServiceClient.cmdi metadata file found in: " + file.getName());
		}
	}

	/**
	 * Creates a bundle containing information that can be used to create a Lexicon Service Client Factory.
	 * @param the lexicon stream
	 * @param the libraries
	 * @param the native libraries
	 * @param the baseDir
	 */
	private static void createBundle(InputStream lexStream, URL[] libs, URL[] natLibs, File baseDir) {
		boolean isLexClient = false;
		String binaryName = null;
		LexSrvcClntBundle bundle = null;
		LexSrvcClntParser parser = null;
		
		try {
			parser = new LexSrvcClntParser(lexStream);
			parser.parse();
			/*
			if (parser.type == null || (!parser.type.equals("direct") && !parser.type.equals("local"))) {
				ClientLogger.LOG.warning("Unsupported lexicon service client type, should be 'direct' or 'local': " + parser.type);
				return;
			}
			*/
			if (!parser.curOsSupported) {
				ClientLogger.LOG.warning("Lexicon service client does not support this Operating System: " + parser.name);
				return;
			}
			if (parser.implementor != null) {
				isLexClient = true;
				binaryName = parser.implementor;
			} else {
				ClientLogger.LOG.warning("The implementing class name has not been specified.");
				return;
			}
			
		} catch (SAXException sax) {
			ClientLogger.LOG.severe("Cannot parse metadata file: " + sax.getMessage());
		}
		
		if (isLexClient) {
			//if (parser.type.equals("direct")) {
				// create a classloader, bundle			
				LexSrvcClntLoader loader = new LexSrvcClntLoader(libs, natLibs);
	
				if (binaryName != null) {
					try {
						Class<?> c = loader.loadClass(binaryName);
						
						// if the above works, assume everything is all right
						bundle = new LexSrvcClntBundle();
						bundle.setName(parser.name);
						bundle.setDescription(parser.description);
						bundle.setParamList(parser.paramList);
						bundle.setLexSrvcClntClass(binaryName);
						//bundle.setLexExecutionType(parser.type);
						bundle.setJavaLibs(libs);
						bundle.setNativeLibs(natLibs);
						bundle.setBaseDir(baseDir);
							
						lexSrvcClntBundles.put(bundle.getName(), bundle);
					} catch (ClassNotFoundException cne) {
						ClientLogger.LOG.severe("Cannot load the lexicon service client class: " + binaryName + " - Class not found");
					} 
				}
				else {
					ClientLogger.LOG.warning("Cannot load the lexicon service client class: Class not found");
				}
			/*
			} else if (parser.type.equals("local")) {
				
				bundle = new LexSrvcClntBundle();
				bundle.setName(parser.name);
				bundle.setName(parser.description);// friendly name
				bundle.setParamList(parser.paramList);
				bundle.setLexSrvcClntClass(binaryName);// reuse the class name for the run command
				bundle.setLexExecutionType(parser.type);
				bundle.setBaseDir(baseDir);
				
				lexSrvcClntBundles.put(bundle.getName(), bundle);
			}
			*/			
		}
	}

	/**
	 * Adds libraries to the designated lists.
	 * @param a file to add or a directory of which files are to be loaded 
	 * @param the list with libraries
	 * @param the list with native libraries
	 */
	private static void addLibs(File file, List<URL> libs, List<URL> natLibs) {
		File[] files = file.listFiles();
		
		for (File f : files) {
			try {
				if (f.isDirectory()) {
					addLibs(f, libs, natLibs);
				} else {
					String name = f.getName().toLowerCase();
					if (name.endsWith("jar") || name.endsWith("zip")) {
						try {
							libs.add(f.toURI().toURL());
						} catch (MalformedURLException mue) {
							ClientLogger.LOG.severe("Cannot create URL for file: " + f.getName());
						}
					} 
//					else if (name.endsWith("dll") || name.endsWith("so") || name.endsWith("jnilib")) {
//						try {
//							natLibs.add(f.toURI().toURL());
//						} catch (MalformedURLException mue) {
//							ClientLogger.LOG.severe("Cannot create URL for file: " + f.getName());
//						}
//					}
				}
			} catch (SecurityException se) {
				ClientLogger.LOG.warning("Cannot read file: " + f.getName());
			}
		}
	}
}
