package mpi.eudico.client.annotator.linkedmedia;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLayoutManager;
import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.client.annotator.commands.ELANCommandFactory;

import mpi.eudico.client.annotator.player.CocoaQTMediaPlayer;
import mpi.eudico.client.annotator.player.ElanMediaPlayer;
import mpi.eudico.client.annotator.player.EmptyMediaPlayer;
import mpi.eudico.client.annotator.player.JMFGraphicMediaPlayer;
import mpi.eudico.client.annotator.player.NoPlayerException;
import mpi.eudico.client.annotator.player.PlayerFactory;
import mpi.eudico.client.annotator.player.QTMediaPlayer;
import mpi.eudico.client.annotator.svg.SVGPrefs;

import mpi.eudico.client.annotator.util.FileExtension;
import mpi.eudico.client.annotator.util.FileUtility;
import mpi.eudico.client.annotator.util.SystemReporting;

import mpi.eudico.client.annotator.viewer.SignalViewer;

import mpi.eudico.server.corpora.clomimpl.abstr.MediaDescriptor;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.io.File;

import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.swing.JOptionPane;


/**
 * A utility class for creating, checking and updating mediadescriptors.
 *
 * @author Han Sloetjes
 */
public class MediaDescriptorUtil {
    /**
     * Checks the existance of the file denoted by the mediadescriptor.
     *
     * @param md the media descriptor
     *
     * @return true when the file exists, false otherwise
     */
    public static boolean checkLinkStatus(MediaDescriptor md) {
        if ((md == null) || (md.mediaURL == null) ||
                (md.mediaURL.length() < 5)) {
            return false;
        }

        //wwj: return true if rtsp presents
        if (md.mediaURL.startsWith("rtsp")) {
            return true;
        }

        // remove the file: part of the URL, leading slashes are no problem
        int colonPos = md.mediaURL.indexOf(':');
        String fileName = md.mediaURL.substring(colonPos + 1);

        // replace all back slashes by forward slashes
        fileName = fileName.replace('\\', '/');

        File file = new File(fileName);

        if (!file.exists()) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Creates a mediadescriptor for the specified file.
     *
     * @param filePath the full path of the file
     *
     * @return a MediaDescriptor
     */
    public static MediaDescriptor createMediaDescriptor(String filePath) {
        if ((filePath == null) || (filePath.length() == 0)) {
            return null;
        }

        String mediaURL = FileUtility.pathToURLString(filePath);

        if (mediaURL == null) {
            return null;
        }

        String mimeType = null;
        String mediaExtension;

        if (mediaURL.indexOf('.') > -1) {
            mediaExtension = mediaURL.substring(mediaURL.lastIndexOf('.') + 1);
        } else {
            mediaExtension = mediaURL.substring(mediaURL.length() - 3); // of no use, at least with JMF
        }

        mimeType = MediaDescriptorUtil.mimeTypeForExtension(mediaExtension);

        MediaDescriptor md = new MediaDescriptor(mediaURL, mimeType);

        return md;
    }

    /**
     * Creates a Vector of mediadescriptors for the specified files.
     *
     * @param fileNames a collection of files
     *
     * @return a Vector of MediaDescriptors
     */
    public static Vector createMediaDescriptors(Vector fileNames) {
        Vector mediaDescriptors = new Vector();

        if (fileNames == null) {
            return mediaDescriptors;
        }

        MediaDescriptor nextMD = null;
mdloop: 
        for (int i = 0; i < fileNames.size(); i++) {
            String path = (String) fileNames.get(i);
            nextMD = MediaDescriptorUtil.createMediaDescriptor(path);

            if (nextMD == null) {
                continue;
            }

            for (int j = 0; j < mediaDescriptors.size(); j++) {
                MediaDescriptor otherMD = (MediaDescriptor) mediaDescriptors.get(j);

                if (otherMD.mediaURL.equals(nextMD.mediaURL)) {
                    // don't add the same file twice?
                    continue mdloop;
                }

                // should this automatic detection of extracted_from remain??
                if (nextMD.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE) &&
                        MediaDescriptorUtil.isVideoType(otherMD)) {
                    if (FileUtility.sameNameIgnoreExtension(nextMD.mediaURL,
                                otherMD.mediaURL)) {
                        nextMD.extractedFrom = otherMD.mediaURL;
                    }
                }

                if (otherMD.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE) &&
                        MediaDescriptorUtil.isVideoType(nextMD)) {
                    if (FileUtility.sameNameIgnoreExtension(nextMD.mediaURL,
                                otherMD.mediaURL)) {
                        otherMD.extractedFrom = nextMD.mediaURL;
                    }
                }
            }

            mediaDescriptors.add(nextMD);
        }

        return mediaDescriptors;
    }

    /**
     * Returns a mime-type for a given file extension.  Works only for a very
     * limited set of known file types.
     *
     * @param fileExtension the file extension
     *
     * @return a Mime-Type String from the <code>MediaDescriptor</code> class
     */
    public static String mimeTypeForExtension(String fileExtension) {
        if ((fileExtension == null) || (fileExtension.length() < 2)) {
            return MediaDescriptor.UNKNOWN_MIME_TYPE;
        }

        String lowExt = fileExtension.toLowerCase();

        for (int i = 0; i < FileExtension.MPEG_EXT.length; i++) {
            if (lowExt.equals(FileExtension.MPEG_EXT[i])) {
                return MediaDescriptor.MPG_MIME_TYPE;
            }
        }

        for (int i = 0; i < FileExtension.WAV_EXT.length; i++) {
            if (lowExt.equals(FileExtension.WAV_EXT[i])) {
                return MediaDescriptor.WAV_MIME_TYPE;
            }
        }

        for (int i = 0; i < FileExtension.QT_EXT.length; i++) {
            if (lowExt.equals(FileExtension.QT_EXT[i])) {
                return MediaDescriptor.QUICKTIME_MIME_TYPE;
            }
        }

        for (int i = 0; i < FileExtension.MISC_AUDIO_EXT.length; i++) {
            if (lowExt.equals(FileExtension.MISC_AUDIO_EXT[i])) {
                return MediaDescriptor.GENERIC_AUDIO_TYPE;
            }
        }

        for (int i = 0; i < FileExtension.MISC_VIDEO_EXT.length; i++) {
            if (lowExt.equals(FileExtension.MISC_VIDEO_EXT[i])) {
                return MediaDescriptor.GENERIC_VIDEO_TYPE;
            }
        }

        // 2010 HS: add for images
        for (int i = 0; i < FileExtension.IMAGE_MEDIA_EXT.length; i++) {
            if (lowExt.equals(FileExtension.IMAGE_MEDIA_EXT[i])) {
            	if (lowExt.equals("jpg") || lowExt.equals("jpeg")) {
            		return MediaDescriptor.JPEG_TYPE;
            	}
            	return "image/" + lowExt;// add all to MediaDescriptor?
            }
        }
        
        return MediaDescriptor.UNKNOWN_MIME_TYPE;
    }

    /**
     * Returns whether the specified mime type is a known video type.
     *
     * @param mimeType the mime type string
     *
     * @return true if the specified mimetype is known to be a video type
     *
     * @see #isVideoType(MediaDescriptor)
     */
    public static boolean isVideoType(String mimeType) {
        if (mimeType == null) {
            return false;
        }

        return (mimeType.equals(MediaDescriptor.GENERIC_VIDEO_TYPE) ||
        mimeType.equals(MediaDescriptor.MPG_MIME_TYPE) ||
        mimeType.equals(MediaDescriptor.QUICKTIME_MIME_TYPE));
    }

    /**
     * Returns whether the specified MediaDescriptor is of a known video type.
     *
     * @param descriptor the mediadescriptor
     *
     * @return true if the specified mediadescriptor is of a known video type
     *
     * @see #isVideoType(String)
     */
    public static boolean isVideoType(MediaDescriptor descriptor) {
        if (descriptor == null) {
            return false;
        }

        return MediaDescriptorUtil.isVideoType(descriptor.mimeType);
    }

    /**
     * Tries to update the mediaplayers in the viewermanager as well as the
     * layoutmanager and finally sets the mediadescriptors in the
     * transcription.
     *
     * @param transcription the Transcription with the old descriptors
     * @param descriptors the new media descriptors
     */
    public static void updateMediaPlayers(TranscriptionImpl transcription,
        Vector descriptors) {
        if ((transcription == null) || (descriptors == null)) {
            return;
        }

        long mediaTime = 0L;        

        //Vector currentDesc = transcription.getMediaDescriptors();
        ViewerManager2 viewerManager = ELANCommandFactory.getViewerManager(transcription);
        ElanLayoutManager layoutManager = ELANCommandFactory.getLayoutManager(transcription);
        SignalViewer signalViewer = layoutManager.getSignalViewer();
        
        // stop the player before destroying
        if(viewerManager.getMasterMediaPlayer() != null && 
        		viewerManager.getMasterMediaPlayer().isPlaying()){
        	viewerManager.getMasterMediaPlayer().stop();
	    }
        
        mediaTime = viewerManager.getMasterMediaPlayer().getMediaTime();

        // make sure all players are connected
        if (layoutManager.getMode() == ElanLayoutManager.SYNC_MODE) {
            layoutManager.connectAllPlayers();
        }

        // the master media player cannot be removed directly
        // replace the master; the master is added to the slaves
        viewerManager.setMasterMediaPlayer(new EmptyMediaPlayer(
                Integer.MAX_VALUE));

        // remove the slaves
        Vector slavePlayers = viewerManager.getSlaveMediaPlayers();
        Vector remPlayers = new Vector(slavePlayers.size());

        for (int i = 0; i < slavePlayers.size(); i++) {
            remPlayers.add(slavePlayers.get(i));
        }

        for (int i = 0; i < remPlayers.size(); i++) {
            ElanMediaPlayer slave = (ElanMediaPlayer) remPlayers.get(i);
            viewerManager.destroyMediaPlayer(slave);
            layoutManager.remove(slave);
        }

        if (signalViewer != null) {
            viewerManager.destroyViewer(signalViewer);
            layoutManager.remove(signalViewer);
        }
        if (System.getProperty("os.name").indexOf("Mac") > 0) {
	        try {
	        	Thread.sleep(6000);
	        } catch (Exception ex) {
	        	
	        }
        }
        // create new players from the descriptors
        MediaDescriptorUtil.createMediaPlayers(transcription, descriptors);
        
        // check audio recognizer panel
        ArrayList newAudioPaths = new ArrayList(6);
        
        if (layoutManager.getSignalViewer() != null) {
        	newAudioPaths.add(layoutManager.getSignalViewer().getMediaPath());
        }
        
    	// there may be other audio files associated with the transcription
    	MediaDescriptor md;
    	for (int i = 0; i < descriptors.size(); i++) {  		
    		md = (MediaDescriptor) descriptors.get(i);
    		if (md.mimeType.equals(MediaDescriptor.GENERIC_AUDIO_TYPE) || md.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE) ) {    			
    			if (!newAudioPaths.contains(md.mediaURL)) {
    			    newAudioPaths.add(md.mediaURL);
    			}
    		}
    	}
    	
    	
    	if (viewerManager.getAudioRecognizerPanel() != null) {
    		// could check here for changes compared to the old setup
    		viewerManager.getAudioRecognizerPanel().setAudioFilePaths(newAudioPaths);     			
    	}
    	
    	viewerManager.setAudioPaths(newAudioPaths);    	
    		
    	
    	 //check video recognizer panel
        ArrayList newVideoPaths = new ArrayList(6);
                 	
    	for (int i = 0; i < descriptors.size(); i++) {  		
    		md = (MediaDescriptor) descriptors.get(i);
    		if (!md.mimeType.equals(MediaDescriptor.GENERIC_AUDIO_TYPE) && !md.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE) ) {    			
    			if (!newVideoPaths.contains(md.mediaURL)) {
    				newVideoPaths.add(md.mediaURL);
    			}
    		}
    	}
    	
    	if(viewerManager.getVideoRecognizerPanel() != null) {
    			viewerManager.getVideoRecognizerPanel().setVideoFilePaths(newVideoPaths);     			
    	} 
    	viewerManager.setVideoPaths(newVideoPaths);
    	
        viewerManager.getMasterMediaPlayer().setMediaTime(mediaTime); 
        transcription.setMediaDescriptors(descriptors);
        transcription.setChanged();
        
        layoutManager.doLayout();
    }

    /**
     * Tries to create the mediaplayers in the viewermanager as well as the
     * layoutmanager and finally sets the mediadescriptors in the
     * transcription.
     *
     * @param transcription the Transcription with the old descriptors
     * @param descriptors the new media descriptors
     */
    public static void createMediaPlayers(TranscriptionImpl transcription,
        List descriptors) {
        if ((transcription == null) || (descriptors == null)) {
            return;
        }

        int numDesc = descriptors.size();

        try {
            ViewerManager2 viewerManager = ELANCommandFactory.getViewerManager(transcription);
            ElanLayoutManager layoutManager = ELANCommandFactory.getLayoutManager(transcription);         
          
            int nrOfPlayers = 0;
            int nrVisualPlayers = 0;
            String masterMediaURL = null;
            String signalSource = null;
            ElanMediaPlayer signalSourcePlayer = null;
            long signalOffset = 0;
            long masterMediaOffset = 0;
            boolean createSvgPlayer = false;

            if (transcription.getSVGFile() != null /*||SVGPrefs.getUseSVG()*/) {
                // this makes all players QT players for now
                createSvgPlayer = true;
            }

            MediaDescriptor curMD;
            ArrayList failedPlayers = null;
            StringBuffer errors = new StringBuffer();
            
            for (int i = 0; i < numDesc; i++) {
                curMD = (MediaDescriptor) descriptors.get(i);

                if (!MediaDescriptorUtil.checkLinkStatus(curMD)) {
                    continue;
                } 
                
                if ((masterMediaURL != null) &&
                        curMD.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE)) { 
                    // if it is extracted wav set signal source and do not create a player
                    if (masterMediaURL.equals(curMD.extractedFrom)) {
                    	if(signalSourcePlayer != null){
                    		layoutManager.remove(signalSourcePlayer);
                        	viewerManager.destroyMediaPlayer(signalSourcePlayer);                    	
                        }   
                        signalSource = curMD.mediaURL;
                        signalOffset = masterMediaOffset;
                        signalSourcePlayer = viewerManager.getMasterMediaPlayer();
                        viewerManager.setSignalSourcePlayer(viewerManager.getMasterMediaPlayer());
                        continue; // create no player
                    }
                }
                
                if(!curMD.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE) &&
                		nrVisualPlayers == Constants.MAX_VISIBLE_PLAYERS){
                	continue;
                }
               

                try {
                    ElanMediaPlayer player = null;
                    if (createSvgPlayer) {
                    	String preferredMF = System.getProperty("PreferredMediaFramework");
                    	if (PlayerFactory.JMF_MEDIA_FRAMEWORK.equals(preferredMF)) {
                    		SVGPrefs.setUseSVG(true);
	                        player = viewerManager.createMediaPlayer(curMD,
	                                PlayerFactory.JMF_MEDIA_FRAMEWORK);		                       
                    	} else {
	                        player = viewerManager.createMediaPlayer(curMD,
	                                PlayerFactory.QT_MEDIA_FRAMEWORK);	                       
                    	}
                    } else if (curMD.mimeType.equals(
                                MediaDescriptor.QUICKTIME_MIME_TYPE)) {
                    	if (SystemReporting.isMacOS()) {
                    		player = viewerManager.createMediaPlayer(curMD,
                                PlayerFactory.COCOA_QT);	
                    	} else {
                    		player = viewerManager.createMediaPlayer(curMD,
	                                PlayerFactory.QT_MEDIA_FRAMEWORK);                    		
                    	}
                    } else {	
                    	if(curMD.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE)){
                    		if(signalSource == null){
                    			player = viewerManager.createMediaPlayer(curMD);                    			
                    		} 
                    	} else {
                    		player = viewerManager.createMediaPlayer(curMD);
                    		
                    	}
                    }
                    
                    if(player == null){
                    	continue;
                    }
                    
                    nrOfPlayers++;

                    if (nrOfPlayers == 1 && player != null) {
                        // here comes the mastermedia player
                        viewerManager.setMasterMediaPlayer(player);
                        masterMediaURL = curMD.mediaURL;
                        masterMediaOffset = curMD.timeOrigin;

                        // if audio than also signal source until proven later 
                        // otherwise by mpg media source
                        if (curMD.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE)) {
                            signalSource = curMD.mediaURL;
                            signalOffset = curMD.timeOrigin;
                            viewerManager.setSignalSourcePlayer(player);
                        }
                    } else if(curMD.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE)){                    	
                            signalSource = curMD.mediaURL;
                            signalOffset = curMD.timeOrigin;
                            // HS Aug 2008: pass this player to the viewermanager; important for synchronisation mode
                            signalSourcePlayer = player;
                            viewerManager.setSignalSourcePlayer(player);

//                        // if this WAV is extracted from the mastermedia, inherit time-origin
//                        if (masterMediaURL.equals(curMD.extractedFrom)) {
//                            signalSource = curMD.mediaURL;
//                            signalOffset = viewerManager.getMasterMediaPlayer()
//                                                        .getMediaDescriptor().timeOrigin;
//                            if(signalSourcePlayer != null){
//                            	viewerManager.destroyMediaPlayer(signalSourcePlayer);
//                            }
//                            viewerManager.setSignalSourcePlayer(viewerManager.getMasterMediaPlayer());
//                        }

                        // wave player is master media but wav is extracted from this mpg
                        /* Mar 06: block removed: the .wav form as master can have its own offset
                           if (viewerManager.getMasterMediaPlayer()
                                                .getMediaDescriptor().mimeType.equals(
                                       MediaDescriptor.WAV_MIME_TYPE) &&
                                   curMD.mediaURL.equals(
                                       viewerManager.getMasterMediaPlayer()
                                                        .getMediaDescriptor().extractedFrom)) {
                               masterMediaOffset = curMD.timeOrigin;
                               // march 06: bug: the video is set as the source for the signal viewer
                               //signalSource = curMD.mediaURL;// bug...
                               //signalOffset = curMD.timeOrigin;// bug...
                               //viewerManager.setSignalSourcePlayer(player);
                               // should the offset of the master media be adjusted to that of the video?
                               // 01-05 no: the user will be warned when he or she gives the files a different offset
                               //viewerManager.getMasterMediaPlayer().setOffset(masterMediaOffset);
                           }
                         */                            
                    }
                    
                    if (player.getVisualComponent() != null && !(curMD.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE))) {
                    	nrVisualPlayers++;
                    }
                    // only add layoutable players to the layout
                    //if (curMD.mimeType.equals(MediaDescriptor.MPG_MIME_TYPE)) {
                    // add no more than 4 visual players
                    if (nrVisualPlayers <= Constants.MAX_VISIBLE_PLAYERS || player.getVisualComponent() == null) {
                    	layoutManager.add(player);
                    	//System.out.println("Player Added... " + System.currentTimeMillis());
                    	// work around for Cocoa player, the constructor cannot (currently) throw
                    	// an exception if a file is not supported. Can only be done after it has been 
                    	// added to a frame
                    	if (player instanceof CocoaQTMediaPlayer) {
                    		try {
                    			Thread.sleep(100);
                    		} catch (InterruptedException ie) {	
                    		}
                    		if (!((CocoaQTMediaPlayer)player).isPlayerCreated()) {
                    			// delayed exception
                    			errors.append("The Cocoa QT Player can not handle the file.\n");
                    			player = PlayerFactory.createQTMediaPlayer(curMD);//only alternative on Mac
                    		}
                    	}
                    }

                    //}
                } catch (NoPlayerException npe) {
                    if (failedPlayers == null) {
                        failedPlayers = new ArrayList();
                    }
                    errors.append(npe.getMessage() + "\n");
                    failedPlayers.add(curMD);
                }
            }

            if (nrOfPlayers == 0) {
                viewerManager.setMasterMediaPlayer(new EmptyMediaPlayer(
                        transcription.getLatestTime()));
                layoutManager.add(viewerManager.getMasterMediaPlayer());
            }

            // Create a signal viewer
            if (signalSource != null) {
                SignalViewer newSignalViewer = viewerManager.createSignalViewer(signalSource);
                if(newSignalViewer != null){
                	newSignalViewer.setOffset(signalOffset);
                	newSignalViewer.preferencesChanged();
                
                	layoutManager.add(newSignalViewer);
                }
            }

            if (createSvgPlayer) {
                if (viewerManager.getMasterMediaPlayer() instanceof QTMediaPlayer) {
                    viewerManager.createQTSVGViewer();
                } else if (viewerManager.getMasterMediaPlayer() instanceof JMFGraphicMediaPlayer) {
                    // effect or renderer
                    // JMFSVGViewer jmfSVGViewer =
                    viewerManager.createJMFSVGViewer();
                }
            }

            layoutManager.doLayout();

            // inform the user of failures...
            if ((failedPlayers != null) && (failedPlayers.size() > 0)) {
                StringBuffer sb = new StringBuffer(
                        "No player could be created for:\n");

                for (int i = 0; i < failedPlayers.size(); i++) {
                    sb.append("- " +
                        ((MediaDescriptor) failedPlayers.get(i)).mediaURL +
                        "\n");
                }
                sb.append(errors.toString());

                JOptionPane.showMessageDialog(ELANCommandFactory.getRootFrame(transcription), sb.toString(),
                    ElanLocale.getString("Message.Warning"),
                    JOptionPane.WARNING_MESSAGE);
            }
        } catch (Exception rex) {
            rex.printStackTrace();
            JOptionPane.showMessageDialog(ELANCommandFactory.getRootFrame(transcription), 
            		"An error occurred while creating media players: " + rex.getMessage(),
                    ElanLocale.getString("Message.Warning"),
                    JOptionPane.WARNING_MESSAGE);
        }
    }

    /**
     * Creates a single media player for a media descriptor. For customized
     * player  creates, e.g. when showing (recreating) a player that has
     * previously been destroyed.
     *
     * @param transcription the transcription
     * @param curMD the medai descriptor
     *
     * @return a player or null
     */
    public static ElanMediaPlayer createMediaPlayer(
        TranscriptionImpl transcription, MediaDescriptor curMD) {
        if ((transcription == null) || (curMD == null)) {
            return null;
        }

        if (!MediaDescriptorUtil.checkLinkStatus(curMD)) {
            return null;
        }

        ElanMediaPlayer player = null;
        boolean createSvgPlayer = false;

        try {
            ViewerManager2 viewerManager = ELANCommandFactory.getViewerManager(transcription);

            // old stuff
            if (transcription.getSVGFile() != null /*||SVGPrefs.getUseSVG()*/) {
                // this makes all players QT players for now
                createSvgPlayer = true;
            }

            if (createSvgPlayer) {
                player = viewerManager.createMediaPlayer(curMD,
                        PlayerFactory.QT_MEDIA_FRAMEWORK);
            } else if (curMD.mimeType.equals(
                        MediaDescriptor.QUICKTIME_MIME_TYPE) && !SystemReporting.isMacOS()) {
	            player = viewerManager.createMediaPlayer(curMD,
	                        PlayerFactory.QT_MEDIA_FRAMEWORK);
            } else {
                player = viewerManager.createMediaPlayer(curMD);
            }
            /*
        	if (player instanceof CocoaQTMediaPlayer) {
        		try {
        			Thread.sleep(100);
        		} catch (InterruptedException ie) {	
        		}
        		if (!((CocoaQTMediaPlayer)player).isPlayerCreated()) {
        			player = PlayerFactory.createQTMediaPlayer(curMD);//only alternative on Mac, only on older systems, (10.4)
        		}
        	}
        	*/
        } catch (NoPlayerException npe) {
        }

        return player;
    }
}
