package mpi.eudico.client.annotator.recognizer.api;


import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.xml.sax.SAXException;

import mpi.eudico.client.annotator.recognizer.data.FileParam;
import mpi.eudico.client.annotator.recognizer.data.Param;
import mpi.eudico.client.annotator.recognizer.impl.VideoTestRecognizer;
import mpi.eudico.client.annotator.recognizer.load.RecognizerLoader;
import mpi.eudico.client.annotator.recognizer.load.RecognizerBundle;
import mpi.eudico.client.annotator.recognizer.load.RecognizerParser;
import mpi.eudico.client.annotator.recognizer.silence.SilenceRecognizer;
import mpi.eudico.client.annotator.util.ClientLogger;
import mpi.eudico.util.ExtClassLoader;

public class AvailabilityDetector {
	/** a template map for audio recognizers */
	private static Map<String, RecognizerBundle> audioRecognizerBundles = new HashMap<String, RecognizerBundle>(6);
	/** a template map for video recognizers */
	private static Map<String, RecognizerBundle> videoRecognizerBundles = new HashMap<String, RecognizerBundle>(6);
	private static RecognizerLoader recognizerLoader;
	
	static {
		init();
	}
	
	/**
	 * Private constructor
	 */
	private AvailabilityDetector() {
		super();
	}

	/**
	 * Check which recognizers are available for a list of media files
	 * The potential recognizers are supposed to return implement the method:
	 * 
	 * 		public boolean setMedia(ArrayList mediaFilePaths);
	 * 
	 * the boolean tells if the recognizer is able to handle the media.
	 * 
	 * @return an ArrayList with the available audio recognizers
	 */
	public static HashMap<String, Recognizer> getAudioRecognizers(ArrayList<String> filePaths) {		
		HashMap<String, Recognizer> audioRecs = new HashMap<String, Recognizer>(6);
		SilenceRecognizer sr = new SilenceRecognizer();
		audioRecs.put(sr.getName(), sr);
//		DemoRecognizer demoRec = new DemoRecognizer();
//		audioRecs.put(demoRec.getName(), demoRec);
		
		Iterator<String> keyIt = audioRecognizerBundles.keySet().iterator();
		String key;
		RecognizerBundle bundle;
		Recognizer rec;
		while (keyIt.hasNext()) {
			key = keyIt.next();
			bundle = audioRecognizerBundles.get(key);
			
			if (bundle.getRecExecutionType().equals("local")) {
				LocalRecognizer localRecognizer = new LocalRecognizer(bundle.getRecognizerClass());
				localRecognizer.setParamList(bundle.getParamList());
				localRecognizer.setName(bundle.getName());
				localRecognizer.setRecognizerType(Recognizer.AUDIO_TYPE);
				localRecognizer.setBaseDir(bundle.getBaseDir());
				if (videoRecognizerBundles.containsKey(key)) {
					localRecognizer.setRecognizerType(Recognizer.MIXED_TYPE);
				}
				audioRecs.put(key, localRecognizer);
			} else if (bundle.getRecExecutionType().equals("shared")) {
				SharedRecognizer sharedRecognizer = new SharedRecognizer(bundle.getRecognizerClass());
				sharedRecognizer.setParamList(bundle.getParamList());// returns a copy of the list
				sharedRecognizer.setName(bundle.getName());
				sharedRecognizer.setRecognizerType(Recognizer.VIDEO_TYPE);
				sharedRecognizer.setBaseDir(bundle.getBaseDir());
				if (videoRecognizerBundles.containsKey(key)) {
					sharedRecognizer.setRecognizerType(Recognizer.MIXED_TYPE);
				}
				audioRecs.put(key, sharedRecognizer);
			} else if (bundle.getJavaLibs() != null) {// assume "direct" ?
				
				if (recognizerLoader == null) {
					recognizerLoader = new RecognizerLoader(bundle.getJavaLibs(), bundle.getNativeLibs());
				} else {
					recognizerLoader.addLibs(bundle.getJavaLibs());
					recognizerLoader.addNativeLibs(bundle.getNativeLibs());
				}
				//RecognizerLoader loader = new RecognizerLoader(bundle.getJavaLibs(), bundle.getNativeLibs());
				
				try {
					//loader.loadNativeLibs();
					//rec = (Recognizer) Class.forName(bundle.getRecognizerClass(), true, loader).newInstance();
					rec = (Recognizer) Class.forName(bundle.getRecognizerClass(), true, recognizerLoader).newInstance();
					rec.setName(bundle.getName());
					audioRecs.put(key, rec);
				} catch (ClassNotFoundException cnfe) {
					ClientLogger.LOG.severe("Cannot load the recognizer class: " + bundle.getRecognizerClass() + " - Class not found");
				} catch (InstantiationException ie) {
					ClientLogger.LOG.severe("Cannot instantiate the recognizer class: " + bundle.getRecognizerClass());
				} catch (IllegalAccessException iae) {
					ClientLogger.LOG.severe("Cannot access the recognizer class: " + bundle.getRecognizerClass());
				} catch (Exception ex) {// any other exception
					ClientLogger.LOG.severe("Cannot load the recognizer: " + bundle.getRecognizerClass() + " - " + ex.getMessage());
				}
			} else {
				ClientLogger.LOG.severe("Cannot load the recognizer: no Java library has been found: " + bundle.getName());
			}
		}
		
		return audioRecs;
		

	}
	
	/**
	 * Maybe add parameters for needed capabilities like file formats
	 * 
	 * @return an ArrayList with the available video recognizers
	 */
	public static HashMap<String, Recognizer> getVideoRecognizers(ArrayList<String> filePaths) {
		HashMap<String, Recognizer> videoRecs = new HashMap<String, Recognizer>(6);
		//VideoTestRecognizer vtr = new VideoTestRecognizer();
		//videoRecs.put(vtr.getName(), vtr);
//		DemoRecognizer demoRec = new DemoRecognizer();
//		videoRecs.put(demoRec.getName(), demoRec);
		
		Iterator<String> keyIt = videoRecognizerBundles.keySet().iterator();
		String key;
		RecognizerBundle bundle;
		Recognizer rec;
		while (keyIt.hasNext()) {
			key = keyIt.next();
			bundle = videoRecognizerBundles.get(key);
			
			if (bundle.getRecExecutionType().equals("local")) {
				LocalRecognizer localRecognizer = new LocalRecognizer(bundle.getRecognizerClass());
				localRecognizer.setParamList(bundle.getParamList());// returns a copy of the list
				localRecognizer.setName(bundle.getName());
				localRecognizer.setRecognizerType(Recognizer.VIDEO_TYPE);
				localRecognizer.setBaseDir(bundle.getBaseDir());
				if (audioRecognizerBundles.containsKey(key)) {
					localRecognizer.setRecognizerType(Recognizer.MIXED_TYPE);
				}
				videoRecs.put(key, localRecognizer);
			} else if (bundle.getRecExecutionType().equals("shared")) {
				SharedRecognizer sharedRecognizer = new SharedRecognizer(bundle.getRecognizerClass());
				sharedRecognizer.setParamList(bundle.getParamList());// returns a copy of the list
				sharedRecognizer.setName(bundle.getName());
				sharedRecognizer.setRecognizerType(Recognizer.VIDEO_TYPE);
				sharedRecognizer.setBaseDir(bundle.getBaseDir());
				if (audioRecognizerBundles.containsKey(key)) {
					sharedRecognizer.setRecognizerType(Recognizer.MIXED_TYPE);
				}
				videoRecs.put(key, sharedRecognizer);
			} else if (bundle.getJavaLibs() != null) {// "direct"
			
				if (recognizerLoader == null) {
					recognizerLoader = new RecognizerLoader(bundle.getJavaLibs(), bundle.getNativeLibs());
				} else {
					recognizerLoader.addLibs(bundle.getJavaLibs());
					recognizerLoader.addNativeLibs(bundle.getNativeLibs());
				}
				//RecognizerLoader loader = new RecognizerLoader(bundle.getJavaLibs(), bundle.getNativeLibs());
				
				try {
					//loader.loadNativeLibs();
					//rec = (Recognizer) Class.forName(bundle.getRecognizerClass(), true, loader).newInstance();
					rec = (Recognizer) Class.forName(bundle.getRecognizerClass(), true, recognizerLoader).newInstance();
					rec.setName(bundle.getName());
					videoRecs.put(key, rec);
				} catch (ClassNotFoundException cnfe) {
					ClientLogger.LOG.severe("Cannot load the recognizer class: " + bundle.getRecognizerClass() + " - Class not found");
				} catch (InstantiationException ie) {
					ClientLogger.LOG.severe("Cannot instantiate the recognizer class: " + bundle.getRecognizerClass());
				} catch (IllegalAccessException iae) {
					ClientLogger.LOG.severe("Cannot access the recognizer class: " + bundle.getRecognizerClass());
				} catch (Exception ex) {// any other exception
					ClientLogger.LOG.severe("Cannot load the recognizer: " + bundle.getRecognizerClass() + " - " + ex.getMessage());
				}
			} else {
				ClientLogger.LOG.severe("Cannot load the recognizer: no Java library has been found ");
			}
		}
		
		return videoRecs;

	}
	
	/**
	 * Returns the parameter list of a recognizer.
	 * 
	 * @param recognizerName the name of the recognizer
	 * 
	 * @return a List containing the parameters
	 */
	public static List<Param> getParamList(String recognizerName) {
		if (recognizerName != null) {
			RecognizerBundle bundle = audioRecognizerBundles.get(recognizerName);
			List<Param> params = null;
			if (bundle != null) {
				params = bundle.getParamList();
			} else {
				bundle = videoRecognizerBundles.get(recognizerName);
				if (bundle != null) {
					params = bundle.getParamList();
				}
			}
			
			if (params != null) {
				List<Param> copyList = new ArrayList<Param>(params.size());
				for (Param p : params) {
					if (p == null) {
						continue;
					}
					try {
						copyList.add((Param) p.clone());
					} catch (CloneNotSupportedException cnse) {
						ClientLogger.LOG.warning("Cannot clone a parameter: " + p.id);
					}
				}
				
				return copyList;
			}
		}
		
		return null;
	}
	
	/**
	 * Detects all installed plug-ins and creates some maps for audio and video.
	 */
	private static void init() {
		// first add the homegrown Silence Recognizer or distribute it as an extension?
		
		File extFile = new File (ExtClassLoader.EXTENSIONS_DIR);
		
		if (!extFile.exists()) {
			ClientLogger.LOG.warning("The extension folder could not be found (" + ExtClassLoader.EXTENSIONS_DIR + ").");
			return;
		}
		if (!extFile.isDirectory()) {
			ClientLogger.LOG.warning("The extension \'folder\' is not a folder (" + ExtClassLoader.EXTENSIONS_DIR + ").");
			return;
		}
		if (!extFile.canRead()) {
			ClientLogger.LOG.warning("The extension folder is not accessible (" + ExtClassLoader.EXTENSIONS_DIR + ").");
			return;
		}
		File[] files = extFile.listFiles();
		for (File f : files) {
			if (f.isDirectory()) {
				detectRecognizer(f);
			} else {
				//load from jar or zip
				String name = f.getName().toLowerCase();
				if (name.endsWith("jar")) {
					detectRecognizerFromJar(f);
				} else if (name.endsWith("zip")) {
					detectRecognizerFromZip(f);
				}
			}
		}
	}
	
	/**
	 * Finds the recognizer.cmdi file in a jar and creates a recognizer bundle.
	 * 
	 * @param file the jar file
	 */
	private static void detectRecognizerFromJar(File file) {
		try {
			JarFile jFile = new JarFile(file);
			JarEntry je = jFile.getJarEntry("recognizer.cmdi");
			if (je == null) {
				ClientLogger.LOG.warning("No plug-in metadata file (recognizer.cmdi) found in " + file.getName());
				return;
			}
			try {
				URL jarUrl = file.toURL();
				URL[] libUrls = new URL[]{jarUrl};
				InputStream mdStream = jFile.getInputStream(je);
				createBundle(mdStream, libUrls, null, file.getParentFile());
			} catch (MalformedURLException mue) {
				ClientLogger.LOG.severe("Cannot create URL for file: " + file.getName());
			} catch (IOException ioe) {
				ClientLogger.LOG.warning("Cannot read the cmdi file from the jar file: " + file.getName());
			}
			
		} catch (IOException ioe) {
			ClientLogger.LOG.warning("Cannot read the jar file: " + file.getName());
		}
	}
	
	/**
	 * Finds the recognizer.cmdi file in a zip and creates a recognizer bundle.
	 * 
	 * @param file the zip file
	 */
	private static void detectRecognizerFromZip(File file) {
		try {
			ZipFile zFile = new ZipFile(file);
			ZipEntry ze = zFile.getEntry("recognizer.cmdi");
			if (ze == null) {
				ClientLogger.LOG.warning("No plug-in metadata file (recognizer.cmdi) found in " + file.getName());
				return;
			}
			try {
				URL zipUrl = file.toURL();
				URL[] libUrls = new URL[]{zipUrl};
				InputStream mdStream = zFile.getInputStream(ze);
				createBundle(mdStream, libUrls, null, file.getParentFile());
			} catch (MalformedURLException mue) {
				ClientLogger.LOG.severe("Cannot create URL for file: " + file.getName());
			} catch (IOException ioe) {
				ClientLogger.LOG.warning("Cannot read the cmdi file from the zip file: " + file.getName());
			}
		} catch (IOException ioe) {
			ClientLogger.LOG.warning("Cannot read the zip file: " + file.getName());
		}
	}
	
	/**
	 * Creates a bundle for the recognizer in this folder.
	 * 
	 * @param file the directory containing a recognizer plug-in
	 */
	private static void detectRecognizer (File file) {// checks have been performed before calling this
		File[] files = file.listFiles();
		File mdFile = null;
		
		for (File f : files) {
			if (f.getName().toLowerCase().equals("recognizer.cmdi")) {
				mdFile = f;
				break;
			}
		}
		
		if (mdFile != null) {
			List<URL> libs = new ArrayList<URL>();
			List<URL> natLibs = new ArrayList<URL>();
			addLibs(file, libs, natLibs);
			
			try {
				InputStream mdStream = new FileInputStream(mdFile);
				URL[] libUrls = null;
				if (libs.size() > 0) {
					libUrls = (URL[]) libs.toArray(new URL[]{});
				}
				URL[] natLibUrls = null;
				if (natLibs.size() > 0) {
					natLibUrls = (URL[]) natLibs.toArray(new URL[]{});
				}
				
				createBundle(mdStream, libUrls, natLibUrls, file);
			} catch (FileNotFoundException fnfe) {
				ClientLogger.LOG.severe("Cannot parse metadata file: " + fnfe.getMessage());
			}
		} else {
			ClientLogger.LOG.severe("No cmdi metadata file found in: " + file.getName());
		}
	}
	
	/**
	 * Creates a parser, a classloader, instantiates the recognizer, creates a bundle 
	 * and adds the bundle to the proper recognizer map(s).
	 * 
	 * @param mdStream the stream representing the "recognizer.cmdi" from a directory or from a jar
	 * @param libs the recognizer's Java libraries
	 * @param natLibs the recognizer's native libraries
	 * @param baseDir the directory the recognizer runs from
	 */
	private static void createBundle(InputStream mdStream, URL[] libs, URL[] natLibs, File baseDir) {
		boolean isDetector = false;
		String binaryName = null;
		RecognizerBundle bundle = null;
		RecognizerParser parser = null;
		
		try {
			parser = new RecognizerParser(mdStream);
			parser.parse();
			if (parser.recognizerType == null || 
					(!parser.recognizerType.equals("direct") && !parser.recognizerType.equals("local") && !parser.recognizerType.equals("shared"))) {
				ClientLogger.LOG.warning("Unsupported recognizer type, should be 'direct', 'local' or 'shared': " + parser.recognizerType);
				return;
			}
			if (!parser.curOsSupported) {
				ClientLogger.LOG.warning("Recognizer does not support this Operating System: " + parser.recognizerName);
				return;
			}
			if (parser.implementor != null) {
				isDetector = true;
				binaryName = parser.implementor;
			} else {
				ClientLogger.LOG.warning("The implementing class name has not been specified.");
				return;
			}
			
		} catch (SAXException sax) {
			ClientLogger.LOG.severe("Cannot parse metadata file: " + sax.getMessage());
		}
		
		if (isDetector) {
			boolean audio = false;
			boolean video = false;
			
			if (parser.paramList != null) {
				for (Param par : parser.paramList) {
					if (par instanceof FileParam) {
						if (((FileParam) par).ioType == FileParam.IN) {
							if (((FileParam) par).contentType == FileParam.AUDIO) {
								audio = true;
							} else if (((FileParam) par).contentType == FileParam.VIDEO) {
								video = true;
							}
						}
					}
				}
			} // else exception?
			
			if (parser.recognizerType.equals("direct")) {
				// create a classloader, bundle			
				RecognizerLoader loader = new RecognizerLoader(libs, natLibs);
	
				if (binaryName != null) {
					try {
						Class<?> c = loader.loadClass(binaryName);
						
						// if the above works, assume everything is all right
						bundle = new RecognizerBundle();
						bundle.setId(parser.recognizerName);
						bundle.setName(parser.description);// friendly name
						bundle.setParamList(parser.paramList);
						bundle.setRecognizerClass(binaryName);
						bundle.setRecExecutionType(parser.recognizerType);
						bundle.setJavaLibs(libs);
						bundle.setNativeLibs(natLibs);
						bundle.setBaseDir(baseDir);
							
						if (audio) {
							audioRecognizerBundles.put(bundle.getName(), bundle);
						}
						if (video) {
							videoRecognizerBundles.put(bundle.getName(), bundle);
						}
						
	//					Recognizer rec = (Recognizer) c.newInstance();
	//					bundle.setRecognizer(rec);
	//					if (bundle.getName() == null) {
	//						bundle.setName(rec.getName());
	//					} else {
	//						if (rec.getName() != null || !rec.getName().equals(bundle.getName())){
	//							bundle.setName(rec.getName());
	//						}
	//					}
	//					if (rec.getRecognizerType() == Recognizer.AUDIO_TYPE) {
	//						audioRecognizerBundles.put(rec.getName(), bundle);
	//					} else if (rec.getRecognizerType() == Recognizer.VIDEO_TYPE) {
	//						videoRecognizerBundles.put(rec.getName(), bundle);
	//					} else if (rec.getRecognizerType() == Recognizer.MIXED_TYPE) {
	//						audioRecognizerBundles.put(rec.getName(), bundle);
	//						videoRecognizerBundles.put(rec.getName(), bundle);
	//					}
					} catch (ClassNotFoundException cne) {
						ClientLogger.LOG.severe("Cannot load the recognizer class: " + binaryName + " - Class not found");
					} 
	//				catch (InstantiationException ie) {
	//					ClientLogger.LOG.severe("Cannot instantiate the recognizer class: " + binaryName);
	//				} catch (IllegalAccessException iae) {
	//					ClientLogger.LOG.severe("Cannot access the recognizer class: " + binaryName);
	//				}
				}// else throw exception?
				else {
					ClientLogger.LOG.warning("Cannot load the recognizer class: Class not found");
				}
			} else if (parser.recognizerType.equals("local") || parser.recognizerType.equals("shared")) {
				
				bundle = new RecognizerBundle();
				bundle.setId(parser.recognizerName);
				bundle.setName(parser.description);// friendly name
				bundle.setParamList(parser.paramList);
				bundle.setRecognizerClass(binaryName);// reuse the class name for the run command
				bundle.setRecExecutionType(parser.recognizerType);
				bundle.setBaseDir(baseDir);
				
				if (audio) {
					audioRecognizerBundles.put(bundle.getName(), bundle);
				}
				if (video) {
					videoRecognizerBundles.put(bundle.getName(), bundle);
				}
			}
		}
	}
	
	/**
	 * Recursively adds jar files and native libraries to the lists.
	 * 
	 * @param file the folder to search for libraries
	 * @param libs the list of Java libraries
	 * @param natLibs the list of native libraries
	 */
	private static void addLibs (File file, List<URL> libs, List<URL> natLibs) {
		File[] files = file.listFiles();
		
		for (File f : files) {
			try {
				if (f.isDirectory()) {
					addLibs(f, libs, natLibs);
				} else {
					String name = f.getName().toLowerCase();
					if (name.endsWith("jar") || name.endsWith("zip")) {
						try {
							libs.add(f.toURL());
						} catch (MalformedURLException mue) {
							ClientLogger.LOG.severe("Cannot create URL for file: " + f.getName());
						}
					} else if (name.endsWith("dll") || name.endsWith("so") || name.endsWith("jnilib")) {
						try {
							natLibs.add(f.toURL());
						} catch (MalformedURLException mue) {
							ClientLogger.LOG.severe("Cannot create URL for file: " + f.getName());
						}
					}
				}
			} catch (SecurityException se) {
				ClientLogger.LOG.warning("Cannot read file: " + f.getName());
			}
		}
	}
}

