package mpi.eudico.client.annotator.recognizer.gui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.border.TitledBorder;
import org.xml.sax.SAXException;

import mpi.eudico.client.annotator.commands.Command;
import mpi.eudico.client.annotator.commands.ELANCommandFactory;
import mpi.eudico.client.annotator.commands.SegmentsToTiersCommand;
import mpi.eudico.client.annotator.gui.FileChooser;
import mpi.eudico.client.annotator.recognizer.api.AvailabilityDetector;
import mpi.eudico.client.annotator.recognizer.api.ParamPreferences;
import mpi.eudico.client.annotator.recognizer.api.Recognizer;
import mpi.eudico.client.annotator.recognizer.api.RecognizerHost;
import mpi.eudico.client.annotator.recognizer.api.SharedRecognizer;
import mpi.eudico.client.annotator.recognizer.data.FileParam;
import mpi.eudico.client.annotator.recognizer.data.Param;
import mpi.eudico.client.annotator.recognizer.data.RSelection;
import mpi.eudico.client.annotator.recognizer.data.Segment;
import mpi.eudico.client.annotator.recognizer.data.Segmentation;
import mpi.eudico.client.annotator.recognizer.io.CsvTierIO;
import mpi.eudico.client.annotator.recognizer.io.CsvTimeSeriesIO;
import mpi.eudico.client.annotator.recognizer.io.ParamIO;
import mpi.eudico.client.annotator.recognizer.io.RecTierWriter;
import mpi.eudico.client.annotator.recognizer.io.XmlTierIO;
import mpi.eudico.client.annotator.recognizer.io.XmlTimeSeriesReader;
import mpi.eudico.client.annotator.util.AnnotationDataRecord;
import mpi.eudico.client.annotator.util.ClientLogger;
import mpi.eudico.client.annotator.util.FileExtension;
import mpi.eudico.client.annotator.util.SystemReporting;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;
import mpi.eudico.client.annotator.ViewerManager2;
import mpi.eudico.client.util.CheckBoxBListCellRenderer;
import mpi.eudico.client.util.SelectEnableObject;
import mpi.eudico.client.util.SelectableObject;
import mpi.eudico.client.annotator.ElanLocaleListener;


public abstract class AbstractRecognizerPanel extends JComponent implements ActionListener, Runnable, 
RecognizerHost, ElanLocaleListener, ItemListener {
	protected ViewerManager2 viewerManager;
	// param panel provided by recognizer or created based on configuration file
	protected JPanel controlPanel;
	protected JPanel progressPanel;
	protected JPanel recognizerAndFilesPanel;
	protected JPanel paramPanel;
	protected JScrollPane jsp;
	protected JButton detachButton;
	protected ArrayList<String> mediaFilePaths;
	protected ArrayList<String> supportedFiles;
	protected SelectionPanel selectionPanel;
	protected SegmentationPanel segmentationPanel;
	protected JLabel recognizerLabel;
	protected JComboBox recognizerList;
	protected JPanel filesSelector;
	protected JLabel filesLabel;
	protected JComboBox filesList;
	protected HashMap<String, Recognizer> recognizers;
	protected Recognizer currentRecognizer;
	protected HashMap<String, Segmentation> segmentations;
	protected JProgressBar progressBar;
	protected JButton startStopButton;
	protected JButton reportButton;
	protected JPanel paramButtonPanel;
	protected JButton saveParamsButton;
	protected JButton loadParamsButton;
	protected boolean isRunning;
	protected int mode = Recognizer.AUDIO_TYPE;
	protected boolean notMono;
	protected boolean reduceFilePrompts = true;
	protected long lastStartTime = 0L;

	/**
	 *  Initializes data structures and user interface components.
	 */
	public AbstractRecognizerPanel(ViewerManager2 viewerManager, ArrayList<String> mediaFilePaths) {
		super();

		this.viewerManager = viewerManager;
		this.mediaFilePaths = mediaFilePaths;
		if (mediaFilePaths != null) {
			supportedFiles = new ArrayList<String>(mediaFilePaths.size());
		} else {
			mediaFilePaths = new ArrayList<String>(0);
			supportedFiles = new ArrayList<String>(0);
		}
		
		segmentations = new HashMap<String, Segmentation>();
		
		initComponents();
		initRecognizers();
	}
	
	protected void initComponents() {
		setLayout(new GridBagLayout());

		recognizerLabel = new JLabel();
		recognizerList = new JComboBox();
		recognizerAndFilesPanel = new JPanel(new GridBagLayout());
		filesLabel = new JLabel();
		filesList = new JComboBox();
		// start with all selected and enabled
		for (int i = 0; i < mediaFilePaths.size(); i++) {
			filesList.addItem(new SelectEnableObject(
					fileNameFromPath(mediaFilePaths.get(i)), true, true));//only filename
		}
		if(filesList.getItemCount() > 0){				
			filesList.setRenderer(new CheckBoxBListCellRenderer());
		} else
			filesList.setRenderer(new JComboBox().getRenderer());
		
		filesList.addActionListener(this);
		
		GridBagConstraints gbc = new GridBagConstraints();
		gbc.insets = new Insets(0, 2, 0, 0);
		gbc.anchor = GridBagConstraints.WEST;
		recognizerAndFilesPanel.add(recognizerLabel, gbc);
		gbc.gridx = 1;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.weightx = 1.0;
		recognizerAndFilesPanel.add(recognizerList, gbc);
		gbc.gridx = 2;
		gbc.fill = GridBagConstraints.NONE;
		gbc.weightx = 0.0;
		recognizerAndFilesPanel.add(filesLabel, gbc);
		gbc.gridx = 3;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.weightx = 1.0;
		recognizerAndFilesPanel.add(filesList, gbc);
		
		
		// add more file(s) label	
		selectionPanel = new SelectionPanel(mode, notMono, viewerManager);
		segmentationPanel = new SegmentationPanel(this, viewerManager);
		segmentationPanel.setEnabled(false);

		// start/stop and progress information panel
		progressPanel = new JPanel();
		progressPanel.setLayout(new BoxLayout(progressPanel, BoxLayout.X_AXIS));
		progressBar = new JProgressBar(0, 100);
		progressBar.setStringPainted(true);
		progressPanel.add(Box.createHorizontalStrut(10));
		progressPanel.add(progressBar);
		progressPanel.add(Box.createHorizontalStrut(10));
		startStopButton = new JButton(ElanLocale.getString("Recognizer.RecognizerPanel.Start"));
		startStopButton.addActionListener(this);
		progressPanel.add(startStopButton);	
		reportButton = new JButton();
		reportButton.addActionListener(this);
		reportButton.setEnabled(false);
		progressPanel.add(Box.createHorizontalStrut(15));
		progressPanel.add(reportButton);
		

		paramPanel = new JPanel(new GridBagLayout());
		paramButtonPanel = new JPanel(new GridBagLayout());
		
		saveParamsButton = new JButton();
		saveParamsButton.addActionListener(this);
		try {
			ImageIcon icon = new ImageIcon(this.getClass().getResource(
					"/toolbarButtonGraphics/general/SaveAs16.gif"));
			saveParamsButton.setIcon(icon);
		} catch (Exception ex) {
			// catch any image loading exception
			saveParamsButton.setText("S");
		}
		loadParamsButton = new JButton();
		loadParamsButton.addActionListener(this);
		try {
			ImageIcon openIcon = new ImageIcon(this.getClass().getResource("/toolbarButtonGraphics/general/Open16.gif"));
			loadParamsButton.setIcon(openIcon);
		} catch (Exception ex) {
			// catch any image loading exception
			loadParamsButton.setText("O");
		}
		gbc = new GridBagConstraints();
		gbc.anchor = GridBagConstraints.NORTHEAST;		
		paramButtonPanel.add(loadParamsButton, gbc);
		
		gbc.gridx = 1;
		gbc.insets = new Insets(0, 2, 0, 2);		
		paramButtonPanel.add(saveParamsButton, gbc);
		
		detachButton = new JButton();
		detachButton.addActionListener(this);
		try {
			ImageIcon icon = new ImageIcon(this.getClass().getResource(
					"/mpi/eudico/client/annotator/resources/Detach.gif"));
			detachButton.setIcon(icon);
		} catch (Exception ex) {
			// catch any image loading exception
			detachButton.setText("D");
		}
		gbc.gridx = 2;
		gbc.insets = new Insets(0, 10, 0, 2);
		paramButtonPanel.add(detachButton, gbc);

		// add components
		gbc = new GridBagConstraints();
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.anchor = GridBagConstraints.NORTHWEST;
		gbc.weightx = 1.0;
		gbc.gridwidth = 2;
		add(recognizerAndFilesPanel, gbc);
		
		gbc.gridx = 0;
		gbc.gridy = 1;
		gbc.fill = GridBagConstraints.VERTICAL;
		gbc.gridwidth = 1;
		gbc.weightx = 0.0;
		gbc.weighty = 1.0;
		add(selectionPanel, gbc);

		gbc.gridx = 0;
		gbc.gridy = 2;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.weightx = 0.0;
		gbc.weighty = 0.0;
		add(segmentationPanel, gbc);

		gbc.gridx = 1;
		gbc.gridy = 1;
		gbc.fill = GridBagConstraints.BOTH;
		gbc.gridwidth = 1;
		gbc.weightx = 1.0;
		gbc.weighty = 1.0;
		add(paramPanel, gbc);
		
		gbc.gridx = 1;
		gbc.gridy = 2;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.gridwidth = 1;
		gbc.weightx = 0.0;
		gbc.weighty = 0.0;
		add(progressPanel, gbc);
	}
	
	/**
	 * Initializes recognizer and related ui elements, only once.
	 */
	protected void initRecognizers() {
		recognizers = getAvailableRecognizers();
		if (currentRecognizer != null) {
			paramPanel.removeAll();
			currentRecognizer = null;
		}

		recognizerList.removeItemListener(this);
		recognizerList.removeAllItems();
		Set<String> names = recognizers.keySet();
		Iterator<String> iter = names.iterator();
		while (iter.hasNext()) {
			recognizerList.addItem(iter.next());
		}
		
		if (recognizerList.getItemCount() == 0) {
			recognizerList.addItem(ElanLocale.getString("Recognizer.RecognizerPanel.No.Recognizers"));
			//paramPanel.add(new JLabel(ElanLocale.getString("Recognizer.RecognizerPanel.No.Parameters")), BorderLayout.CENTER);
			jsp = new JScrollPane(getParamPanel(null));
			jsp.setBackground(getBackground());
			jsp.getViewport().setBackground(getBackground());
			jsp.getVerticalScrollBar().setUnitIncrement(20);
			//paramPanel.add(new JScrollPane(getParamPanel(null)));
			GridBagConstraints gbc = new GridBagConstraints();
			gbc.anchor = GridBagConstraints.NORTHWEST;
			gbc.fill = GridBagConstraints.BOTH;
			gbc.gridy = 0;
			gbc.weightx = 1.0;
			gbc.weighty = 1.0;
			paramPanel.add(jsp, gbc);
			startStopButton.setEnabled(false);
			recognizerList.setEnabled(false);
			selectionPanel.setEnabled(false);
			segmentationPanel.setEnabled(false);
			paramButtonPanel.setVisible(false);
//			loadParamsButton.setEnabled(false);
//			saveParamsButton.setEnabled(false);
		} else {
			recognizerList.setSelectedIndex(0);
			setRecognizer((String) recognizerList.getSelectedItem());
			recognizerList.setEnabled(true);
			selectionPanel.setEnabled(true);
			//segmentationPanel.setEnabled(true);;
		}
		recognizerList.addItemListener(this);
	}
	
	/**
	 * To be implemented by subclasses. Depends on the type of recognizers, audio or video.
	 * 
	 * @return the currently available recognizers
	 */
	protected abstract HashMap<String, Recognizer> getAvailableRecognizers();
	
	/**
	 * Sets the recognizer, gets the parameter panel, updates the files list.
	 *  
	 * @param name the name of the recognizer
	 */
	protected void setRecognizer(String name) {
		if (currentRecognizer != null) {
			// store preferences
			if (controlPanel instanceof ParamPreferences) {
				Map<String, Object> prefs = ((ParamPreferences) controlPanel).getParamPreferences();
				if (prefs != null) {
					Preferences.set(currentRecognizer.getName(), 
						((ParamPreferences) controlPanel).getParamPreferences(), null);
				}
			}
			currentRecognizer.dispose();
			paramPanel.removeAll();
			segmentations.clear();
			segmentationPanel.setEnabled(false);
			reportButton.setEnabled(false);
		}
		currentRecognizer = (Recognizer) recognizers.get(name);
		currentRecognizer.setRecognizerHost(this);
		
		controlPanel = currentRecognizer.getControlPanel();
		if (controlPanel != null) {
			jsp = new JScrollPane(controlPanel);
			//jsp.setBorder(null);
			jsp.getVerticalScrollBar().setUnitIncrement(20);
			GridBagConstraints gbc = new GridBagConstraints();
			gbc.anchor = GridBagConstraints.NORTHWEST;
			gbc.fill = GridBagConstraints.BOTH;
			gbc.gridy = 0;
			gbc.weightx = 1.0;
			gbc.weighty = 1.0;
			paramPanel.add(jsp, gbc);
			gbc.gridy = 1;
			gbc.anchor = GridBagConstraints.NORTHEAST;
			gbc.fill = GridBagConstraints.NONE;
			gbc.weightx = 0.0;
			gbc.weighty = 0.0;
			paramPanel.add(paramButtonPanel, gbc);
		} else {
			// check if there are parameters, create factory panel
			controlPanel = getParamPanel(currentRecognizer);
			if (controlPanel != null) {
				jsp = new JScrollPane(controlPanel);
				//jsp.setBorder(null);
				jsp.getVerticalScrollBar().setUnitIncrement(20);
				//paramPanel.add(jsp, BorderLayout.CENTER);
				GridBagConstraints gbc = new GridBagConstraints();
				gbc.anchor = GridBagConstraints.NORTHWEST;
				gbc.fill = GridBagConstraints.BOTH;
				gbc.gridy = 0;
				gbc.weightx = 1.0;
				gbc.weighty = 1.0;
				paramPanel.add(jsp, gbc);
				gbc.gridy = 1;
				gbc.anchor = GridBagConstraints.NORTHEAST;
				gbc.fill = GridBagConstraints.NONE;
				gbc.weightx = 0.0;
				gbc.weighty = 0.0;
				paramPanel.add(paramButtonPanel, gbc);
			} else {
				jsp = null;
				// label, no configurable params
				paramPanel.add(new JLabel(ElanLocale.getString("Recognizer.RecognizerPanel.No.Parameters")), new GridBagConstraints());
				paramPanel.repaint();
			}
		}
		
		if (controlPanel instanceof ParamPreferences) {
			Object prefs = Preferences.get(currentRecognizer.getName(), null);
			if (prefs instanceof Map) {
				((ParamPreferences) controlPanel).setParamPreferences((HashMap<String, Object>) prefs);
			}
			loadParamsButton.setEnabled(true);
			saveParamsButton.setEnabled(true);
		} else {
			loadParamsButton.setEnabled(false);
			saveParamsButton.setEnabled(false);
		}
		
		startStopButton.setEnabled(true);
		progressBar.setString("");
		progressBar.setValue(0);
		
		// check the current media files
		supportedFiles.clear();
		String fp;
		SelectEnableObject seo;
		// enable/disable files in list
		for (int i = 0; i < mediaFilePaths.size(); i++) {
			fp = mediaFilePaths.get(i);
			seo = (SelectEnableObject) filesList.getItemAt(i);
			if (currentRecognizer.canHandleMedia(fp)) {
				supportedFiles.add(fp);
				// update the visible list of media files, use same index, order is the same				
				seo.setEnabled(true);
				seo.setSelected(true);
			} else {
				seo.setEnabled(false);
				seo.setSelected(false);
			}
		}
		updateSelectionPanel();
		// enable/disable start button
		startStopButton.setEnabled((supportedFiles.size() > 0));

		if (currentRecognizer.getExamplesSupport() == Recognizer.EXAMPLE_SEGMENTS_NOT_SUPPORTED) {
			selectionPanel.setEnabled(false);
			selectionPanel.clearSelections();
		} else {
			selectionPanel.setEnabled(true);
		}
		
		validate();
	}
	
	/**
	 * Returns a factory created user interface for setting parameters for the recognizer.
	 * 
	 * @param recognizer the recognizer
	 * 
	 * @return a parameter panel
	 */
	protected JPanel getParamPanel(Recognizer recognizer) {
		if (recognizer != null) {
			List<Param> params = AvailabilityDetector.getParamList(recognizer.getName());
			if (params != null) {
				ParamPanelContainer ppc = new ParamPanelContainer(recognizer.getName(), params);
				ppc.validate();
				return ppc;
			}
		}
		
		return null;
	}
	
	/**
	 * Called after a change in the set of linked media files.
	 */
	protected void updateFileList(){
		
		filesList.removeAllItems();

		for (int i = 0; i < mediaFilePaths.size(); i++) {
			filesList.addItem(new SelectEnableObject(
					fileNameFromPath(mediaFilePaths.get(i)), true, true));//only filename
		}
		if(filesList.getItemCount() < 1){	
			JComboBox temp = new JComboBox();
			filesList.setRenderer(temp.getRenderer());
		} else if( !(filesList.getRenderer() instanceof CheckBoxBListCellRenderer) ){
			filesList.setRenderer(new CheckBoxBListCellRenderer());					
		}
	}
	
	/**
	 * Extracts the file name from a path.
	 * 
	 * @param path the file path
	 * @return the file name
	 */
	String fileNameFromPath(String path) {
		if (path == null) {
			return "Unknown";
		}
		// assume all paths have forward slashes
		int index = path.lastIndexOf('/');
		if (index > -1 && index < path.length() - 1) {
			return path.substring(index + 1);
		}
		
		return path;
	}
	
	
	/**
	 * Returns a list of currently selected media files.
	 * 
	 * @return a list of currently selected media files
	 */
	List<String> getCurrentFileSelection() {
		List<String> curFiles = new ArrayList<String>(4);
		
		Object iter;
		SelectEnableObject seo;
		
		for (int i = 0; i < filesList.getModel().getSize(); i++) {
			iter = filesList.getModel().getElementAt(i);
			
			if (iter instanceof SelectEnableObject) {
				seo = (SelectEnableObject) iter;
				if (seo.isSelected()) {
					if (SystemReporting.isWindows()) {
						String path = mediaFilePaths.get(i);
						if (path.startsWith("///")) {// is this necessary?
							path = path.substring(3);
						}
						path = path.replace('/', '\\');
						curFiles.add(path);
					} else {
						curFiles.add(mediaFilePaths.get(i));// get from the media file path array
					}
				}
			}
		}
		
		return curFiles;
	}
	
	/**
	 * Informs the selection panel of changes in the media file selection.  
	 */
	private void updateSelectionPanel() {
		if (mode == Recognizer.VIDEO_TYPE) {
			selectionPanel.setMediaFilePaths(getCurrentFileSelection());
		}
	}
	
	/**
	 * Handling of combobox selection events.
	 * @param e the event
	 */
	public void itemStateChanged(ItemEvent e) {
		if (e.getSource() == recognizerList && e.getStateChange() == ItemEvent.SELECTED) {
			if (currentRecognizer != null && isBusy()) {
				// tell the user that the current recognizer is still running
				JOptionPane.showMessageDialog(this,
					    currentRecognizer.getName() + ": " + ElanLocale.getString("Recognizer.RecognizerPanel.Warning.Busy"),
					    currentRecognizer.getName() + " " + ElanLocale.getString("Recognizer.RecognizerPanel.Warning.Busy2"),
					    JOptionPane.PLAIN_MESSAGE);
				// restore the current recognizers name in the combo box
				recognizerList.setSelectedItem(currentRecognizer.getName());
				return;
			}
			// reset the current segmentations
			//segmentations = new HashMap<String, Segmentation>();
			segmentations.clear();
			// remove current recognizer GUI
			paramPanel.removeAll();
			// set the new current recognizer
			setRecognizer((String) recognizerList.getSelectedItem());
		}
		
	}
	
	public void actionPerformed(ActionEvent e) {
		Object source = e.getSource();
		
		if (source.equals(startStopButton)) {
			if (currentRecognizer == null) {
				return;
			}
			if (isRunning) {
				startStopButton.setText(ElanLocale.getString(
				"Recognizer.RecognizerPanel.Start"));
				currentRecognizer.stop();
				isRunning = false;
				if (progressBar.isIndeterminate()) {
					progressBar.setIndeterminate(false);
				}
				progressBar.setString(ElanLocale.getString("Recognizer.RecognizerPanel.Canceled"));
				progressBar.setValue(0);
				reportButton.setEnabled(true);
			} else {
				startRecognizer();
			}
		} 
		// an item listener does not allow selection (and therefore changing) of the selected (==visible) item 
		else if (source == filesList) {			
			Object item = filesList.getSelectedItem();
			SelectEnableObject seo;
			if (item instanceof SelectEnableObject) {
				seo =  (SelectEnableObject) item;
				if (seo.isEnabled()) {
					boolean old = ((SelectableObject) item).isSelected();
					((SelectableObject) item).setSelected(!old);					
				}
			} 
			
			Object iter;
			int numsel = 0;
			for (int i = 0; i < filesList.getModel().getSize(); i++) {
				iter = filesList.getModel().getElementAt(i);
				if (iter instanceof SelectEnableObject) {
					seo = (SelectEnableObject) iter;
					if (seo.isSelected()) {
						filesList.setSelectedIndex(i);
						numsel++;
						break;
					}
				}
			}
			
			startStopButton.setEnabled((numsel > 0));
			updateSelectionPanel();			
		} else if (source == reportButton) {
			if (currentRecognizer != null) {
				showReport(currentRecognizer.getReport());
			}
		} else if (source == saveParamsButton) {
			saveParameterFile();
		} else if (source == loadParamsButton) {
			loadParameterFile();
		} else if (source == detachButton) {
			detachParamPanel();
		}
	}
	
	protected void startRecognizer() {
		if (isRunning) {
			return;
		}
		// retrieve the selected media files and pass them to the recognizer
		// return if there are no media files selected
		List<String> selMediaFiles = getCurrentFileSelection();
		
		if (currentRecognizer.getExamplesSupport() == Recognizer.EXAMPLE_SEGMENTS_REQUIRED ||
				currentRecognizer.getExamplesSupport() == Recognizer.EXAMPLE_SEGMENTS_OPTIONAL) {
			boolean selectionsExist = selectionPanel.hasSelections();
			List<FileParamPanel> ffps = new ArrayList<FileParamPanel>(4);
			
			// check if there are "tier" file input parameters
			if (controlPanel instanceof ParamPanelContainer) {
				ParamPanelContainer ppc = (ParamPanelContainer) controlPanel;
				int numPanels = ppc.getNumPanels();
				AbstractParamPanel app;
				FileParamPanel fpp;
				for (int i = 0; i < numPanels; i++) {
					app = ppc.getParamPanel(i);
					if (app instanceof FileParamPanel) {
						fpp = (FileParamPanel) app;
						if (fpp.isInputType() && 
								(fpp.getContentType() == FileParam.TIER || fpp.getContentType() == FileParam.CSV_TIER) ) {
							ffps.add(fpp);						
						}
					}
				}				
			}
			// 
			int numNotFilled = 0;
			boolean[] filled = new boolean[ffps.size()];
			int count = 0;
			for(FileParamPanel panel : ffps) {
				Object filePath = panel.getParamValue();
				if (filePath instanceof String) {
					if (((String) filePath).length() == 0) {
						numNotFilled++;
						filled[count] = false;
					} else {// else check if it is a valid path??
						filled[count] = true;
					}
				} else {
					numNotFilled++;
					filled[count] = false;
				}
				count++;
			}
			
			if (selectionsExist && ffps.isEmpty()) {
				// assume the the recognizer is a Java plugin and has direct access to the selections
			} else if (selectionsExist) {			
				if (numNotFilled == 1) {
					// ask for file name
					FileParamPanel targetFfp = null;
					for (int i = 0; i < filled.length; i++) {
						if (!filled[i]) {
							targetFfp = ffps.get(i);
							break;
						}
					}
					if (targetFfp != null) {
						int contentType = targetFfp.getContentType();
						String filepath = null;
						if (reduceFilePrompts) {
							filepath = autoCreateOutputFile(mediaFilePaths, targetFfp.getParamName(), contentType);
						} else {						
							filepath = promptForTierFile(contentType, (String) targetFfp.getParamValue());
						}
						if (filepath == null) {
							return;
						}
						File tf = new File(filepath);
						try { 
							if (tf.exists() && !reduceFilePrompts) {
		                        int answer = JOptionPane.showConfirmDialog(this,
		                                ElanLocale.getString("Message.Overwrite"),
		                                ElanLocale.getString("SaveDialog.Message.Title"),
		                                JOptionPane.YES_NO_OPTION,
		                                JOptionPane.WARNING_MESSAGE);
	
		                        if (answer != JOptionPane.YES_OPTION) {
		                            return;
		                        }
							}
						} catch (Exception ex) {// any exception
							return;
						}
						targetFfp.setParamValue(filepath);
						try {
							writeTierFile(filepath, contentType);
						} catch (IOException ioe) {
							// show message
							JOptionPane.showMessageDialog(this, ElanLocale.getString(
									"Recognizer.RecognizerPanel.Warning.SaveFailed")  + ioe.getMessage(), 
									ElanLocale.getString("Message.Warning"), JOptionPane.ERROR_MESSAGE);
							return;
						}
					}
				} else {
					// ask for file name and for which parameter to store the tier file
					List<String> paramList = new ArrayList<String>(ffps.size());
					FileParamPanel targetFfp = null;
					// first add the empty parameters
					for (int i = 0; i < filled.length; i++) {
						if (!filled[i]) {
							targetFfp = ffps.get(i);
							paramList.add(targetFfp.paramName);
						}
					}
					// then add the ones that are filled
					for (int i = 0; i < filled.length; i++) {
						if (filled[i]) {
							targetFfp = ffps.get(i);
							paramList.add(targetFfp.paramName);
						}
					}
					targetFfp = null;
					String[] options = paramList.toArray(new String[]{});
					Object selected  = JOptionPane.showInputDialog(this, 
							ElanLocale.getString("Recognizer.SelectionsPanel.SaveSelections")  
							+ "\n" + ElanLocale.getString("Recognizer.SelectionsPanel.SaveSelections2"),
							ElanLocale.getString("Recognizer.SelectionsPanel.Title"), 
							JOptionPane.PLAIN_MESSAGE, null, options, options[0]);
					
					if (selected != null) {
						
						for (FileParamPanel fp : ffps) {
							if (fp.paramName.equals(selected)) {
								targetFfp = fp;
								break;
							}
						}
						// prompt for file
						int contentType = targetFfp.getContentType();
						String filepath = null;
						if (reduceFilePrompts) {
							filepath = autoCreateOutputFile(mediaFilePaths, targetFfp.getParamName(), contentType);
						} else {						
							filepath = promptForTierFile(contentType, (String) targetFfp.getParamValue());
						}

						if (filepath == null) {
							return;
						}
						// check whether the file already exists
						File tf = new File(filepath);
						try { 
							if (tf.exists() && !reduceFilePrompts) {
		                        int answer = JOptionPane.showConfirmDialog(this,
		                                ElanLocale.getString("Message.Overwrite"),
		                                ElanLocale.getString("SaveDialog.Message.Title"),
		                                JOptionPane.YES_NO_OPTION, 
		                                JOptionPane.WARNING_MESSAGE);
	
		                        if (answer != JOptionPane.YES_OPTION) {
		                            return;
		                        }
							}
						} catch (Exception ex) {// any exception
							return;
						}
						
						targetFfp.setParamValue(filepath);
						try {
							writeTierFile(filepath, contentType);
						} catch (IOException ioe) {
							// show message
							JOptionPane.showMessageDialog(this, ElanLocale.getString(
									"Recognizer.RecognizerPanel.Warning.SaveFailed")  + ioe.getMessage(), 
									ElanLocale.getString("Message.Warning"), JOptionPane.ERROR_MESSAGE);
							return;
						}
					} // no selection, continue without saving the selections?
				}
			} else if (!selectionsExist) {
				if (currentRecognizer.getExamplesSupport() == Recognizer.EXAMPLE_SEGMENTS_REQUIRED && 
						((!ffps.isEmpty() && numNotFilled > 0))) {
					// warn and return
					JOptionPane.showMessageDialog(this,
						    ElanLocale.getString("Recognizer.RecognizerPanel.Warning.Selection3"),
						    currentRecognizer.getName(),
						    JOptionPane.PLAIN_MESSAGE);
					return;
				} else if (currentRecognizer.getExamplesSupport() == Recognizer.EXAMPLE_SEGMENTS_REQUIRED && 
						ffps.isEmpty()) {
					JOptionPane.showMessageDialog(this,
						    ElanLocale.getString("Recognizer.RecognizerPanel.Warning.Selection"),
						    currentRecognizer.getName() + " " + 
						    ElanLocale.getString("Recognizer.RecognizerPanel.Warning.Selection2"),
						    JOptionPane.PLAIN_MESSAGE);
					return;
				}
			}
		}
		
		// check if the param panel is filled adequately
		if (controlPanel instanceof ParamPanelContainer) {
			ParamPanelContainer ppc = (ParamPanelContainer) controlPanel;
			int numPanels = ppc.getNumPanels();
			AbstractParamPanel app;
			for (int i = 0; i < numPanels; i++) {
				app = ppc.getParamPanel(i);
				if (app instanceof FileParamPanel) {
					if ( !((FileParamPanel) app).isInputType() && !((FileParamPanel) app).isOptional()) {
						String val = (String) ((FileParamPanel) app).getParamValue();
						if (val == null || val.length() == 0) {
							if (reduceFilePrompts) {
								String filePath = autoCreateOutputFile(mediaFilePaths, 
										((FileParamPanel) app).getParamName(), ((FileParamPanel) app).getContentType());
								if (filePath != null) {
									((FileParamPanel) app).setParamValue(filePath);
									val = filePath;
								}
							} 
							// check again
							if (val == null || val.length() == 0)  {
								// prompt and return
								JOptionPane.showMessageDialog(this,									    
									    ElanLocale.getString("Recognizer.RecognizerPanel.Warning.EmptyParam") + " \"" +
									    app.description + "\" " +
									    ElanLocale.getString("Recognizer.RecognizerPanel.Warning.EmptyParam2"),
									    ElanLocale.getString("Recognizer.RecognizerPanel.Warning.EmptyParam"),
									    JOptionPane.ERROR_MESSAGE);
								return;
							}
						} else {
							// check if file exists and set access
							createOutputFile(val);
						}
					}
				}
			}
		}
		progressBar.setValue(0);
		progressBar.setString("");
		// everything ok, pass the current values to the recognizer
		if (controlPanel instanceof ParamPanelContainer) {
			ParamPanelContainer ppc = (ParamPanelContainer) controlPanel;
			int numPanels = ppc.getNumPanels();
			AbstractParamPanel app;
			Object value;
			for (int i = 0; i < numPanels; i++) {
				app = ppc.getParamPanel(i);
				value = app.getParamValue();
				if (value instanceof Float) {
					currentRecognizer.setParameterValue(app.paramName, ((Float) value).floatValue());
				} if (value instanceof Double) {
					currentRecognizer.setParameterValue(app.paramName, ((Double) value).floatValue());
				} else if (value instanceof String) {
					currentRecognizer.setParameterValue(app.paramName, (String) value);
				}
			}
		}
		// store preferences
		if (controlPanel instanceof ParamPreferences) {
			Map<String, Object> prefs = ((ParamPreferences) controlPanel).getParamPreferences();
			if (prefs != null) {
				Preferences.set(currentRecognizer.getName(), 
					((ParamPreferences) controlPanel).getParamPreferences(), null);
			}
		}
		// clear the list of segmentations created by a previous run
		segmentations.clear();

		currentRecognizer.setMedia(selMediaFiles);
		
		startStopButton.setText(ElanLocale.getString("Button.Cancel"));
		reportButton.setEnabled(false);
		isRunning = true;
		progressBar.setString(ElanLocale.getString("Recognizer.RecognizerPanel.Recognizing"));
		new Thread(this).start();		
	}
	
	/**
	 * Prompts the user to specify a location where to store the selections.
	 * @param contentType FileParam.TIER or FileParam.CSV_TIER
	 * 
	 * @return the path or null if canceled
	 */
	private String promptForTierFile(int contentType, String startingPoint) {
		String prefPath = null;
		if (startingPoint != null && startingPoint.length() > 0) {
			prefPath = startingPoint;
		} 
		
		String[] extensions = null;
		if (contentType == FileParam.CSV_TIER) {
			extensions = FileExtension.CSV_EXT;
		} else {
			extensions = FileExtension.XML_EXT;
		}
		FileChooser chooser = new FileChooser(this);		
		
		chooser.createAndShowFileDialog(null, FileChooser.SAVE_DIALOG,  extensions, "Recognizer.Path");
		
		File f = chooser.getSelectedFile();
		if (f != null) {			
			return f.getAbsolutePath();
		} else {
			return null;
		}		
	}
	
	/**
	 * Creates a filename based on the first media file and the parameter identifier.
	 * Instead of prompting the user to specify a path. 
	 * The file is also created if it does not exist.
	 * 
	 * @param mediaFiles the selected media files
	 * @param paramName the name/identifier of a parameter
	 * @param contentType csv or xml or other
	 * 
	 * @return the path to a file
	 */
	private String autoCreateOutputFile(List<String> mediaFiles, String paramName, int contentType) {
		if (mediaFiles != null && mediaFiles.size() > 0) {
			String firstMed = mediaFiles.get(0);
			int dirIndex = firstMed.lastIndexOf(File.separator);// '/'?
			StringBuilder dirPath = null;
			
			if (dirIndex > -1 && dirIndex < firstMed.length() - 1) {
				dirPath = new StringBuilder(firstMed.substring(0, dirIndex + 1));
			} else {
				// no valid last path separator
				int index = firstMed.lastIndexOf('.');
				if (index > -1 && index < firstMed.length() - 1) {
					dirPath = new StringBuilder(firstMed.substring(0, index + 1));
				} else {
					dirPath = new StringBuilder(firstMed);
				}
			}
			
			if (dirPath != null) {
				dirPath.append(paramName.replace(' ', '_'));
				if (contentType == FileParam.CSV_TIER || contentType == FileParam.CSV_TS) {
					dirPath.append(".csv");
				} else if (contentType == FileParam.TIER || contentType == FileParam.TIMESERIES) {
					dirPath.append(".xml");
				}
				try {
					File out = new File(dirPath.toString());
					if (!out.exists()) {
						out.createNewFile();
						if (currentRecognizer instanceof SharedRecognizer) {
							try {
								changeFileAccess(out);
							} catch (Exception ex) {// just catch any exception that can occur
								ClientLogger.LOG.warning("Cannot change the file permissions: " + ex.getMessage());
							}
						}
					}
				} catch (IOException ioe) {
					ClientLogger.LOG.warning("Cannot create the file: " + ioe.getMessage());
				} catch (SecurityException se) {
					ClientLogger.LOG.warning("Cannot create the file: " + se.getMessage());
				}
				
				return dirPath.toString();
			}
		}
		
		return null;
	}
	
	/**
	 * Create an output file for a recognizer based on a user specified path 
	 * and set the access rights.
	 * 
	 * @param path the path to the file
	 */
	private void createOutputFile(String path) {
		if (path == null) {
			return;
		}
		try {
			File out = new File(path);
			if (!out.exists()) {
				out.createNewFile();
				if (currentRecognizer instanceof SharedRecognizer) {
					try {
						changeFileAccess(out);
					} catch (Exception ex) {// just catch any exception that can occur
						ClientLogger.LOG.warning("Cannot change the file permissions: " + ex.getMessage());
					}
				}
			}
		} catch (IOException ioe) {
			ClientLogger.LOG.warning("Cannot create the file: " + ioe.getMessage());
		} catch (SecurityException se) {
			ClientLogger.LOG.warning("Cannot create the file: " + se.getMessage());
		}
	}
	
	/**
	 * Tries to make the output file readable, writable and executable for all users.
	 * 
	 * @param f the file
	 */
	private void changeFileAccess(File f) {
		if (f == null) {
			return;
		}
		// java 1.6 method via reflection, on Windows Vista this doesn't change anything even if true is returned
		/*
		Class<?>[] params = new Class[2];
		params[0] = boolean.class;
		params[1] = boolean.class;
		Object[] values = new Object[2];
		values[0] = Boolean.TRUE;
		values[1] = Boolean.FALSE;
		try {
			Method m = f.getClass().getMethod("setExecutable", params);
			Object result = m.invoke(f, values);
			if (result instanceof Boolean) {
				ClientLogger.LOG.info("Set executable: " + result);
			}
			m = f.getClass().getMethod("setWritable", params);
			result = m.invoke(f, values);
			if (result instanceof Boolean) {
				ClientLogger.LOG.info("Set writable: " + result);
			}
		} catch (NoSuchMethodException nsme) {
		*/
			// no java 1.6, try system tools
			ArrayList<String> coms = new ArrayList<String>(5);
			
			if (SystemReporting.isWindows()) {				
				coms.add("CACLS");
				coms.add("\"" + f.getAbsolutePath() + "\"");
				coms.add("/E");
				coms.add("/G");
				coms.add("Everyone:f"); //avatech fails "No mapping between account names and security IDs was done."			
			} else {// MacOS, Linux
				coms.add("chmod");
				coms.add("a+rwx");
				coms.add("\"" + f.getAbsolutePath() + "\"");
			}
			
			ProcessBuilder pb = new ProcessBuilder(coms);
			pb.redirectErrorStream(true);
			try {
				Process proc = pb.start();
				int exit = proc.exitValue();
				if (exit != 0) {
					ClientLogger.LOG.warning("Could not set the file access attributes via using native tool, error: " + exit);
				}
			} catch (IOException ioe) {
				ClientLogger.LOG.warning("Cannot set the file access attributes via native tool");
			}
		/*	
		} catch (Exception ex) {
			// any other exception
			ClientLogger.LOG.warning("Cannot set the file access attributes");
		}
		*/
	}
	
	/**
	 * When the recognizer has finished, check if there is output like 
	 * tier or timeseries files and ask if the segments and 
	 * the tracks should be loaded. 
	 */
	private void checkOutput() {
		if (isRunning) {
			return;
		}
		boolean loadTS = false;
		boolean loadTiers = false;
		boolean tsAvailable = false;
		boolean tierFileAvailable = false;

		if (controlPanel instanceof ParamPanelContainer) {			
			ParamPanelContainer ppc = (ParamPanelContainer) controlPanel;
			int numPanels = ppc.getNumPanels();
			AbstractParamPanel app;
			FileParamPanel fpp;
			// first loop for prompt
			for (int i = 0; i < numPanels; i++) {
				app = ppc.getParamPanel(i);
				if (app instanceof FileParamPanel) {
					fpp = (FileParamPanel) app;
					if (!fpp.isInputType() && 
							(fpp.getContentType() == FileParam.CSV_TS || fpp.getContentType() == FileParam.TIMESERIES)) {
						Object file = fpp.getParamValue();
						if (file instanceof String) {
							File f = new File((String) file);
							
							if (f.exists() && f.canRead() && f.lastModified() > lastStartTime) {
								tsAvailable = true;					
							}
						}	
					} else if (!fpp.isInputType() && 
							(fpp.getContentType() == FileParam.TIER || fpp.getContentType() == FileParam.CSV_TIER)) {
						Object file = fpp.getParamValue();
						if (file instanceof String) {
							File f = new File((String) file);
							
							if (f.exists() && f.canRead() && f.lastModified() > lastStartTime) {
								tierFileAvailable = true;
								//break;						
							}
						}	
					}
				}
			}
			
			boolean tiersAvailable = segmentations.size() > 0;
			if (!tiersAvailable && !tsAvailable && !tierFileAvailable) {
				JOptionPane.showMessageDialog(this, 
						ElanLocale.getString("Recognizer.RecognizerPanel.Warning.NoOutput"), 
						ElanLocale.getString("Message.Warning"), JOptionPane.WARNING_MESSAGE);
				return;
			}
			// ask the user whether tracks and tiers should be loaded
			List<SelectableObject> resources = new ArrayList<SelectableObject>(4);

			if (tiersAvailable || tierFileAvailable) {
				boolean sel = true;
				Object pref = Preferences.get("Recognizer.RecognizerPanel.Tiers", null);

				if (pref instanceof Boolean) {
					sel = ((Boolean) pref).booleanValue();
				}
				resources.add(new SelectableObject(
						ElanLocale.getString("Recognizer.RecognizerPanel.Tiers"), sel));

			}
			if (tsAvailable) {
				boolean sel = true;
				Object pref = Preferences.get("Recognizer.RecognizerPanel.TimeSeries", null);

				if (pref instanceof Boolean) {
					sel = ((Boolean) pref).booleanValue();
				}
				resources.add(new SelectableObject(
						ElanLocale.getString("Recognizer.RecognizerPanel.TimeSeries"), sel));
			}
			
			LoadOutputPane lop = new LoadOutputPane(resources);
			int option = JOptionPane.showConfirmDialog(this, 
					lop,
					"", 
					JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
			if (option != JOptionPane.YES_OPTION) {
				return;
			} else {

				SelectableObject selObj;
				for (int i = 0; i < resources.size(); i++) {
					selObj = resources.get(i);
					if (selObj.getValue().equals(ElanLocale.getString("Recognizer.RecognizerPanel.Tiers"))) {
						loadTiers = selObj.isSelected();
						Preferences.set("Recognizer.RecognizerPanel.Tiers", new Boolean(loadTiers), null, false, false);
					}
					else if (selObj.getValue().equals(ElanLocale.getString("Recognizer.RecognizerPanel.TimeSeries"))) {
						loadTS = selObj.isSelected();
						Preferences.set("Recognizer.RecognizerPanel.TimeSeries", new Boolean(loadTS), null, false, false);
					}
				}
			}
			
			if (loadTiers) {
				if (tierFileAvailable && !tiersAvailable) {
					// load them from file if possible
					List<File> csvFiles = new ArrayList<File>(4);
					List<File> xmlFiles = new ArrayList<File>(4);
					for (int i = 0; i < numPanels; i++) {
						app = ppc.getParamPanel(i);
						if (app instanceof FileParamPanel) {
							fpp = (FileParamPanel) app;
							if (!fpp.isInputType() && 
									(fpp.getContentType() == FileParam.CSV_TIER || fpp.getContentType() == FileParam.TIER)) {
								Object file = fpp.getParamValue();
								if (file instanceof String) {
									File f = new File((String) file);
									if (f.exists() && f.canRead() && f.lastModified() > lastStartTime) {
										if (fpp.getContentType() == FileParam.CSV_TIER) {
											csvFiles.add(f);
										} else {
											xmlFiles.add(f);
										}
									}
								}
							}
						}
					}
					
					for (File csvFile : csvFiles) {
						if (csvFile.exists() && csvFile.canRead() && csvFile.lastModified() > lastStartTime) {
							CsvTierIO cio = new CsvTierIO();
							List<Segmentation> segm = cio.read(csvFile);
							if (segm != null && segm.size() > 0) {
								for (Segmentation s : segm) {
									addSegmentation(s);
								}
							}
						}
					}
					for (File xmlFile : xmlFiles) {
						if (xmlFile.exists() && xmlFile.canRead() && xmlFile.lastModified() > lastStartTime) {
							XmlTierIO xio = new XmlTierIO(xmlFile);
							List<Segmentation> segm = xio.parse();
							if (segm != null && segm.size() > 0) {
								for (Segmentation s : segm) {
									addSegmentation(s);
								}
							}
						}
					}
				}
	            // needs the transcription
				HashMap<String, List<AnnotationDataRecord>> segmentationMap = 
					new HashMap<String, List<AnnotationDataRecord>>();
				Segmentation seg;
		        Segment segment;
		        ArrayList segments = null;
		        ArrayList<AnnotationDataRecord> records = null;
		        ArrayList<Segmentation> segs = getSegmentations();
		        for (int i = 0; i < segs.size(); i++) {
		            seg = (Segmentation) segs.get(i);

		            if (seg == null) {
		                continue;
		            }

		            segments = seg.getSegments();
		            records = new ArrayList<AnnotationDataRecord>();

		            for (int j = 0; j < segments.size(); j++) {
		                segment = (Segment) segments.get(j);
		                records.add(new AnnotationDataRecord("", segment.label,
		                        segment.beginTime, segment.endTime));
		            }

		            segmentationMap.put(seg.getName(), records);
		        }

		        // create command
		        Command com = (SegmentsToTiersCommand) ELANCommandFactory.createCommand(
		        		viewerManager.getTranscription(),
		                ELANCommandFactory.SEGMENTS_2_TIER);
		        //com.addProgressListener(this);
		        //progressBar.setIndeterminate(false);
		        //progressBar.setValue(0);
		        com.execute(viewerManager.getTranscription(), new Object[] { segmentationMap });

			}
			
			if (loadTS) {
				List<File> csvFiles = new ArrayList<File>(4);
				List<File> xmlFiles = new ArrayList<File>(4);
				for (int i = 0; i < numPanels; i++) {
					app = ppc.getParamPanel(i);
					if (app instanceof FileParamPanel) {
						fpp = (FileParamPanel) app;
						if (!fpp.isInputType() && 
								(fpp.getContentType() == FileParam.CSV_TS || fpp.getContentType() == FileParam.TIMESERIES)) {
							Object file = fpp.getParamValue();
							if (file instanceof String) {
								File f = new File((String) file);
								if (f.exists() && f.canRead() && f.lastModified() > lastStartTime) {
									if (fpp.getContentType() == FileParam.CSV_TS) {
										csvFiles.add(f);
									} else {
										xmlFiles.add(f);
									}
								}
							}
						}
					}
				}
				
				List<Object> tracks = new ArrayList<Object>(10);
				for (File f : csvFiles) {
					CsvTimeSeriesIO csvIO = new CsvTimeSeriesIO(f);
					List<Object> result = csvIO.getAllTracks();
					if (result != null) {
						tracks.addAll(result);
					}
				}
				for (File f : xmlFiles) {
					XmlTimeSeriesReader xmlIO = new XmlTimeSeriesReader(f);
					try {
						List<Object> result = xmlIO.parse();
						if (result != null) {
							tracks.addAll(result);
						}
					} catch (IOException ioe) {
						JOptionPane.showMessageDialog(this, 
								ElanLocale.getString("Recognizer.RecognizerPanel.Warning.LoadFailed") + "\n" + 
								ioe.getMessage(), 
								ElanLocale.getString("Message.Error"), JOptionPane.ERROR_MESSAGE);
					} catch (SAXException sax) {
						JOptionPane.showMessageDialog(this, 
								ElanLocale.getString("Recognizer.RecognizerPanel.Warning.LoadFailed") + "\n" + 
								sax.getMessage(), 
								ElanLocale.getString("Message.Error"), JOptionPane.ERROR_MESSAGE);
					}
				}
				
				Command com = ELANCommandFactory.createCommand(viewerManager.getTranscription(), 
						ELANCommandFactory.ADD_TRACK_AND_PANEL);
				com.execute(viewerManager, new Object[]{tracks});
			}
		}
	}
	
	/**
	 * This method will run in a separate Thread to decouple the recognizer from ELAN
	 */
	public void run() {
		if (currentRecognizer != null) {
			currentRecognizer.start();
			lastStartTime = System.currentTimeMillis();
		}
	}

	/**
	 * Takes care of giving running recognizers a chance to stop gracefully
	 * ELAN should call this method before it quits
	 *
	 */
	public void stopRecognizers() {
		if (currentRecognizer != null) {
			currentRecognizer.stop();
			isRunning = false;
		}
	}

	/**
	 * Tells if there is one or more recognizer busy
	 * 
	 * @return true if there is some recognizing going on
	 */
	public boolean isBusy() {
		return isRunning;
	}

	/**
	 * Sets the localized labels for ui elements.
	 */
	public void updateLocale() {
		selectionPanel.updateLocale();
		segmentationPanel.updateLocale();
		if (currentRecognizer != null) {
			currentRecognizer.updateLocale(ElanLocale.getLocale());
		}
		startStopButton.setText(ElanLocale.getString("Recognizer.RecognizerPanel.Start"));
		reportButton.setText(ElanLocale.getString("Recognizer.RecognizerPanel.Report"));
		recognizerLabel.setText(ElanLocale.getString("Recognizer.RecognizerPanel.Recognizer"));
		filesLabel.setText(ElanLocale.getString("Recognizer.RecognizerPanel.Files"));
		progressPanel.setBorder(new TitledBorder(ElanLocale.getString("Recognizer.RecognizerPanel.Progress")));
		paramPanel.setBorder(new TitledBorder(ElanLocale.getString("Recognizer.RecognizerPanel.Parameters")));
		detachButton.setToolTipText(ElanLocale.getString("Detachable.detach"));
		saveParamsButton.setToolTipText(ElanLocale.getString("Recognizer.RecognizerPanel.SaveParameters"));
		loadParamsButton.setToolTipText(ElanLocale.getString("Recognizer.RecognizerPanel.LoadParameters"));
	}
	
	//
	// RecognizerHost interface implementation
	//
	
	/**
	 * Add a segmentation to the 
	 */
	public void addSegmentation(Segmentation segmentation) {
		segmentations.put(segmentation.getName(), segmentation);
		segmentationPanel.setEnabled(true);
	}
	
	public ArrayList<Segmentation> getSegmentations() {
		return new ArrayList<Segmentation>(segmentations.values());
	}
	
	/**
	 * By calling this method a recognizer gives information about the progress of its recognition task
	 * 
	 * @param progress  a float between 0 and 1 with 1 meaning that the task is completed
	 */
	public void setProgress(float progress) {
		if (progress < 0) {
			if (!progressBar.isIndeterminate()) {
				progressBar.setIndeterminate(true);
			}
		} else {
			if (progressBar.isIndeterminate()) {
				progressBar.setIndeterminate(false);
			}
			progressBar.setValue((int) (100 * progress));
			if (progress >= 1) {
				progressBar.setString(ElanLocale.getString("Recognizer.RecognizerPanel.Ready"));
				startStopButton.setText(ElanLocale.getString("Recognizer.RecognizerPanel.Start"));
				isRunning = false;
				reportButton.setEnabled(true);
				// get report??
				// check for any output
				checkOutput();
			}
		}
	}

	/**
	 * By calling this method a recognizer gives information about the progress of its recognition task
	 * 
	 * @param progress  a float between 0 and 1 with 1 meaning that the task is completed
	 * @param message a progress message
	 */
	public void setProgress(float progress, String message) {
		setProgress(progress);
		if (isRunning) {// still running?
			if (message != null) {
				progressBar.setString(message);
			} else {
				progressBar.setString("");
			}
		}
	}

	/**
	 * By calling this method the recognizer gets the selections for an audio signal 
	 * 
	 * @return An ArrayList with RSelection Objects 
	 */
	public ArrayList<RSelection> getSelections() {
		return selectionPanel.getSelections();
	}

	/**
	 * Called by the recognizer to signal that a fatal error occurred.
	 * 
	 * @param message a description of the error
	 */
	public void errorOccurred(String message) {
		JOptionPane.showMessageDialog(this, ElanLocale.getString("Recognizer.RecognizerPanel.Error") + "\n" + message, 
				ElanLocale.getString("Message.Error"), JOptionPane.ERROR_MESSAGE);
		// just to be sure
		if (isRunning) {
			currentRecognizer.stop();
		}
		setProgress(1f);
	}
	
	/**
	 * Preliminary!
	 * Attaches the parameter panel to the main panel again.
	 */
	public void attachParamPanel(JComponent paramComp) {
		// is the param scrollpane
		if (paramComp == jsp) {
			GridBagConstraints gbc = new GridBagConstraints();
			gbc.anchor = GridBagConstraints.NORTHWEST;
			gbc.fill = GridBagConstraints.BOTH;
			gbc.gridy = 0;
			gbc.weightx = 1.0;
			gbc.weighty = 1.0;
			paramPanel.add(jsp, gbc);
			paramButtonPanel.setVisible(true);
		}
	}
	
	/**
	 * Preliminary!
	 * Detaches the parameter panel from the main panel.
	 */
	public void detachParamPanel() {
		paramPanel.remove(jsp);
		paramPanel.repaint();
		paramButtonPanel.setVisible(false);
		ParamDialog dialog = new ParamDialog(ELANCommandFactory.getRootFrame(
				viewerManager.getTranscription()), this, jsp);
		dialog.pack();
		Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
		Rectangle curRect = dialog.getBounds();
		// add some extra width for a possible scrollbar in the scrollpane
		dialog.setBounds(screen.width / 2, 10, 
				Math.min(screen.width / 2 - 20, curRect.width + 30), Math.min(screen.height - 30, curRect.height));
		dialog.setVisible(true);
	}
	
	/**
	 * Loads the parameter settings from a PARAM XML file and applies them to the current recognizer. 
	 */
	protected void loadParameterFile() {
		if (currentRecognizer == null || controlPanel == null || !(controlPanel instanceof ParamPreferences)) {
			return;
		}
		
		FileChooser chooser = new FileChooser(this);
		chooser.createAndShowFileDialog(null, FileChooser.OPEN_DIALOG, FileExtension.XML_EXT, "Recognizer.Path");
		File selFile = chooser.getSelectedFile();
		if (selFile != null && selFile.canRead()) {
			try {
				ParamIO pio = new ParamIO();
				Map<String, Object> parMap = pio.read(selFile);
				((ParamPreferences) controlPanel).setParamPreferences(parMap);
			} catch (IOException ioe) {
				// 
				JOptionPane.showMessageDialog(this, ElanLocale.getString(
					"Recognizer.RecognizerPanel.Warning.LoadFailed")  + ioe.getMessage(), 
					ElanLocale.getString("Message.Warning"), JOptionPane.ERROR_MESSAGE);
			}
		}
	}
	
	/**
	 * Writes the current parameter settings to a PARAM XML file. 
	 */
	protected void saveParameterFile() {
		if (currentRecognizer == null || controlPanel == null || !(controlPanel instanceof ParamPreferences)) {
			return;
		}
		
		FileChooser chooser = new FileChooser(this);
		chooser.createAndShowFileDialog(null, FileChooser.SAVE_DIALOG, FileExtension.XML_EXT, "Recognizer.Path");	
		File f = chooser.getSelectedFile();
		if(f != null){	
			ParamIO pio = new ParamIO();
			// hier... or get parameters from the recognizer?
			Map<String, Object> paramMap = ((ParamPreferences) controlPanel).getParamPreferences();
			
			try {
				pio.write(paramMap, f);
			} catch (IOException ioe) {
				// message
				JOptionPane.showMessageDialog(this, ElanLocale.getString(
						"Recognizer.RecognizerPanel.Warning.SaveFailed")  + ioe.getMessage(), 
						ElanLocale.getString("Message.Warning"), JOptionPane.ERROR_MESSAGE);
			}			
		}
	}
	
	/**
	 * Writes the selections from the selection panel to a csv or xml file.
	 * It is assumed that a check has been performed on the list of selections and that
	 * the file is save to save to.
	 * 
	 * @param filepath the path to the file
	 * @param contentType CSV_TIER or TIER (xml)
	 * @throws IOException any io error that can occur during writing
	 */
	protected void writeTierFile(String filepath, int contentType) throws IOException {
		RecTierWriter xTierWriter = new RecTierWriter();
		File f = new File(filepath);
		xTierWriter.write(f, selectionPanel.getTiersAndSelections(), true);
		//xTierWriter.write(f, selectionPanel.getSelections());
	}
	
	/**
	 * Creates a dialog with a text area containing the report.
	 * 
	 * @param report the report
	 */
	protected void showReport(String report) {
		if (report != null) {
			JDialog rD = new JDialog(ELANCommandFactory.getRootFrame(viewerManager.getTranscription()), true);
			JTextArea ta = new JTextArea(report);
			ta.setLineWrap(true);
			ta.setWrapStyleWord(true);
			ta.setEditable(false);
			rD.getContentPane().setLayout(new BorderLayout());
			rD.getContentPane().add(new JScrollPane(ta), BorderLayout.CENTER);
			rD.pack();
			// set minimum size
			rD.setSize(Math.max(rD.getWidth(), 400), Math.max(rD.getHeight(), 300));
			// set maximum size
			rD.setSize(Math.min(rD.getWidth(), 800), Math.min(rD.getHeight(), 600));
			rD.setLocationRelativeTo(this);
			
			rD.setVisible(true);
		} else {
			JOptionPane.showMessageDialog(this, ElanLocale.getString("Recognizer.RecognizerPanel.No.Report"), 
				ElanLocale.getString("Message.Warning"), JOptionPane.INFORMATION_MESSAGE);
		}
	}

	@Override
	protected void finalize() throws Throwable {
		if (currentRecognizer != null) {
			if (isRunning) {
				currentRecognizer.stop();
				currentRecognizer.dispose();
			}
			currentRecognizer = null;
		}
		super.finalize();
	}
	
	
}
