/**
 * 
 */
package mpi.eudico.client.annotator.recognizer.gui;

import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JTextField;
import mpi.eudico.client.annotator.gui.FileChooser;
import mpi.eudico.client.annotator.recognizer.data.FileParam;
import mpi.eudico.client.annotator.util.FileExtension;

/**
 * A panel to specify a file or folder for input for or output from a detector.
 *  
 * @author Han Sloetjes
 */
public class FileParamPanel extends AbstractParamPanel implements ActionListener {
	private boolean inputType;
	private int contentType;
	private boolean optional;
	private String initialPath;
	
	private JTextField fileField;
	private JButton browseButton;
	
	/**
	 * The name of the parameter.
	 * 
	 * @param paramName the name of the parameter
	 * @param inputType if true, this is for selecting an input file for the detector, otherwise
	 * a file or folder for output needs to be specified
	 * @param contentType the type of the content of the file, tier, timeseries, audio, video, auxiliary. 
	 * A constant from FileParam.
	 * @param optional if true this parameter does not have to be set
	 */
	public FileParamPanel(String paramName, String description, boolean inputType, int contentType, boolean optional) {
		super(paramName, description);
		this.inputType = inputType;
		this.contentType = contentType;
		this.optional = optional;
		initComponents();
	}
	
	/**
	 * Constructor accepting a FileParam as an argument.
	 * 
	 * @param param the file param object
	 */
	public FileParamPanel(FileParam param) {
		super(param);
		if (param != null) {
			inputType = (param.ioType == FileParam.IN);
			optional = param.optional;
			contentType = param.contentType;
			initComponents();
		}
	}

	/**
	 * Adds a textfield and a browse button to the panel.
	 */
	@Override
	protected void initComponents() {
		super.initComponents();
		ImageIcon icon = null;
		try {
			if (inputType) {
				icon = new ImageIcon(this.getClass().getResource("/toolbarButtonGraphics/general/Open16.gif"));
			} else {
				icon = new ImageIcon(this.getClass().getResource("/toolbarButtonGraphics/general/Save16.gif"));
			}
		} catch (Exception ex) {
			
		}
		fileField = new JTextField();
		fileField.setEditable(true);
		browseButton = new JButton();// or icon?
		if (icon != null) {
			browseButton.setIcon(icon);
		} else {
			browseButton.setText("...");
		}
		browseButton.addActionListener(this);
		
		GridBagConstraints gbc = new GridBagConstraints();
		gbc.gridwidth = 2;
		gbc.gridy = 1;
		gbc.anchor = GridBagConstraints.WEST;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.weightx = 1.0;
		gbc.weighty = 0.0;
		gbc.insets = new Insets(1, 1, 4, 1);
		add(fileField, gbc);
		gbc.gridx = 2;
		gbc.fill = GridBagConstraints.NONE;
		gbc.weightx = 0.0;
		add(browseButton, gbc);
		
		StringBuilder builder = new StringBuilder("<html><table><tr>");
		if (inputType) {
			if (!optional) {
				builder.append("<td style=\"color:red\">IN ");
			} else {
				builder.append("<td style=\"color:green\">IN ");
			}
		} else {
			if (!optional) {
				builder.append("<td style=\"color:red\">OUT ");
			} else {
				builder.append("<td style=\"color:green\">OUT ");
			}
		}
		if (contentType == FileParam.CSV_TIER || contentType == FileParam.CSV_TS) {
			builder.append("<small>[csv]</small>: </td><td>");
		} else if (contentType == FileParam.TIER || contentType == FileParam.TIMESERIES) {
			builder.append("<small>[xml]</small>: </td><td>");
		} else {
			builder.append("<small>[aux]</small>: </td><td>");
		}
		builder.append(description);
		builder.append(" <i>[" + paramName + "]</i>");
		builder.append("</td><tr></table></html>");
		descLabel.setText(builder.toString());
	}


	/**
	 * Returns the path to the file or folder.
	 * 
	 * @see mpi.eudico.client.annotator.recognizer.gui.AbstractParamPanel#getParamValue()
	 */
	@Override
	protected Object getParamValue() {
		return fileField.getText();
	}

	/**
	 * Sets a previously stored value for the path.
	 * 
	 * @see mpi.eudico.client.annotator.recognizer.gui.AbstractParamPanel#setParamValue(java.lang.Object)
	 */
	@Override
	protected void setParamValue(Object value) {
		if (fileField != null && value instanceof String) {
			initialPath = (String) value;
			fileField.setText(initialPath);
		}

	}

	/**
	 * Returns whether this file is optional.
	 * 
	 * @return whether this file is optional
	 */
	public boolean isOptional() {
		return optional;
	}
	
	/**
	 * 
	 * @return the content type, tier or timeseries currently, in csv or xml
	 */
	public int getContentType() {
		return contentType;
	}
	
	/**
	 * 
	 * @return whether this is an input or output file panel
	 */
	public boolean isInputType() {
		return inputType;
	}
	
	/**
	 * Only the browse button can currently be the source.
	 */
	public void actionPerformed(ActionEvent e) {
		FileChooser chooser = new FileChooser(this);
		chooser.setCurrentDirectory(initialPath);
		String[] extensions = null;
		if (contentType == FileParam.CSV_TIER || contentType == FileParam.CSV_TS) {
			extensions = FileExtension.CSV_EXT;
		} else if (contentType == FileParam.TIER || contentType == FileParam.TIMESERIES) {
			extensions = FileExtension.XML_EXT;
		}
		int dialogType = -1;

		if (inputType) {
			dialogType = FileChooser.OPEN_DIALOG;			
		} else {
			// output
			dialogType = FileChooser.SAVE_DIALOG;
		}
		
		chooser.createAndShowFileDialog(null, dialogType, extensions, null);
		File f = chooser.getSelectedFile();
		if(f != null){
			initialPath = f.getAbsolutePath();
			fileField.setText(initialPath);
		}
	}
}
