package mpi.eudico.client.annotator.recognizer.gui;

import java.awt.GridLayout;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.JPanel;
import javax.swing.border.EmptyBorder;

import mpi.eudico.client.annotator.recognizer.api.ParamPreferences;
import mpi.eudico.client.annotator.recognizer.data.FileParam;
import mpi.eudico.client.annotator.recognizer.data.NumParam;
import mpi.eudico.client.annotator.recognizer.data.Param;
import mpi.eudico.client.annotator.recognizer.data.TextParam;

/**
 * A container panel for parameter panels.
 * 
 * @author Han Sloetjes
 */
public class ParamPanelContainer extends JPanel implements ParamPreferences {
	private String recogizerName;
	private GridLayout gLayout;
	private List<AbstractParamPanel> panels;
	
	/**
	 * A container panel for parameter panels.
	 * 
	 * @param recogizerName the name of the recognizer
	 */
	public ParamPanelContainer(String recogizerName) {
		super();
		this.recogizerName = recogizerName;
		gLayout = new GridLayout(0, 1, 2, 2);
		this.setLayout(gLayout);
		this.setBorder(new EmptyBorder(2, 2, 2, 0));
		panels = new ArrayList<AbstractParamPanel>(10);
	}
	
	/**
	 * A container panel for parameter panels.
	 * 
	 * @param recogizerName the name of the recognizer
	 * @param params the list of parameters of the recognizer 
	 */
	public ParamPanelContainer(String recogizerName, List<Param> params) {
		super();
		this.recogizerName = recogizerName;
		gLayout = new GridLayout(0, 1, 2, 2);
		this.setLayout(gLayout);
		this.setBorder(new EmptyBorder(2, 2, 2, 0));
		if (params != null) {
			panels = new ArrayList<AbstractParamPanel>(params.size());
			for (Param p : params) {
				if (p instanceof NumParam) {
					addParamPanel(new NumParamPanel((NumParam) p));
				} else if (p instanceof TextParam) {
					addParamPanel(new TextParamPanel((TextParam) p));
				} else if (p instanceof FileParam) {
					FileParamPanel fpp = new FileParamPanel((FileParam) p);
					if (((FileParam) p).ioType == FileParam.IN && 
							(((FileParam) p).contentType == FileParam.AUDIO || ((FileParam) p).contentType == FileParam.VIDEO)) {
						fpp.setEnabled(false);
						continue;// don't add for the time being, is taken care of in other ui elements
					}
					addParamPanel(fpp);
				}
			}
		} else {
			panels = new ArrayList<AbstractParamPanel>(10);
		}
		//setBackground(Color.RED);
	}

	/**
	 * Returns the name of the recognizer this panel is presenting a ui for.
	 * 
	 * @return the name of the recognizer
	 */
	public String getRecognizerName() {
		return recogizerName;
	}
	
	/**
	 * Stores the current values.
	 * 
	 * @return a map with the current values
	 */
	public Map<String, Object> getParamPreferences() {
		Map<String, Object> storedPrefs = new HashMap<String, Object>(panels.size());
		
		for (AbstractParamPanel p : panels) {
			storedPrefs.put(p.getParamName(), p.getParamValue());
		}
		
		return storedPrefs;
	}

	/**
	 * Restores previously stored values.
	 * 
	 * @param storedPrefs the (previously used) stored values
	 */
	public void setParamPreferences(Map<String, Object> storedPrefs) {
		String key;
		Object val;
		Iterator<String> keyIt = storedPrefs.keySet().iterator();
		
		while (keyIt.hasNext()) {
			key = keyIt.next();
			for (AbstractParamPanel p : panels) {
				if (p.getParamName() != null && p.getParamName().equals(key)) {
					p.setParamValue(storedPrefs.get(key));
					break;
				}
			}
		}
		
	}

	/**
	 * Returns the number of panels (= the number of parameters).
	 * 
	 * @return the number of panels
	 */
	public int getNumPanels() {
		return panels.size();
	}
	
	/**
	 * Returns the panel at the given index.
	 * 
	 * @param index the index 
	 * @return the panel at the index or null (in case of index out of bounds)
	 */
	public AbstractParamPanel getParamPanel(int index) {
		if (index >= 0 && index < panels.size()) {
			return panels.get(index);
		}
		
		return null;
	}
	
	/**
	 * Adds a panel to the container.
	 * 
	 * @param app the panel
	 */
	public void addParamPanel(AbstractParamPanel app) {
		gLayout.setRows(gLayout.getRows() + 1);
		add(app);
		panels.add(app);
	}
}
