package mpi.eudico.client.annotator.recognizer.gui;

import java.awt.BorderLayout;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.Shape;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;
import javax.swing.border.TitledBorder;
import mpi.eudico.client.annotator.Selection;
import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.ViewerManager2;
import mpi.eudico.client.annotator.commands.ELANCommandFactory;
import mpi.eudico.client.annotator.gui.FileChooser;
import mpi.eudico.client.annotator.gui.TierSelectionDialog;
import mpi.eudico.client.annotator.player.ElanMediaPlayer;
import mpi.eudico.client.annotator.player.VideoFrameGrabber;
import mpi.eudico.client.annotator.recognizer.data.AudioSegment;
import mpi.eudico.client.annotator.recognizer.data.RSelection;
import mpi.eudico.client.annotator.recognizer.data.Segment;
import mpi.eudico.client.annotator.recognizer.data.Segmentation;
import mpi.eudico.client.annotator.recognizer.data.SelectionComparator;
import mpi.eudico.client.annotator.recognizer.data.VideoSegment;
import mpi.eudico.client.annotator.recognizer.io.RecTierWriter;
import mpi.eudico.client.annotator.svg.Graphics2DEditor;
import mpi.eudico.client.annotator.util.FileExtension;
import mpi.eudico.server.corpora.clom.Tier;
import mpi.eudico.server.corpora.clomimpl.abstr.AbstractAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;

public class SelectionPanel extends JPanel implements ActionListener {
	private Selection selection;
	private ViewerManager2 viewerManager;
	private TitledBorder border;
	private JPanel buttonPanel;
	private JButton addSelection;
	private JButton addSelection1;
	private JButton addSelection2;
	private JButton removeSelection;
	private JPanel buttonPanel2;
	private JButton addFromTier;
	private JButton addFromTier2;
	private JList selectionList;
	private DefaultListModel selectionModel;
	public static final int AUDIO_MODE = 0; 
	public static final int VIDEO_MODE = 1;
	private int avMode = AUDIO_MODE;
	private boolean stereoMode = true;
	private List<String> mediaFiles;
	
	public SelectionPanel(int mode, boolean stereoMode, ViewerManager2 viewerManager) {
		super();
		avMode = mode;
		this.stereoMode = stereoMode;
		this.viewerManager = viewerManager;
		this.selection = viewerManager.getSelection();
		
		border = new TitledBorder(ElanLocale.getString("Recognizer.SelectionsPanel.Title"));
		setBorder(border);
		setLayout(new BorderLayout());

		buttonPanel = new JPanel();
		buttonPanel.setLayout(new GridLayout(1, 3));
		
		addSelection = new JButton(ElanLocale.getString("Recognizer.SelectionsPanel.Add"));
		addSelection.addActionListener(this);
		
		addSelection1 = new JButton(ElanLocale.getString("Recognizer.SelectionsPanel.Add.Channel1"));
		addSelection1.addActionListener(this);
		
		addSelection2 = new JButton(ElanLocale.getString("Recognizer.SelectionsPanel.Add.Channel2"));
		addSelection2.addActionListener(this);
		
		removeSelection = new JButton(ElanLocale.getString("Recognizer.SelectionsPanel.Remove"));
		removeSelection.addActionListener(this);
		
		if (stereoMode) {
			buttonPanel.add(addSelection1);
			buttonPanel.add(addSelection2);
			buttonPanel.add(removeSelection);
		} else {
			buttonPanel.add(addSelection);
			buttonPanel.add(removeSelection);
		}
		//
		buttonPanel2 = new JPanel();
		buttonPanel2.setLayout(new GridLayout(1, 2));
		
		addFromTier = new JButton(ElanLocale.getString("Recognizer.SelectionsPanel.Add.FromTier"));
		addFromTier.addActionListener(this);
		buttonPanel2.add(addFromTier);
		if (stereoMode) {
			addFromTier.setText(ElanLocale.getString("Recognizer.SelectionsPanel.Add.FromTier1"));
			addFromTier2 = new JButton(ElanLocale.getString("Recognizer.SelectionsPanel.Add.FromTier2"));
			addFromTier2.addActionListener(this);		
			buttonPanel2.add(addFromTier2);
		}
		
		JPanel compPanel = new JPanel();
		compPanel.setLayout(new BoxLayout(compPanel, BoxLayout.Y_AXIS));
		compPanel.add(buttonPanel);
		compPanel.add(buttonPanel2);
		add(compPanel, BorderLayout.NORTH);
		//
		selectionModel = new DefaultListModel();
		selectionList = new JList(selectionModel);
		selectionList.setCellRenderer(new SelectionListRenderer());
		selectionList.addMouseListener(new MouseHandler());
		selectionList.setBackground(getBackground());
		JScrollPane scrollPane = new JScrollPane(selectionList);
		add(scrollPane, BorderLayout.CENTER);
		
		// this forces the minimal width of the selection panel
		add(Box.createHorizontalStrut(240), BorderLayout.SOUTH);
	}
	
	public void updateLocale() {
		border.setTitle(ElanLocale.getString("Recognizer.SelectionsPanel.Title"));
		addSelection.setText(ElanLocale.getString("Recognizer.SelectionsPanel.Add"));
		addSelection1.setText(ElanLocale.getString("Recognizer.SelectionsPanel.Add.Channel1"));
		addSelection2.setText(ElanLocale.getString("Recognizer.SelectionsPanel.Add.Channel2"));
		removeSelection.setText(ElanLocale.getString("Recognizer.SelectionsPanel.Remove"));
		addFromTier.setText(ElanLocale.getString("Recognizer.SelectionsPanel.Add.FromTier"));
		if (addFromTier2 != null) {
			addFromTier.setText(ElanLocale.getString("Recognizer.SelectionsPanel.Add.FromTier1"));
			addFromTier2.setText(ElanLocale.getString("Recognizer.SelectionsPanel.Add.FromTier2"));
		}
	}
	
	/**
	 * Sets the new stereo mode.
	 * 
	 * @param stereoMode true if separate segments for channel one and channel two should be supported
	 */
	public void setStereoMode(boolean stereoMode) {
		if (this.stereoMode == stereoMode) {
			return;
		}
		this.stereoMode = stereoMode;
		if (stereoMode) {
			buttonPanel.removeAll();
			buttonPanel.add(addSelection1);
			buttonPanel.add(addSelection2);
			buttonPanel.add(removeSelection);
			buttonPanel2.add(addFromTier);
			buttonPanel2.add(addFromTier2);
		} else {
			buttonPanel.removeAll();
			buttonPanel.add(addSelection);
			buttonPanel.add(removeSelection);
			buttonPanel2.add(addFromTier);
		}
		//buttonPanel.getParent().getParent().validate();
		this.getParent().validate();
		// remove old selections
		//selections.clear();
		//selectionList.removeAll();
		selectionModel.removeAllElements();
	}
	
	/**
	 * Returns the current stereo mode.
	 * 
	 * @return true if currently segments for the left and right channel can be specified
	 */
	public boolean getStereoMode() {
		return stereoMode;
	}
	
	public void actionPerformed(ActionEvent e) {
		Object source = e.getSource();
		
		if (source.equals(addSelection)) {
			if (selection.getBeginTime() < selection.getEndTime()) {
				if (avMode == AUDIO_MODE) {
					addSelection(new AudioSegment(selection.getBeginTime(), selection.getEndTime(), null, 1));
				} else {
					addSelection(new VideoSegment(selection.getBeginTime(), selection.getEndTime(), null, null));
				}
			}
		} else if (source.equals(addSelection1)) {// audio segments
			if (selection.getBeginTime() < selection.getEndTime()) {
				addSelection(new AudioSegment(selection.getBeginTime(), selection.getEndTime(), null, 1));
			}
		} else if (source.equals(addSelection2)) {
			if (selection.getBeginTime() < selection.getEndTime()) {
				addSelection(new AudioSegment(selection.getBeginTime(), selection.getEndTime(), null, 2));
			}
		} else if (source.equals(removeSelection)) {
			int[] selIndices = selectionList.getSelectedIndices();
			if (selIndices != null) {
				for (int i = selIndices.length - 1; i >= 0; i--) {					
					selectionModel.remove(selIndices[i]);
				}
			}
		} else if (source == addFromTier) {
			addTier(1);
		} else if (source == addFromTier2) {
			addTier(2);
		}
		
	}
	
	/**
	 * Notifies the selection panel of the files that are currently selected for the recognizer.
	 * Used in video mode to determine which video file to show for changing the region of interest.
	 * 
	 * @param mediaFiles
	 */
	void setMediaFilePaths(List<String> mediaFiles) {
		if (avMode == VIDEO_MODE) {
			this.mediaFiles = mediaFiles;
		}
	}
	
	/**
	 *  Makes sure the selections are ordered and refuse double entries
	 *  
	 * @param selection new selection String
	 */
	/*
	private void updateSelections(String selection) {
		boolean inserted = false;
		for (int i = 0; i < selections.size(); i++) {
			if (selection.equals((String) selections.get(i))) {
				inserted = true; // nothing to do, this selection is already in the list
				break;
			} else if (selection.compareTo((String) selections.get(i)) < 0) { // ordering based on begin time string
				selections.insertElementAt(selection, i);
				inserted = true;
				break;
			}
		}
		if (!inserted) { // add selection to the end of the list
			selections.add(selection);
		}
		// update the GUI list
		selectionList.setListData(selections);
	}
	*/
	
	private void addSelection(RSelection sel) {
		if (selectionModel.isEmpty()) {
			selectionModel.addElement(sel);
			return;
		}
		
		Object iter;
		RSelection otherSel;
		
		for (int i = 0; i < selectionModel.getSize(); i++) {
			iter = selectionModel.get(i);
			if (iter instanceof RSelection) {
				otherSel = (RSelection) iter;
				if (otherSel.beginTime == sel.beginTime && otherSel.endTime == sel.endTime) {
					if (otherSel instanceof AudioSegment && sel instanceof AudioSegment) {
						if (((AudioSegment)otherSel).channel == ((AudioSegment)sel).channel) {
							// same selection already in the list
							return;
						} else {
							if (((AudioSegment)sel).channel < ((AudioSegment)otherSel).channel) {
								selectionModel.add(i, sel);
								return;
							} else {
								if (i < selectionModel.getSize() - 1) {
									selectionModel.add(i, sel);
									return;
								} else {
									selectionModel.addElement(sel);
									return;
								}
							}
						}
					} else {
						// same selection already in the list
						return;
					}
				} else if (otherSel.beginTime == sel.beginTime && otherSel.endTime > sel.endTime) {
					selectionModel.add(i, sel);// add before
					return;
				} else if (sel.beginTime < otherSel.beginTime) {
					selectionModel.add(i, sel);// add before
					return;
				}
			}
			if (i == selectionModel.getSize() - 1) {
				// not yet inserted
				selectionModel.addElement(sel);
			}
		}	
	}
	
	private void addSegmentation(Segmentation segmentation) {
		// tiers at the beginning of the list
		if (selectionModel.isEmpty()) {
			selectionModel.addElement(segmentation);
			return;
		}
		
		Object iter;
		//Segmentation otherSel;
		
		for (int i = 0; i < selectionModel.getSize(); i++) {
			iter = selectionModel.get(i);
			if (iter instanceof RSelection) {
				selectionModel.add(i, segmentation);
				return;
			}
		}
		// not yet added
		selectionModel.addElement(segmentation);
	}
	
	// for video, ignore the channel
	private void addTier(int channel) {
		List tiers = viewerManager.getTranscription().getTiers();
		if (tiers.size() == 0) {
			JOptionPane.showMessageDialog(this, ElanLocale.getString("LabelAndNumberDialog.Warning.NoTiers"), 
					ElanLocale.getString("Message.Warning"), JOptionPane.WARNING_MESSAGE);
			// message no tiers
			return;
		}
		List<String> tierNames = new ArrayList<String>(tiers.size());
		Tier t;
		for (int i = 0; i < tiers.size(); i++) {
			t = (Tier) tiers.get(i);
			tierNames.add(t.getName());
		}
		Window w = SwingUtilities.getWindowAncestor(this);
		TierSelectionDialog tsd = null;
		if (w instanceof Frame) {
			tsd = new TierSelectionDialog((Frame) w, tierNames, new ArrayList<String>(0));
		} else if (w instanceof Dialog) {
			tsd = new TierSelectionDialog((Dialog) w, tierNames, new ArrayList<String>(0));
		}
		tsd.setLocationRelativeTo(this);
		tsd.setVisible(true);
		List<String> selTiers = tsd.getValue();
		if (selTiers == null || selTiers.size() == 0) {
			return;
		}
		TierImpl ti;
		List anns;
		for (String name : selTiers) {
			ti = (TierImpl) viewerManager.getTranscription().getTierWithId(name);
			
			if (ti != null) {					
				anns = ti.getAnnotations();
				ArrayList<RSelection> segments = new ArrayList<RSelection>(anns.size());
				AbstractAnnotation aa;
				for (int j = 0; j < anns.size(); j++) {
					aa = (AbstractAnnotation) anns.get(j);
					segments.add(new AudioSegment(aa.getBeginTimeBoundary(), aa.getEndTimeBoundary(), 
							aa.getValue(), channel));
				}
				Segmentation segmentation = new Segmentation(name, segments, "", channel);
				addSegmentation(segmentation);
			}
		}		
	}
	
	/**
	 * Shows a popup for Audio and Video segments
	 */
	private void handlePopUp(Point p) {
		int row = selectionList.locationToIndex(p);
		
		if (row > -1) {
			Object sel = selectionModel.elementAt(row);

			if (sel instanceof AudioSegment) {
				// create popup with one setLabel item
				final AudioSegment as = (AudioSegment) sel;
				JPopupMenu popup = new JPopupMenu();
				JMenuItem mi = new JMenuItem(new LabelAction(as));
				popup.add(mi);
				popup.add(new SaveTierAction(null));
				popup.show(selectionList, p.x, p.y);				
			} else if (sel instanceof VideoSegment) {
				// create popup with a setLabel and setShape item
				final VideoSegment vs = (VideoSegment) sel;
				JPopupMenu popup = new JPopupMenu();
				JMenuItem mi = new JMenuItem(new LabelAction(vs));
				popup.add(mi);
				mi = new JMenuItem(new ShapeAction(vs));
				popup.add(mi);
				popup.add(new SaveTierAction(null));
				popup.show(selectionList, p.x, p.y);
			} else if (sel instanceof Segmentation) {
				final Segmentation tier = (Segmentation) sel;
				JPopupMenu popup = new JPopupMenu();
				JMenuItem mi = new JMenuItem(new SaveTierAction(tier));
				popup.add(mi);
				popup.add(new SaveTierAction(null));
				popup.show(selectionList, p.x, p.y);
			} else if (!selectionModel.isEmpty()) {
				JPopupMenu popup = new JPopupMenu();
				popup.add(new SaveTierAction(null));
				popup.show(selectionList, p.x, p.y);
			}
		}
	}
	
	/**
	 * Prompts the user to specify a location where to store the tier/selections.
	 * 
	 * @return the path or null if canceled
	 */
	private String promptForTierFile() {
		ArrayList<String[]> extensions = new ArrayList<String[]>();
		extensions.add(FileExtension.CSV_EXT);

		FileChooser chooser = new FileChooser(this);
		chooser.createAndShowFileDialog(null, FileChooser.SAVE_DIALOG, extensions,FileExtension.XML_EXT, "Recognizer.Path", null);
		
		File f = chooser.getSelectedFile();
		if (f != null) {			
			return f.getAbsolutePath();
		} 
		
		return null;
	}
	
	/**
	 * 
	 * @return an ArrayList with recognizer.data.RSelection Objects
	 */
	public ArrayList<RSelection> getSelections() {
		ArrayList<RSelection> selectionObjects = new ArrayList<RSelection>();
		
		Object iter;
		boolean needsSorting = false;
		
		for (int i = 0; i < selectionModel.getSize(); i++) {
			iter = selectionModel.get(i);
			if (iter instanceof RSelection) {
				selectionObjects.add((RSelection) iter);
			} else if (iter instanceof Segmentation) {
				selectionObjects.addAll(((Segmentation) iter).getSegments());
				needsSorting = true;
			}
		}
		// sort the list
		if (needsSorting && selectionObjects.size() > 1) {
			Collections.sort(selectionObjects, new SelectionComparator());
		}
		
		return selectionObjects;
	}
	
	
	/**
	 * Returns the selections without "flattening" groups (tiers). 
	 * The list can contain both RSelection objects and Segmentation objects.
	 *   
	 * @return a list of selections and/or segmentation objects
	 */
	public ArrayList<Object> getTiersAndSelections() {
		ArrayList<Object> copy = new ArrayList<Object>(selectionModel.getSize());
		Object iter;
		for (int i = 0; i < selectionModel.getSize(); i++) {
			iter = selectionModel.get(i);
			copy.add(iter);
		}
		
		return copy;
	}
	
	/**
	 * Returns whether there is at least one selection/segment.
	 * 
	 * @return true if there is at least one selection, false otherwise 
	 */
	public boolean hasSelections() {
		int numSels = 0;
		Object iter;
		
		for (int i = 0; i < selectionModel.getSize(); i++) {
			iter = selectionModel.get(i);
			if (iter instanceof RSelection) {
				numSels ++;
				break;
			} else if (iter instanceof Segmentation) {
				if (((Segmentation) iter).getSegments().size() > 0) {
					numSels++;
					break;
				}
			}
		}
		
		return numSels > 0;
	}
	
	/**
	 * Clears the list of selections.
	 */
	public void clearSelections() {
		if (!selectionModel.isEmpty()) {
			selectionModel.clear();
		}
	}
	
	@Override
	public void setEnabled(boolean enabled) {
		super.setEnabled(enabled);
		if (addSelection != null) {
			addSelection.setEnabled(enabled);
		}
		if (addSelection1 != null) {
			addSelection1.setEnabled(enabled);
		}
		if (addSelection2 != null) {
			addSelection2.setEnabled(enabled);
		}
		if (addFromTier != null) {
			addFromTier.setEnabled(enabled);
		}
		if (addFromTier2 != null) {
			addFromTier2.setEnabled(enabled);
		}
		if (removeSelection != null) {
			removeSelection.setEnabled(enabled);
		}
	}


	/**
	 * Mouse handler class.
	 * 
	 * @author Han Sloetjes
	 */
	public class MouseHandler extends MouseAdapter {
		public void mouseClicked(MouseEvent event) {
			// set the selection to the double clicked entry in the JList
			if (event.getClickCount() > 1) {
				Object o = selectionList.getSelectedValue();
				if (o instanceof RSelection) {
					long bt = ((RSelection) o).beginTime;
					long et = ((RSelection) o).endTime;
					
					selection.setSelection(bt, et);
					viewerManager.getMasterMediaPlayer().setMediaTime(bt);
				}
			}
		}
		
		public void mousePressed(MouseEvent e) {
			// right mouse popup handling
	        if ((SwingUtilities.isRightMouseButton(e) && (e.getButton() == MouseEvent.BUTTON1 ^ e.isMetaDown())) 
	                || e.isPopupTrigger()) {
	        	//int row = selectionList.locationToIndex(e.getPoint());
	        	handlePopUp(e.getPoint());
	        }
		}
	}
	
	/** 
	 * An action to set/change the label of a segment.
	 * 
	 * @author Han Sloetjes
	 */
	class LabelAction extends AbstractAction {
		private Segment segment;

		/**
		 * Constructor with a segment as an argument.
		 * 
		 * @param segment the segment to set the label for
		 */
		public LabelAction(Segment segment) {
			super(ElanLocale.getString("Recognizer.SelectionsPanel.SetLabel"));
			this.segment = segment;
		}

		/**
		 * Shows an input dialog for this segment.
		 */
		public void actionPerformed(ActionEvent e) {
			String initial = "";
			if (segment.label != null) {
				initial = segment.label;
			}
			String val = (String) JOptionPane.showInputDialog(SelectionPanel.this, 
					ElanLocale.getString("Recognizer.SelectionsPanel.SetLabelDesc"), 
					ElanLocale.getString("Recognizer.SelectionsPanel.SetLabel"),
					JOptionPane.PLAIN_MESSAGE, null, null,
					initial);			
			if (val != null) {//null means input dialog canceled
				segment.label = val; //can be empty
			}			
		}		
	}
	
	/**
	 * An action to set/change a region of interest in a video stream.
	 * 
	 * @author Han Sloetjes
	 */
	class ShapeAction extends AbstractAction {
		private VideoSegment segment;
		
		public ShapeAction(VideoSegment segment) {
			super(ElanLocale.getString("Recognizer.SelectionsPanel.SetShape"));
			this.segment = segment;
		}
		
		public void actionPerformed(ActionEvent e) {
			if (mediaFiles == null || mediaFiles.size() == 0) {
				// show warning, no media file
				JOptionPane.showMessageDialog(SelectionPanel.this, 
						ElanLocale.getString("Recognizer.SelectionsPanel.WarnNoMedia"), 
						ElanLocale.getString("Message.Warning"), JOptionPane.WARNING_MESSAGE);
				return;
			}
			
			String mediaPathToUse = null;
			if (mediaFiles.size() > 1) {
				String[] options = new String[mediaFiles.size()];
				String p;
				for (int i = 0; i < mediaFiles.size(); i++) {
					p = mediaFiles.get(i);
					int index = p.lastIndexOf('/');// assume forward slashes
					if (index > -1 && index < p.length() - 1) {
						p = p.substring(index + 1);
					}
					options[i] = p;
				}
				String sel = (String) JOptionPane.showInputDialog(SelectionPanel.this, 
						ElanLocale.getString("Recognizer.SelectionsPanel.SelectMedia"), "", 
						JOptionPane.QUESTION_MESSAGE, null, options, options[0]);
				if (sel != null) {
					for (int i = 0; i < options.length; i++) {
						if (options[i].equals(sel)) {
							mediaPathToUse = mediaFiles.get(i);
							break;
						}
					} 
				}
			} else {
				mediaPathToUse = mediaFiles.get(0);
			}
			
			ElanMediaPlayer player = null;
			String url;
			
			if (viewerManager.getMasterMediaPlayer().getMediaDescriptor() != null) {
				url = viewerManager.getMasterMediaPlayer().getMediaDescriptor().mediaURL;
				if (url != null && url.endsWith(mediaPathToUse)) {
					player = viewerManager.getMasterMediaPlayer();
				}
			}
			
			if (player == null) {
				ElanMediaPlayer slayer;
				List slaves = viewerManager.getSlaveMediaPlayers();
				for (int i = 0; i < slaves.size(); i++) {
					slayer = (ElanMediaPlayer) slaves.get(i);
					if (slayer.getMediaDescriptor() != null) {
						url = slayer.getMediaDescriptor().mediaURL;
						if (url != null && url.endsWith(mediaPathToUse)) {
							player = slayer;
							break;
						}
					}
				}
			}
			
			if (player == null) {
				// show warning, no player
				JOptionPane.showMessageDialog(SelectionPanel.this, 
						ElanLocale.getString("Recognizer.SelectionsPanel.WarnNoPlayer"), 
						ElanLocale.getString("Message.Warning"), JOptionPane.WARNING_MESSAGE);
				return;
			}
			
			if (!(player instanceof VideoFrameGrabber)) {
				JOptionPane.showMessageDialog(SelectionPanel.this, 
						ElanLocale.getString("Recognizer.SelectionsPanel.WarnNoGrabber"), 
						ElanLocale.getString("Message.Warning"), JOptionPane.WARNING_MESSAGE);
				return;
			}
			// we have the player
			//System.out.println("Player: " + player.getMediaDescriptor().mediaURL);
			List<Shape> curSh = new ArrayList<Shape> (4);
			if (segment.shape != null) {
				curSh.add(segment.shape);
			}
			
			Graphics2DEditor editor = new Graphics2DEditor(
					ELANCommandFactory.getRootFrame(viewerManager.getTranscription()), (VideoFrameGrabber) player, 
					segment.beginTime, segment.endTime, curSh);
			editor.setVisible(true);
			List<Shape> shapes = editor.getShapes();
			if (shapes != null && shapes.size() > 0) {
				segment.shape = shapes.get(0);
			}
		}
		
	}
	
	class SaveTierAction extends AbstractAction {
		private Segmentation segmentation;

		/**
		 * @param segmentation
		 */
		public SaveTierAction(Segmentation segmentation) {
			super();
			if (segmentation != null) {
				putValue(Action.NAME, ElanLocale.getString("Recognizer.SelectionsPanel.SaveTier"));
			} else {
				putValue(Action.NAME, ElanLocale.getString("Recognizer.SelectionsPanel.SaveSelections"));
			}
			this.segmentation = segmentation;
		}

		public void actionPerformed(ActionEvent e) {		
			String filePath = promptForTierFile();
			File tf = new File(filePath);
			try { 
				if (tf.exists()) {
                    int answer = JOptionPane.showConfirmDialog(SelectionPanel.this,
                            ElanLocale.getString("Message.Overwrite"),
                            ElanLocale.getString("SaveDialog.Message.Title"),
                            JOptionPane.YES_NO_OPTION,
                            JOptionPane.WARNING_MESSAGE);

                    if (answer == JOptionPane.NO_OPTION) {
                        return;
                    }
				}
			} catch (Exception ex) {// any exception
				return;
			}
			
			if (segmentation != null) {
				List<Object> list = new ArrayList<Object>(1);
				list.add(segmentation);
				try {
					RecTierWriter xTierWriter = new RecTierWriter();
					xTierWriter.write(tf, list, false);
				} catch (IOException ioe) {
					// show message
					JOptionPane.showMessageDialog(SelectionPanel.this, ElanLocale.getString(
							"Recognizer.RecognizerPanel.Warning.SaveFailed")  + ioe.getMessage(), 
							ElanLocale.getString("Message.Warning"), JOptionPane.ERROR_MESSAGE);
				}
			} else if (!selectionModel.isEmpty()) {
				try {
					RecTierWriter xTierWriter = new RecTierWriter();
					xTierWriter.write(tf, getTiersAndSelections(), true);
				} catch (IOException ioe) {
					// show message
					JOptionPane.showMessageDialog(SelectionPanel.this, ElanLocale.getString(
							"Recognizer.RecognizerPanel.Warning.SaveFailed")  + ioe.getMessage(), 
							ElanLocale.getString("Message.Warning"), JOptionPane.ERROR_MESSAGE);
				}
			}
		}
		
		
	}
}