package mpi.eudico.client.annotator.recognizer.io;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import mpi.eudico.client.annotator.recognizer.data.FileParam;
import mpi.eudico.client.annotator.recognizer.data.NumParam;
import mpi.eudico.client.annotator.recognizer.data.Param;
import mpi.eudico.client.annotator.recognizer.data.TextParam;
import mpi.eudico.client.annotator.util.ClientLogger;

/**
 * A class for reading and writing of (AVATecH project specific) parameter in XML format.
 *  
 * @author Han Sloetjes
 *
 */
public class ParamIO {

	public ParamIO() {
	}

	/**
	 * Reads parameters from the specified file.
	 * 
	 * @param f the parameter file
	 * @return a Map of parameter name-value pairs or null if no parameters were found
	 * 
	 * @throws IOException if reading or parsing fails
	 */
	public Map<String, Object> read(File f) throws IOException {
		if (f == null) {
			new IOException("Cannot read from file: file is null");
		}
		
		try {
			XMLReader reader = XMLReaderFactory.createXMLReader(
	    		"org.apache.xerces.parsers.SAXParser");
			ParamHandler ph = new ParamHandler(); 
			reader.setContentHandler(ph);

			reader.parse(f.getAbsolutePath());
			
			return ph.getParamMap();
		} catch (SAXException sax) {
			ClientLogger.LOG.warning("Parsing failed: " + sax.getMessage());
			throw new IOException(sax.getMessage());
		}
	}
	
	/**
	 * Writes the parameters from the specified list to the specified XML file.
	 * 
	 * @param paramList a list of parameters
	 * @param f the destination file
	 * 
	 * @throws IOException if writing fails 
	 */
	public void write(List<Param> paramList, File f) throws IOException {
		if (f == null) {
			new IOException("Cannot write to file: file is null");
		}
		
		PrintWriter writer = new PrintWriter(new BufferedWriter(new OutputStreamWriter(
				new FileOutputStream(f), "UTF-8")));// utf-8 is always supported, I guess
		writer.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		writer.println("<PARAM xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"file:avatech-call.xsd\">");
		
		if (paramList != null && paramList.size() > 0) {

			for (Param p : paramList) {
				writer.print("<param name=\"" + p.id + "\">");
				if (p instanceof NumParam) {
					writer.print(((NumParam) p).current);
				} else if (p instanceof TextParam) {
					writer.print(((TextParam) p).curValue);
				} else if (p instanceof FileParam) {
					String path = ((FileParam) p).filePath;
					// in case of file protocol strip the protocol part
					if (path.startsWith("file:")) {
						path = path.substring(5);
					}
					writer.print(path);
				}					
				writer.println("</param>");
			}
		}
		
		writer.println("</PARAM>");
		
		writer.close();
	}
	
	/**
	 * Writes the parameters from the specified map to the specified XML file.
	 * 
	 * @param paramMap a mapping of parameter key-value pairs
	 * @param f the destination file
	 * 
	 * @throws IOException if writing fails 
	 */
	public void write(Map<String, Object> paramMap, File f) throws IOException {
		if (f == null) {
			new IOException("Cannot write to file: file is null");
		}
		
		PrintWriter writer = new PrintWriter(new BufferedWriter(new OutputStreamWriter(
				new FileOutputStream(f), "UTF-8")));// utf-8 is always supported, I guess
		writer.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		writer.println("<PARAM xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"file:avatech-call.xsd\">");
		
		if (paramMap != null && paramMap.size() > 0) {
			String key = null;
			Object val = null;
			
			Iterator<String> parIt = paramMap.keySet().iterator();
			while (parIt.hasNext()) {
				key = parIt.next();
				val = paramMap.get(key);
				
				writer.print("<param name=\"" + key + "\">");
				if (val instanceof String) {
					writer.print((String) val);
				} else if (val instanceof Float) {
					writer.print((Float) val);
				} else if (val instanceof Double) {
					writer.print((Double) val);
				}
				
				writer.println("</param>");
			}
		}
		
		writer.println("</PARAM>");
		
		writer.close();
	}
	
	// ###############  Parser content handler  #############################################
	class ParamHandler implements ContentHandler {
		private String curContent = "";
		private String curName = null;
		
		private final String PARAM = "param";
		private final String NAME = "name";
		private Map<String, Object> params;
		
		/**
		 * Constructor.
		 */
		public ParamHandler() {
			super();
			params= new HashMap<String, Object>();
		}

		/**
		 * Returns the map of parameters.
		 * 
		 * @return the map of parameters
		 */
		public Map<String, Object> getParamMap() {
			return params;
		}

		public void characters(char[] ch, int start, int length)
				throws SAXException {
			curContent += new String(ch, start, length);			
		}

		public void startElement(String nameSpaceURI, String name,
	            String rawName, Attributes attributes) throws SAXException {
			if (name.equals(PARAM)) {
				curName = attributes.getValue(NAME);
			}
			
		}

		public void endElement(String nameSpaceURI, String name, String rawName)
				throws SAXException {
			if (name.equals(PARAM)) {
				curContent = curContent.trim();
				if (curName != null) {
					params.put(curName, curContent);
				}
				curContent = "";
			}			
		}

		public void endDocument() throws SAXException {			
		}
		
		public void endPrefixMapping(String arg0) throws SAXException {
		}

		public void ignorableWhitespace(char[] arg0, int arg1, int arg2)
				throws SAXException {	
		}

		public void processingInstruction(String arg0, String arg1)
				throws SAXException {
		}

		public void setDocumentLocator(Locator arg0) {
		}

		public void skippedEntity(String arg0) throws SAXException {
		}

		public void startDocument() throws SAXException {
		}

		public void startPrefixMapping(String arg0, String arg1)
				throws SAXException {
		}
		
	}
}
