package mpi.eudico.client.annotator.tier;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeSet;
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.gui.multistep.MultiStepPane;
import mpi.eudico.client.annotator.gui.multistep.StepPane;
import mpi.eudico.client.annotator.util.ClientLogger;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;
import mpi.eudico.server.corpora.clomimpl.type.Constraint;
import mpi.eudico.server.corpora.clomimpl.type.LinguisticType;

/**
 * Abstract Step pane to specify the attributes for the new tier.
 * Allows to specify a tierName, linguistic type and the parent tier
 * for the new tier
 * 
 * @author Jeffrey Lemein
 * @updatedBy aarsom
 * @version November, 2011
 */
public abstract class AbstractDestTierAndTypeSpecStepPane extends StepPane implements ListSelectionListener, ActionListener, KeyListener, ClientLogger{
	
	
	private JLabel newTierNameLabel, decideRootOrChildLabel, lingTypeLabel;
	private JScrollPane lingTypeTableScrollPane;
	private JPanel destinationTierConfigurationPanel;
	private ButtonGroup rootChildButtonGroup;
	private JTextField newTierNameField;	
	private JTable linguisticTypeTable;
	private Insets globalInset, singleTabInset;
	private TreeSet<String> tierSet;
	
	protected JRadioButton rootTierRB, childTierRB;
	protected JComboBox parentTierCB;
	
	private TranscriptionImpl transcription;
	public List<String> rootTierTypeNames;	
	public List<String> childTierTypeNames;	
	
	/**
	 * Constructor
	 */
	public AbstractDestTierAndTypeSpecStepPane(MultiStepPane mp, TranscriptionImpl trans){
		super(mp);
		globalInset = new Insets(5, 10, 5, 10);
		singleTabInset = new Insets(0, 30, 0, 10);
		transcription = trans;
		initComponents();
	}	

	protected void initComponents(){			
		newTierNameLabel = new JLabel( ElanLocale.getString("DestTierAndType.Label.newTierNameLabel") );
		lingTypeLabel = new JLabel( ElanLocale.getString("DestTierAndType.Label.lingTypeLabel") );
		
		//create text field
		newTierNameField = new JTextField();
		newTierNameField.addKeyListener(this);
		
		//create parent tier combobox + checkbox
		decideRootOrChildLabel = new JLabel(ElanLocale.getString("DestTierAndType.Label.RootChildLabel"));
		rootTierRB = new JRadioButton(ElanLocale.getString("DestTierAndType.Radio.RootTier"), true);
		rootTierRB.addActionListener(this);
		childTierRB = new JRadioButton(ElanLocale.getString("DestTierAndType.Radio.ChildTier"));
		childTierRB.addActionListener(this);
		
		rootChildButtonGroup = new ButtonGroup();
		rootChildButtonGroup.add(rootTierRB);
		rootChildButtonGroup.add(childTierRB);		
				
		parentTierCB = new JComboBox();	
		
		//create table
		linguisticTypeTable = new JTable();
		linguisticTypeTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		linguisticTypeTable.setModel(new LinguisticTypeTableModel());
		linguisticTypeTable.getColumnModel().getColumn(0).setHeaderValue( ElanLocale.getString("DestTierAndType.Column.LinguisticType") );
		linguisticTypeTable.getColumnModel().getColumn(1).setHeaderValue( ElanLocale.getString("DestTierAndType.Column.Stereotype") );
		linguisticTypeTable.getSelectionModel().addListSelectionListener(this);
		lingTypeTableScrollPane = new JScrollPane(linguisticTypeTable);
		
		//layout components
		destinationTierConfigurationPanel = new JPanel(new GridBagLayout());
		destinationTierConfigurationPanel.setBorder(new TitledBorder(ElanLocale.getString("DestTierAndType.Panel.Title.DestTierConfiguration")));
		
		GridBagConstraints gbc = new GridBagConstraints();
		gbc.gridx = 0;
		gbc.gridy = 0;
		gbc.gridwidth = 2;
		gbc.gridheight = 1;
		gbc.anchor = GridBagConstraints.WEST;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.weightx = 1.0;
		gbc.weighty = 0.0;
		gbc.insets = globalInset;
		destinationTierConfigurationPanel.add(newTierNameLabel, gbc);
		
		gbc.gridy = 1;
		destinationTierConfigurationPanel.add(newTierNameField, gbc);
		
		gbc.gridy = 2;
		destinationTierConfigurationPanel.add(decideRootOrChildLabel, gbc);
		
		gbc.gridy = 3;
		gbc.insets = singleTabInset;
		destinationTierConfigurationPanel.add(rootTierRB, gbc);
		
		gbc.gridy = 4;
		gbc.gridwidth = 1;
		gbc.weightx = 0.0;
		destinationTierConfigurationPanel.add(childTierRB, gbc);
		
		gbc.gridx = 1;
		gbc.weightx = 1.0;
		gbc.insets = globalInset;
		destinationTierConfigurationPanel.add(parentTierCB, gbc);
		
		gbc.gridx = 0;
		gbc.gridy = 5;
		gbc.gridwidth = 2;
		destinationTierConfigurationPanel.add(lingTypeLabel, gbc);
		
		gbc.gridy = 6;
		gbc.weighty = 1.0;
		gbc.fill = GridBagConstraints.BOTH;
		destinationTierConfigurationPanel.add(lingTypeTableScrollPane, gbc);
		
		setLayout(new GridBagLayout());
		gbc = new GridBagConstraints();
		gbc.gridx = 0;
		gbc.gridy = 0;
		gbc.weighty = 1.0;
		gbc.weightx = 1.0;
		gbc.gridwidth = 1;
		gbc.gridheight = 1;
		gbc.fill = GridBagConstraints.BOTH;
		gbc.anchor = GridBagConstraints.WEST;
		gbc.insets = globalInset;			
		add(destinationTierConfigurationPanel, gbc);
	}
	
	public abstract String getStepTitle();
	
	public void enterStepForward(){		
		tierSet = (TreeSet<String>) multiPane.getStepProperty("AllTiers");		
		rootTierTypeNames = 	(List<String>) multiPane.getStepProperty("RootTierTypes");
		childTierTypeNames = 	(List<String>) multiPane.getStepProperty("ChildTierTypes");
				
		((LinguisticTypeTableModel)linguisticTypeTable.getModel()).updateLinguisticTypes(null);		
				
		//get selected tiers
		Vector tierList = (Vector) multiPane.getStepProperty("SelectedTiers");
		parentTierCB.removeAllItems();
		
		//add tier names that are selected in tier table in step 1
		for( int i=0; i<tierList.size(); i++ ){
			//TierImpl tier = (TierImpl) transcription.getTierWithId( (String)tierList.get(i) );
			parentTierCB.addItem((String)tierList.get(i));
			//parentTierCB.addItem( tierList.get(i).toString() );
		}			
		updateButtonStates();
	}	

	public void enterStepBackward(){
		updateButtonStates();
	}
	
	public boolean leaveStepForward(){
		String newTierName = newTierNameField.getText().trim();
		
		if( tierSet.contains(newTierName) )
			JOptionPane.showMessageDialog(this, ElanLocale.getString("DestTierAndTypeStepPane.Message1.Part1") +
					newTierName + ElanLocale.getString("DestTierAndType.Message1.Part2"), 
					ElanLocale.getString("DestTierAndType.Message.Title"), JOptionPane.ERROR_MESSAGE);
		else if( newTierName.length() <= 0 )
			JOptionPane.showMessageDialog(this, ElanLocale.getString("DestTierAndType.Message2"), 
					ElanLocale.getString("DestTierAndType.Message.Title"), JOptionPane.ERROR_MESSAGE);
		
		//retrieve name for destination tier
		multiPane.putStepProperty("DestinationTierName", newTierNameField.getText().trim());		
				
		//retrieve linguistic type
		int selectedRow = linguisticTypeTable.getSelectedRow();
		String linguisticType;
				
		linguisticType = (String)linguisticTypeTable.getModel().getValueAt(selectedRow, 0);
		
		multiPane.putStepProperty("linguisticType", linguisticType);
		
		//parent tier	
		String parentTierName = null;
		if( childTierRB.isSelected() ){
			parentTierName = (String) parentTierCB.getSelectedItem();
			//parentTier = transcription.getTierWithId(tierName);
		}				
		multiPane.putStepProperty("ParentTierName", parentTierName);
		
		return true;
	}		
	
	/**
	 * Updates the button states according to some constraints (like everything has to be filled in, consistently)
	 */
	public void updateButtonStates(){		
		parentTierCB.setEnabled(childTierRB.isSelected());
		
		String newTierName = newTierNameField.getText().trim();
		boolean validName = false;
		boolean validType = false;
		String message = null;
		if(newTierName.length() > 0 && !tierSet.contains(newTierName)) {
			validName = true;			
		} else{			
			if(newTierName.length() <= 0){
				message =  ElanLocale.getString("DestTierAndType.Message2");
			}else{
				message =  ElanLocale.getString("DestTierAndTypeStepPane.Message1.Part1") + 
						newTierName + ElanLocale.getString("DestTierAndType.Message1.Part2");
			}
		}
		
		if(validName){
			if(linguisticTypeTable.getSelectedRowCount() > 0){
				validType = true;
			}else{
				message = ElanLocale.getString("DestTierAndType.Message3");
			}
		}
		
		multiPane.setButtonToolTipText(MultiStepPane.NEXT_BUTTON, message);
		
		multiPane.setButtonEnabled(MultiStepPane.NEXT_BUTTON, validType && validName);
		multiPane.setButtonEnabled(MultiStepPane.PREVIOUS_BUTTON, true);
	}

	/**
	 * Value changed event handler that updates button states when new selection is made
	 */
	public void valueChanged(ListSelectionEvent arg0) {
		updateButtonStates();	
	}

	/**
	 * Not used
	 */
	public void keyPressed(KeyEvent e) {}

	/**
	 * Key release handler that updates the button states
	 */
	public void keyReleased(KeyEvent e) {
		updateButtonStates();
	}

	/**
	 * Key typed handler that updates the button states
	 */
	public void keyTyped(KeyEvent e) {
		updateButtonStates();
	}
	
	//action listener for parent tier checkbox and parent tier combobox
	public void actionPerformed(ActionEvent e) {
		//because a checkbox is selected, update the possible linguistic types
		String parentTierName = null;
		LinguisticTypeTableModel model = (LinguisticTypeTableModel) linguisticTypeTable.getModel();
		
		if( childTierRB.isSelected() )
			parentTierName = (String)parentTierCB.getSelectedItem();
		
		model.updateLinguisticTypes(parentTierName);
		
		updateButtonStates();
	}
	
	/**
	 * Table Model for  linguistic types
	 * @author Jeffrey Lemein
	 * @version July, 2010
	 */
	private class LinguisticTypeTableModel extends AbstractTableModel{
		private String[] columnNames;
		private String[][] rowData;
		private boolean multipleFileMode;
		
		/**
		 * Constructor
		 */
		public LinguisticTypeTableModel(){
			columnNames = new String[2];
			columnNames[0] = new String( ElanLocale.getString("DestTierAndType.Column.LinguisticType") );
			columnNames[1] = new String(ElanLocale.getString("DestTierAndType.Column.Stereotype") );
			
			multipleFileMode = transcription == null;
			updateLinguisticTypes(null);
		}
		
		/**
		 * Adds the linguistic types to the linguistic type table (by looking at the parent tier name and ...)
		 * @param parentTierName
		 */
		public void updateLinguisticTypes(String parentTierName){	
			List<String> linguisticTypeNames = new ArrayList<String>();
			String stereotypeName = null;			
			//SINGLE FILE MODE
			if( !multipleFileMode ){
				//only one file is used
				List<LinguisticType> linguisticTypeList = transcription.getLinguisticTypes();
				if( parentTierName == null ){
					//to be created tier will be root tier (so only add top-level linguistic types)
					for( LinguisticType lt : linguisticTypeList )
						if( !lt.hasConstraints() && !lt.isUsingControlledVocabulary()  ){
							linguisticTypeNames.add(lt.getLinguisticTypeName());							
						}
					stereotypeName = "";
				} else{
					//to be created tier has a parent tier (so included_in, top-level linguistic type (if parent has same linguistic type))
					TierImpl parentTier = (TierImpl) transcription.getTierWithId(parentTierName);
					
					for( LinguisticType lt : linguisticTypeList ){
						//don't add linguistic types that are using controlled vocabulary
						if( lt.isUsingControlledVocabulary() )
							continue;
						
						//add linguistic type if it equals included_in
						if( lt.hasConstraints() && lt.getConstraints().getStereoType() == Constraint.INCLUDED_IN ){
							linguisticTypeNames.add(lt.getLinguisticTypeName());							
							//JOptionPane.showMessageDialog(null, lt.getLinguisticTypeName() + " added");
							continue;
						}
						stereotypeName = Constraint.stereoTypes[ Constraint.INCLUDED_IN ];
						
						//and if parent tier has top level linguistic type, add this type too
						/**if( !lt.hasConstraints() ){
							linguisticTypeNames.add(lt.getLinguisticTypeName());
							stereotypeNames.add("");
							//JOptionPane.showMessageDialog(null, lt.getLinguisticTypeName() + " added");
							continue;
						}	**/						
					}
				}
			//MULTIPLE FILE MODE
			}else{
				//we work with multiple transcription/files
			
				//NEW ROOT TIER
				if( parentTierName == null && rootTierTypeNames != null){
					linguisticTypeNames.addAll(rootTierTypeNames);
					stereotypeName = "";
				//NEW CHILD TIER
				}else{
					//a parent tier is selected, so add included_in
					if(childTierTypeNames != null){
						linguisticTypeNames.addAll(childTierTypeNames);
						stereotypeName = Constraint.stereoTypes[ Constraint.INCLUDED_IN ];
					}
				}
			}
			
			//add the linguistic type with its stereotype
			final int SIZE = linguisticTypeNames.size();
			rowData = new String[SIZE][];
			for( int i=0; i<SIZE; i++ )
				rowData[i] = new String[]{ linguisticTypeNames.get(i), stereotypeName };			
				
			fireTableDataChanged();
		}
		
		public int getColumnCount() {
			return columnNames.length;
		}

		public int getRowCount() {
			if( rowData != null )
				return rowData.length;
			else
				return 0;
		}

		public Object getValueAt(int row, int col) {
			return rowData[row][col];
		}		
	}
}