package mpi.eudico.client.annotator.viewer;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.InlineEditBoxListener;
import mpi.eudico.client.annotator.Preferences;
import mpi.eudico.client.annotator.commands.Command;
import mpi.eudico.client.annotator.commands.ELANCommandFactory;
import mpi.eudico.client.annotator.gui.EditTierDialog;
import mpi.eudico.client.annotator.gui.InlineEditBox;
import mpi.eudico.client.annotator.util.AnnotationTransfer;
import mpi.eudico.client.annotator.util.DragTag2D;
import mpi.eudico.client.annotator.util.SystemReporting;
import mpi.eudico.client.annotator.util.Tag2D;
import mpi.eudico.client.annotator.util.Tier2D;

import mpi.eudico.client.mediacontrol.ControllerEvent;
import mpi.eudico.client.mediacontrol.StartEvent;
import mpi.eudico.client.mediacontrol.StopEvent;
import mpi.eudico.client.mediacontrol.TimeEvent;
import mpi.eudico.client.util.TierAssociation;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.ExternalReference;
import mpi.eudico.server.corpora.clom.Tier;
import mpi.eudico.server.corpora.clom.TimeSlot;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.AbstractAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.AlignableAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;
import mpi.eudico.server.corpora.clomimpl.type.Constraint;
import mpi.eudico.server.corpora.clomimpl.type.LinguisticType;
import mpi.eudico.server.corpora.event.ACMEditEvent;
import mpi.eudico.server.corpora.event.ACMEditListener;

import mpi.eudico.util.CVEntry;
import mpi.eudico.util.ControlledVocabulary;
import mpi.eudico.util.ExternalCV;
import mpi.eudico.util.ExternalCVEntry;
import mpi.eudico.util.TimeFormatter;

import java.awt.AlphaComposite;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.LayoutManager2;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.image.BufferedImage;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
/* Mod by Mark */
import java.util.Map;
/* --- END --- */
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollBar;
import javax.swing.JSlider;
import javax.swing.JToolTip;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.UIManager;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;


/**
 * This viewer shows annotations of multiple tiers relative to a time scale.<br>
 * The value of each tag is truncated in such a way that it does not extent
 * beyond the available space at a given resolution.
 *
 * @author Han Sloetjes
 * @version 0.1 2/7/2003
 * @version Aug 2005 Identity removed
 * @version June 2008 msPerPixel changed to float, new zooming functions
 */
public class TimeLineViewer extends TimeScaleBasedViewer
    implements ComponentListener, MouseListener, MouseMotionListener,
        MouseWheelListener, KeyListener, AdjustmentListener, ActionListener,
        MultiTierViewer, ACMEditListener, ChangeListener, InlineEditBoxListener {
    /** default number of pixels that represents one second */
    static final int PIXELS_FOR_SECOND = 100;
    private TranscriptionImpl transcription;
    private boolean timeRulerVisible;
    private int rulerHeight;
    private TimeRuler ruler;
    private Font font;
    private Font tooltipFont;
    private FontMetrics metrics;
    private BufferedImage bi;
    private Graphics2D big2d;
    private AlphaComposite alpha04;
    private AlphaComposite alpha07;
    private BasicStroke stroke;
    private BasicStroke stroke2;
    private Color dropHighLlightColor;
    //private BasicStroke stroke3;
    private HashMap prefTierFonts;
    private Map<String,Color> highlightColors;
    private float msPerPixel;
    
    /** default value of milliseconds per pixel */
    public final int DEFAULT_MS_PER_PIXEL = 10;

    /**
     * The resolution in number of pixels for a second. This is not a
     * percentage value. Historically resolution = PIXELS_FOR_SECOND  factor,
     * where factor = 100 / menu_resolution_percentage_value.
     */
    private int resolution;
    private int imageWidth;
    private int imageHeight;
    private long crossHairTime;
    private int crossHairPos;
    private long intervalBeginTime;
    private long intervalEndTime;
    private int verticalScrollOffset;
    private long selectionBeginTime;
    private long selectionEndTime;
    private int selectionBeginPos;
    private int selectionEndPos;
    private long dragStartTime;
    private Point dragStartPoint;
    private Point dragEndPoint;
    private long splitTime;
    private Color symAnnColor = Constants.SHAREDCOLOR1;

    /** width of border area where auto scrolling starts */
    public final int SCROLL_OFFSET = 16;
    private DragScroller scroller;
    private JPopupMenu popup;
    private ButtonGroup zoomBG;
    private JMenu zoomMI;
    private JRadioButtonMenuItem customZoomMI;
    private ButtonGroup fontSizeBG;
    private JMenu fontMenu;
    private JCheckBoxMenuItem timeScaleConMI;
    private JCheckBoxMenuItem activeAnnStrokeBoldMI;
    private JCheckBoxMenuItem hScrollBarVisMI;
    private JCheckBoxMenuItem tickerModeMI;
    private JCheckBoxMenuItem timeRulerVisMI;
    private JCheckBoxMenuItem reducedTierHeightMI;

    // menu items that can be enabled / disabled
    private JMenuItem newAnnoMI;
    private JMenuItem newAnnoBeforeMI;
    private JMenuItem newAnnoAfterMI;
    private JMenuItem modifyAnnoMI;
    private JMenuItem modifyAnnoDCMI;
    private JMenuItem mergeAnnoNextMI;
    private JMenuItem mergeAnnoBeforeMI;
    private JMenuItem splitAnnotationMI;
    private JMenuItem modifyAnnoTimeMI;
    private JMenuItem deleteAnnoValueMI;
    private JMenuItem deleteAnnoMI;
    private JMenuItem activeTierMI;
    private JMenuItem deleteTierMI;
    private JMenuItem changeTierMI;
    private JMenuItem zoomSelectionMI;
    private JMenuItem deleteSelAnnosMI;
    // copy / paste menu items
    private JMenuItem copyAnnoMI;
    private JMenuItem copyAnnoTreeMI;
    private JMenuItem pasteAnnoHereMI;
    private JMenuItem pasteAnnoTreeHereMI;
    private JMenuItem shiftActiveAnnMI;
    
    private boolean timeScaleConnected;
    private boolean panMode;
    private boolean tickerMode = false;

    // do or don't show empty slots on a child tier
    private boolean showEmptySlots;
    private boolean aaStrokeBold;

    /** Holds value of property DOCUMENT ME! */
    protected int pixelsForTierHeight;

    /** Holds value of property DOCUMENT ME! */
    protected int pixelsForTierHeightMargin;

    //new storage fields
    private ArrayList<Tier2D> allTiers;
    private List<Tier2D> visibleTiers;
    private Tag2D hoverTag2D;
    private int hoverTierIndex;
    private List<Tag2D> selectedAnnotations;

    /** Holds value of property DOCUMENT ME! */
    protected Tag2D cursorTag2D;
    private int cursorTierIndex;
    private Tier2D rightClickTier;
    private long rightClickTime;

    // vertical scrolling
    private JScrollBar scrollBar;
    private JScrollBar hScrollBar;
    private boolean hScrollBarVisible = true;
    private JPanel zoomSliderPanel;
    private JSlider zoomSlider;
    private final int ZOOMSLIDER_WIDTH = 100;
    private JPanel corner;

    /** default scrollbar width */
    private final int defBarWidth;
    private int[] tierYPositions;
    private int tooltipFontSize;

    /** ar the control panel that receives the setTierPositions call */
    MultiTierControlPanel multiTierControlPanel;

    // editing
    private InlineEditBox editBox;
    private boolean deselectCommits = false;
    private boolean forceOpenControlledVocabulary = false;
    private DragTag2D dragEditTag2D;
    private boolean dragEditing;
    private Color dragEditColor = Color.green;

    /** Holds value of property DOCUMENT ME! */
    private final int DRAG_EDIT_MARGIN = 8;

    /** Holds value of property DOCUMENT ME! */
    private final int DRAG_EDIT_CENTER = 0;

    /** Holds value of property DOCUMENT ME! */
    private final int DRAG_EDIT_LEFT = 1;

    /** Holds value of property DOCUMENT ME! */
    private final int DRAG_EDIT_RIGHT = 2;
    private final int DRAG_EDIT_UP = 3;
    private final int DRAG_EDIT_DOWN = 4;
    private int dragEditMode = 0;
    
    private boolean clearSelOnSingleClick= true;

    // the parent's boundaries
    private long dragParentBegin = -1L;
    private long dragParentEnd = -1L;

    private boolean reducedTierHeight = false;
    private int rhDist = 3;
    
    // a flag for the scroll thread
    /** Holds value of property DOCUMENT ME! */
    boolean stopScrolling = true;
    private Object tierLock = new Object();
    /** a flag to decide whether to use a BufferedImage or not. This is always advised but 
     * leads to strange painting artifacts on some systems (XP/Vista, jre version and graphics
     * hardware/driver may play a role) */
    private boolean useBufferedImage = false;
    /** a flag to decide which painting strategy is to be used. The call to playerIsPlaying 
     * isn't always consistent between frameworks */
    private boolean isPlaying = false;
        
    private boolean centerAnnotation = true;
    private boolean delayedAnnotationActivation = false;
    private boolean activateNewAnnotation = true;
    private Annotation lastCreatedAnnotation;

    /**
     * Constructs a new TimeLineViewer.<br>
     * Takes care of some one time initialization and adds listeners.
     */
    public TimeLineViewer() {
        initViewer();
        initTiers();
        defBarWidth = getDefaultBarWidth() + 2;
        addComponentListener(this);
        addMouseListener(this);
        addMouseMotionListener(this);
        addMouseWheelListener(this);
        addKeyListener(this);
        setOpaque(true);
        setDoubleBuffered(true);
        String bufImg = System.getProperty("useBufferedImage");
        if (bufImg != null && bufImg.toLowerCase().equals("true")) {
        	useBufferedImage = true;
        }
//        useBufferedImage = false;
        System.out.println("TL use buffered image: " + useBufferedImage);
    }

    /**
     * Constructs a new TimeLineViewer using the specified transcription.<br>
     * Calls the no-arg constructor first.
     *
     * @param transcription the transcription containing the data for the
     *        viewer
     */
    public TimeLineViewer(Transcription transcription) {
        this();
        this.transcription = (TranscriptionImpl) transcription;
        paintBuffer();
        initTiers(); 
    }

    /**
     * Overrides <code>JComponent</code>'s processKeyBinding by always
     * returning false. Necessary for the proper working of (menu) shortcuts
     * in Elan.
     *
     * @param ks DOCUMENT ME!
     * @param e DOCUMENT ME!
     * @param condition DOCUMENT ME!
     * @param pressed DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected boolean processKeyBinding(KeyStroke ks, KeyEvent e,
        int condition, boolean pressed) {
        return false;
    }

    /**
     * Performs the initialization of fields and sets up the viewer.<br>
     */
    private void initViewer() {
        font = Constants.DEFAULTFONT;
        setFont(font);
        metrics = getFontMetrics(font);
        tooltipFontSize = getDefaultTooltipFontSize();
        tooltipFont = font.deriveFont((float) tooltipFontSize);
        prefTierFonts = new HashMap();
        selectedAnnotations = new ArrayList<Tag2D>(10);
        
        // Keep the tool tip showing
        int dismissDelay = Integer.MAX_VALUE;
        ToolTipManager.sharedInstance().setDismissDelay(dismissDelay);

        timeRulerVisible = true;
        ruler = new TimeRuler(font, TimeFormatter.toString(0));
        rulerHeight = ruler.getHeight();
        stroke = new BasicStroke();
        stroke2 = new BasicStroke(2.0f);
        //stroke3 = new BasicStroke(3.0f);
        dropHighLlightColor = new Color(255, 255, 255, 192);
        msPerPixel = 10;
        resolution = PIXELS_FOR_SECOND;
        crossHairTime = 0L;
        crossHairPos = 0;
        intervalBeginTime = 0L;
        intervalEndTime = 0L;
        verticalScrollOffset = 0;

        selectionBeginTime = 0L;
        selectionEndTime = 0L;
        selectionBeginPos = 0;
        selectionEndPos = 0;
        dragStartTime = 0;
        timeScaleConnected = true;

        imageWidth = 0;
        imageHeight = 0;
        alpha04 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.4f);
        alpha07 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.7f);

        //pixelsForTierHeight = font.getSize() * 3; //hardcoded for now
        //pixelsForTierHeight = font.getSize() + 24;
        pixelsForTierHeight = calcTierHeight();
        pixelsForTierHeightMargin = 2; // hardcoded for now

        scrollBar = new JScrollBar(JScrollBar.VERTICAL, 0, 50, 0, 200);
        scrollBar.setUnitIncrement(pixelsForTierHeight / 2);
        scrollBar.setBlockIncrement(pixelsForTierHeight);
        scrollBar.addAdjustmentListener(this);

        hScrollBar = new JScrollBar(JScrollBar.HORIZONTAL, 0, 50, 0, 400);
        hScrollBar.setUnitIncrement(10);
        hScrollBar.setBlockIncrement(40);
        hScrollBar.addAdjustmentListener(this);

        setLayout(null);
        //setLayout(new TLLayoutManager());
        add(scrollBar);
        add(hScrollBar);
        // the range is from 10% to 1000%, in order to have the button more in the center at 100% use a different max value
        zoomSlider = new JSlider(10, 300, 100);
        zoomSlider.putClientProperty("JComponent.sizeVariant", "small");// On MacOS regular, small, mini
        zoomSliderPanel = new JPanel(null);
        zoomSliderPanel.add(zoomSlider);
        zoomSlider.addChangeListener(this);
        zoomSlider.setToolTipText(String.valueOf(zoomSlider.getValue()));
        add(zoomSliderPanel);
        corner = new JPanel();
        add(corner);

        editBox = new InlineEditBox(true);
        editBox.setFont(font);
        editBox.setVisible(false);
        add(editBox);
        editBox.addInlineEditBoxListener(this);
    }   
    
    /**
     * Called when the viewer will become invisible e.g when switching to an other working mode.
     */
    public void isClosing(){
    	if(editBox != null && editBox.isVisible()){
    		Object val = Preferences.get("InlineEdit.DeselectCommits", null);
    		if (val instanceof Boolean && (Boolean)val) {
    			editBox.commitEdit();
    		} else {
    			editBox.cancelEdit();
    		}
        }
    }
    
    /**
     * Retrieves the default, platform specific width of a scrollbar.
     *
     * @return the default width, or 20 when not found
     */
    private int getDefaultBarWidth() {
        int width = 20;

        if (UIManager.getDefaults().get("ScrollBar.width") != null) {
            width = ((Integer) (UIManager.getDefaults().get("ScrollBar.width"))).intValue();
        }

        return width;
    }

    /**
     * Initialise tiers and tags.
     */
    private void initTiers() {
        allTiers = new ArrayList<Tier2D>(20);
        visibleTiers = new ArrayList<Tier2D>(allTiers.size());

        if (transcription == null) {
        	tierYPositions = new int[0];
            return;
        }

        extractTiers();
        tierYPositions = new int[allTiers.size()];

        //allTiers is filled, set all tiers visible
        // not neccessary anymore

        /*
           Iterator it = allTiers.iterator();
           while(it.hasNext()) {
               visibleTiers.add(it.next());
           }
         */
    }

    /**
     * Extract all Tiers from the Transcription. Store the information in
     * Tier2D and Tag2D objects.
     */
    private void extractTiers() {
        Tier2D tier2d;

        Iterator tierIter = transcription.getTiers().iterator();

        while (tierIter.hasNext()) {
            TierImpl tier = (TierImpl) tierIter.next();
            tier2d = createTier2D(tier);

            allTiers.add(tier2d);
        }
    }

    private Tier2D createTier2D(TierImpl tier) {
        Tier2D tier2d = new Tier2D(tier);
        Tag2D tag2d;
        int xPos;
        int tagWidth;
        TierImpl markTier = null;
        CVEntry[] entries = null;
        ControlledVocabulary cv = transcription.getControlledVocabulary(
        		tier.getLinguisticType().getControlledVocabylaryName());
        if (cv != null) {
        	entries = cv.getEntries();
        } else {
        	// if this tier has no Controlled Vocabulary associated with it, check whether
        	// there is a Symb. Association dependent tier that does have a CV and use the
        	// colors of these entries for the parent annotations on this tier
        	markTier = TierAssociation.findMarkerTierFor(transcription, tier);
        	if (markTier != null) {
        		cv = transcription.getControlledVocabulary(
        				markTier.getLinguisticType().getControlledVocabylaryName());
        		if (cv != null) {
        			entries = cv.getEntries();
        		}
        	}
        }

        if (entries != null && entries.length == 0) {
        	entries = null;
        }
        
        Annotation a;
        List ch;
        Iterator annotIter = tier.getAnnotations().iterator();

        while (annotIter.hasNext()) {
            a = (Annotation) annotIter.next();

            //System.out.println("Annotation: " + a);
            tag2d = new Tag2D(a);
            xPos = timeToPixels(a.getBeginTimeBoundary());
            tag2d.setX(xPos);
            tagWidth = timeToPixels(a.getEndTimeBoundary()) - xPos;
            tag2d.setWidth(tagWidth);
            tag2d.setTruncatedValue(truncateString(a.getValue(), tagWidth,
                    metrics));
            if (entries != null) {
            	if (markTier == null) {
	            	for (CVEntry e : entries) {
	            		if (e.getValue().equals(a.getValue())) {
	            			tag2d.setColor(e.getPrefColor());
	            			break;
	            		}
	            	}
            	} else {
            		ch = ((AbstractAnnotation)a).getChildrenOnTier(markTier);
            		if (ch.size() >= 1) {// should be 1 max
            			Annotation ma = (Annotation) ch.get(0);
    	            	for (CVEntry e : entries) {
    	            		if (e.getValue().equals(ma.getValue())) {
    	            			tag2d.setColor(e.getPrefColor());
    	            			break;
    	            		}
    	            	}
            		}           		
            	}
            }
            
            tier2d.addTag(tag2d);

            if (a == getActiveAnnotation()) {
                cursorTag2D = tag2d;
            }
        }

        return tier2d;
    }

    /**
     * When the resolution or zoom level of the viewer has been changed the
     * Tag2D x position, width and truncated string value needs to be
     * recalculated.
     */
    private void recalculateTagSizes() {
        Tier2D tier2d;
        Tag2D tag2d;
        int xPos;
        int tagWidth;
        Font tierFont;
        FontMetrics tierMetrics;
        Iterator tierIt = allTiers.iterator();

        while (tierIt.hasNext()) {
            tier2d = (Tier2D) tierIt.next();
            tierFont = getFontForTier(tier2d.getTier());
            tierMetrics = getFontMetrics(tierFont);
            Iterator tagIt = tier2d.getTags();

            while (tagIt.hasNext()) {
                tag2d = (Tag2D) tagIt.next();
                xPos = timeToPixels(tag2d.getBeginTime());
                tag2d.setX(xPos);
                tagWidth = timeToPixels(tag2d.getEndTime()) - xPos;
                tag2d.setWidth(tagWidth);
                tag2d.setTruncatedValue(truncateString(tag2d.getValue(),
                        tagWidth, tierMetrics));
            }
        }
    }

    /**
     * Re-processes the annotations of a tier.<br>
     * Necessary after removal of an unknown number of annotations.
     *
     * @param tier2d the Tier2D
     */
    private void reextractTagsForTier(Tier2D tier2d) {
        if ((transcription == null) || (tier2d == null)) {
            return;
        }

        //int index = transcription.getTiers(userIdentity).indexOf(tier2d.getTier());
        //TierImpl tier = null;
        //if (index > -1) {
        //    tier = (TierImpl) transcription.getTiers(userIdentity).get(index);
        //}
        TierImpl tier = tier2d.getTier();

        if (tier == null) {
            return;
        }
        Font prefFont = getFontForTier(tier);
        FontMetrics tierMetrics = getFontMetrics(prefFont);
        tier2d.getTagsList().clear();

        CVEntry[] entries = null;
        TierImpl markTier = null;
        ControlledVocabulary cv = transcription.getControlledVocabulary(
        		tier.getLinguisticType().getControlledVocabylaryName());
        if (cv != null) {
        	entries = cv.getEntries();
        } else {
        	markTier = TierAssociation.findMarkerTierFor(transcription, tier);
        	if (markTier != null) {
        		cv = transcription.getControlledVocabulary(
        				markTier.getLinguisticType().getControlledVocabylaryName());
        		if (cv != null) {
        			entries = cv.getEntries();
        		}
        	}
        }
        
        Tag2D tag2d;
        int xPos;
        int tagWidth;
        List ch ;
        Iterator annotIter = tier.getAnnotations().iterator();

        while (annotIter.hasNext()) {
            Annotation a = (Annotation) annotIter.next();

            //System.out.println("Annotation: " + a);
            tag2d = new Tag2D(a);
            xPos = timeToPixels(a.getBeginTimeBoundary());
            tag2d.setX(xPos);
            tagWidth = timeToPixels(a.getEndTimeBoundary()) - xPos;
            tag2d.setWidth(tagWidth);
            tag2d.setTruncatedValue(truncateString(a.getValue(), tagWidth,
            		tierMetrics));
            if (entries != null) {
            	if (markTier == null) {
	            	for (CVEntry e : entries) {
	            		if (e.getValue().equals(a.getValue())) {
	            			tag2d.setColor(e.getPrefColor());
	            			break;
	            		}
	            	}
            	} else {
            		ch = ((AbstractAnnotation)a).getChildrenOnTier(markTier);
            		if (ch.size() >= 1) {// should be 1 max
            			Annotation ma = (Annotation) ch.get(0);
    	            	for (CVEntry e : entries) {
    	            		if (e.getValue().equals(ma.getValue())) {
    	            			tag2d.setColor(e.getPrefColor());
    	            			break;
    	            		}
    	            	}
            		}           		
            	}	
            }
            
            tier2d.addTag(tag2d);

            if (a == getActiveAnnotation()) {
                cursorTag2D = tag2d;
            }
        }
    }

    /**
     * Create a truncated String of a tag's value to display in the viewer.
     *
     * @param string the tag's value
     * @param width the available width for the String
     * @param fMetrics the font metrics
     *
     * @return the truncated String
     */
    private String truncateString(String string, int width, FontMetrics fMetrics) {
        String line = string.replace('\n', ' ');

        if (fMetrics != null) {
            int stringWidth = fMetrics.stringWidth(line);

            if (stringWidth > (width - 4)) { // truncate

                int i = 0;
                String s = "";
                int size = line.length();

                while (i < size) {
                    if (fMetrics.stringWidth(s) > (width - 4)) {
                        break;
                    } else {
                        s = s + line.charAt(i++);
                    }
                }

                if (!s.equals("")) {
                    line = s.substring(0, s.length() - 1);
                } else {
                    line = s;
                }
            }
        }

        return line;
    }

    /**
     * Paint to a buffer.<br>
     * First paint the top ruler, next the current selection and finally paint
     * the tags of the visible tiers.
     */
    private void paintBuffer() {
    	if (!useBufferedImage  && /*!playerIsPlaying()*/ !isPlaying) {
    		repaint();
    		return;
    	}
        if ((getWidth() <= 0) || (getHeight() <= 0)) {
            return;
        }
        //long bpBegin = System.currentTimeMillis();
        if (((getWidth() - defBarWidth) != imageWidth) ||
                (imageHeight != ((visibleTiers.size() * pixelsForTierHeight) +
                rulerHeight))) {
            imageWidth = getWidth() - defBarWidth;
            imageHeight = (visibleTiers.size() * pixelsForTierHeight) +
                rulerHeight;

            if ((imageWidth <= 0) || (imageHeight <= 0)) {
                return;
            }

            intervalEndTime = intervalBeginTime +
                (int) (imageWidth * msPerPixel);

            if (timeScaleConnected) {
                setGlobalTimeScaleIntervalEndTime(intervalEndTime);
            }

        }

        if ((bi == null) || (bi.getWidth() < imageWidth) ||
                (bi.getHeight() < imageHeight)) {
            bi = new BufferedImage(imageWidth, imageHeight,
                    BufferedImage.TYPE_INT_RGB);
            big2d = bi.createGraphics();
        }
        
        if (bi.getHeight() > imageHeight) {
        	imageHeight = bi.getHeight();
        }
        if (SystemReporting.antiAliasedText) {
	        big2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
	            RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
        }
        //big2d.setFont(font);
        big2d.setColor(Constants.DEFAULTBACKGROUNDCOLOR);
        big2d.fillRect(0, 0, imageWidth, bi.getHeight());

        // mark the area beyond the media time
        long dur = getMediaDuration();
        int xx = xAt(dur);

        if (intervalEndTime > dur) {
        	if (xx >= 0 && xx <= imageWidth) {
        		big2d.setColor(Color.LIGHT_GRAY);         
        		big2d.drawLine(xx, 0, xx, bi.getHeight());
        	}
        	
        	if (xx <= imageWidth) {
        		xx = xx < 0 ? 0 : xx;
                if (!SystemReporting.isMacOS()) {
                	// this slows down Mac performance enormously, don't know why
                	big2d.setColor(UIManager.getColor("Panel.background"));
                } else {
                	//big2d.setColor(Constants.DEFAULTBACKGROUNDCOLOR);
                }
        		big2d.fillRect((xx < 0 ? 0 : xx + 1), 0, imageWidth - xx, bi.getHeight());
        	}
        }

        big2d.translate(-((int)(intervalBeginTime / msPerPixel)), 0);
        /*paint time ruler */
//        if (timeRulerVisible) {
//	        big2d.setColor(Constants.DEFAULTFOREGROUNDCOLOR);
//	        //big2d.translate(-(intervalBeginTime / msPerPixel), 0.0);	
//	        ruler.paint(big2d, intervalBeginTime, imageWidth, msPerPixel,
//	            SwingConstants.TOP);
//	        big2d.setFont(font);
//        }
        ///end ruler
        // paint a slightly dif. background color for every other tier
        int y = rulerHeight;
        int ax = timeToPixels(intervalBeginTime); // == -translation.x

        big2d.setColor(Constants.LIGHTBACKGROUNDCOLOR);
        if (intervalBeginTime < dur) {
        	int numStripes = Math.max(visibleTiers.size(), imageHeight / pixelsForTierHeight);
	        for (int i = 0; i < numStripes; i++) {
	            if (i % 2 != 0) {
	                big2d.fillRect(ax, y + i * pixelsForTierHeight, xx, pixelsForTierHeight);
	            }
	        }
        }
        //paint selection
        if (selectionBeginPos != selectionEndPos) {
            int beginPos = timeToPixels(getSelectionBeginTime());
            int endPos = timeToPixels(getSelectionEndTime());
            big2d.setColor(Constants.SELECTIONCOLOR);
            big2d.setComposite(alpha04);
            big2d.fillRect(beginPos, 0, (endPos - beginPos), rulerHeight);
            big2d.setComposite(AlphaComposite.Src);
            big2d.fillRect(beginPos, rulerHeight, (endPos - beginPos),
                imageHeight - rulerHeight);
        }
        
        /* Mod by Mark */
        /* Overlay the tier highlight colors */
        if(highlightColors != null) {
	        big2d.setComposite(alpha04);
	        for(int i = 0; i < visibleTiers.size(); i++) {
	        	Color highlightColor = highlightColors.get(((Tier2D)visibleTiers.get(i)).getName());
	        	if(highlightColor != null) {
	        		big2d.setColor(highlightColor);
	        		big2d.fillRect(ax, y + (i * pixelsForTierHeight),
	                        imageWidth - (imageWidth - xx), pixelsForTierHeight);
	        	}
	        }
	        big2d.setComposite(AlphaComposite.Src);
        }
		/* --- END --- */
        // paint tags
        //int x;
        //int w;
        int h = pixelsForTierHeight - (2 * pixelsForTierHeightMargin);

        Tier2D tier2d;
        Tag2D tag2d;
        Font tf = null;

        synchronized (tierLock) {
            Iterator visIt = visibleTiers.iterator();
            
            	int count = 0;
            while (visIt.hasNext()) {
                tier2d = (Tier2D) visIt.next();
                tf = getFontForTier(tier2d.getTier());
                big2d.setFont(tf);
                
                count++;               
                
                if (tier2d.isActive()) {    
                    big2d.setColor(Constants.ACTIVETIERCOLOR);
                    big2d.setComposite(alpha07);
                    big2d.fillRect(ax, y, imageWidth, pixelsForTierHeight);
                    big2d.setComposite(AlphaComposite.Src);
                }

                Iterator tagIt = tier2d.getTags();

                while (tagIt.hasNext()) {
                    tag2d = (Tag2D) tagIt.next();

                    if (tag2d.getEndTime() < intervalBeginTime) {
                        continue; //don't paint
                    } else if (tag2d.getBeginTime() > intervalEndTime) {
                        break; // stop looping this tier
                    }

                    //paint tag at this position
                    paintTag(big2d, tag2d, tag2d.getX(),
                        y + pixelsForTierHeightMargin, tag2d.getWidth(), h);
                }

                y += pixelsForTierHeight;
            }
        }
        // end paint tags
        big2d.setTransform(new AffineTransform()); //reset transform
        big2d.setFont(font);
        /*paint time ruler */       
        if (timeRulerVisible) {
        	big2d.setColor(Constants.DEFAULTBACKGROUNDCOLOR);
        	big2d.fillRect(0, 0, imageWidth, verticalScrollOffset + rulerHeight);
	        big2d.setColor(Constants.SELECTIONCOLOR);
	        big2d.drawLine(0, verticalScrollOffset + rulerHeight, imageWidth, verticalScrollOffset + rulerHeight);
	        big2d.translate(-((int)(intervalBeginTime / msPerPixel)), verticalScrollOffset);
	        //paint selection
	        if (selectionBeginPos != selectionEndPos) {
	            int beginPos = timeToPixels(getSelectionBeginTime());
	            int endPos = timeToPixels(getSelectionEndTime());
	            big2d.setColor(Constants.SELECTIONCOLOR);
	            big2d.setComposite(alpha04);
	            big2d.fillRect(beginPos, 0, (endPos - beginPos), rulerHeight);
	            big2d.setComposite(AlphaComposite.Src);
	        }
	        big2d.setColor(Constants.DEFAULTFOREGROUNDCOLOR);
	        ruler.paint(big2d, intervalBeginTime, imageWidth, msPerPixel,
	            SwingConstants.TOP);
	        //big2d.setFont(font);
	        big2d.setTransform(new AffineTransform()); //reset transform
        }
        ///end ruler
        //big2d.dispose(); // does not work properly in jdk 1.4
        //System.out.println("TL paint: " + (System.currentTimeMillis() - bpBegin));
        repaint();
        //??
//        Toolkit.getDefaultToolkit().sync();
    }

    /**
     * Override <code>JComponent</code>'s paintComponent to paint:<br>
     * - a BufferedImage with a ruler, the selection and the tags<br>
     * - the current selection Tag<br>
     * - the "mouse over" Tag<br>
     * - the time ruler when timeRulerVisible is true<br>
     * - the cursor / crosshair - empty slots - the drag edit tag
     *
     * @param g the graphics object
     */
    public void paintComponent(Graphics g) {
        super.paintComponent(g);

        Graphics2D g2d = (Graphics2D) g;
        if (SystemReporting.antiAliasedText) {
	        g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
	                RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
        }
        
        //synchronized (getTreeLock()) {
	        	
	        if (!useBufferedImage && /*!playerIsPlaying() */!isPlaying) {
	        	paintUnbuffered(g2d);
	        	return;
	        }
	        
	        int h = getHeight();
	        int ww = getWidth();
	        // scrolling related fill
	        g2d.setColor(Constants.DEFAULTBACKGROUNDCOLOR);
	        g2d.fillRect(0, 0, imageWidth, h);
	        
	        // mark the area beyond the media time
	        long dur = getMediaDuration();
	        int xx = xAt(dur);

	        if (intervalEndTime > dur) {
	        	if (xx >= 0 && xx <= ww) {
	        		g2d.setColor(Color.LIGHT_GRAY);         
	        		g2d.drawLine(xx, 0, xx, h);
	        	}
	        	
	        	if (xx <= ww) {
	        		xx = xx < 0 ? 0 : xx;
	                if (!SystemReporting.isMacOS()) {
	                	// this slows down Mac performance enormously, don't know why
	                	g2d.setColor(UIManager.getColor("Panel.background"));
	                } else {
	                	//big2d.setColor(Constants.DEFAULTBACKGROUNDCOLOR);
	                }
	        		g2d.fillRect((xx < 0 ? 0 : xx + 1), 0, ww - xx, h);
	        	}
	        }
	        
	        if (bi != null) {
	            g2d.translate(0, -verticalScrollOffset);
	            //paint selection in the part not occupied by the image
	            if (selectionBeginPos != selectionEndPos && bi.getHeight() < h) {
	                g2d.setColor(Constants.SELECTIONCOLOR);
	                g2d.fillRect(selectionBeginPos, 0, 
	                		(selectionEndPos - selectionBeginPos),
	                    h);
	            }
	            g2d.drawImage(bi, 0, 0, this);
	            g2d.translate(0, verticalScrollOffset);
	        }
	
	        g2d.setFont(font);
	
	        /* don't paint the hoverTag for now
	           if (hoverTag2D != null) {
	               //System.out.println("tag: " + hoverTag2D);
	               int x = xAt(hoverTag2D.getBeginTime());
	               int w = xAt(hoverTag2D.getEndTime()) - x;
	               int y = (rulerHeight + hoverTierIndex * pixelsForTierHeight + pixelsForTierHeightMargin) - verticalScrollOffset;
	               int he = pixelsForTierHeight - 2 * pixelsForTierHeightMargin;
	               paintHoverTag2D(g2d, hoverTag2D, x, y, w, he);
	           }
	         */
	        if ((cursorTag2D != null) &&
	                visibleTiers.contains(cursorTag2D.getTier2D())) {
	            //int x = xAt(cursorTag2D.getBeginTime());
	            //int w = xAt(cursorTag2D.getEndTime()) - x;
	            //int x = (int) ((cursorTag2D.getBeginTime() / msPerPixel) -
	            //    (intervalBeginTime / msPerPixel));
	        	int x =  cursorTag2D.getX() -
	        			(int) (intervalBeginTime / msPerPixel);
	            //int w = (int) ((cursorTag2D.getEndTime() -
	            //    cursorTag2D.getBeginTime()) / msPerPixel);
	            //int w = (int) ((int) Math.ceil(cursorTag2D.getEndTime() / msPerPixel) - x);
	            int w = cursorTag2D.getWidth();
	            int y = (rulerHeight + (cursorTierIndex * pixelsForTierHeight) +
	                pixelsForTierHeightMargin) - verticalScrollOffset;
	            int he = pixelsForTierHeight - (2 * pixelsForTierHeightMargin);
	            paintCursorTag2D(g2d, cursorTag2D, x, y, w, he);
	        }
	
	        //paint empty slots
	        if (showEmptySlots) {
	            for (int i = 0; i < visibleTiers.size(); i++) {
	                TierImpl ti = ((Tier2D) visibleTiers.get(i)).getTier();
	
	                if (ti.getParentTier() == null) {
	                    continue;
	                } else {
	                    if (!ti.isTimeAlignable()) {
	                        int y = (rulerHeight + (i * pixelsForTierHeight) +
	                            pixelsForTierHeightMargin) - verticalScrollOffset;
	                        int he = pixelsForTierHeight -
	                            (2 * pixelsForTierHeightMargin);
	                        paintEmptySlots(g2d, ti, y, he);
	                    }
	                }
	            }
	        }
	
	        // paint the dragEdit annotation
	        if (dragEditTag2D != null) {
	            //long newTime = pixelToTime(dragEditTag2D.getX());
	            //int x = (int) ((dragEditTag2D.getBeginTime() / msPerPixel) -
	            //	(intervalBeginTime / msPerPixel));
	            int x = (int) (dragEditTag2D.getX() -
	                (intervalBeginTime / msPerPixel));
	
	            //int w = (int) ((dragEditTag2D.getEndTime() -
	            //dragEditTag2D.getBeginTime()) / msPerPixel);
	            int w = dragEditTag2D.getWidth();
	            int y = (rulerHeight + (cursorTierIndex * pixelsForTierHeight) +
	                
	                /*(getTierIndexForAnnotation(dragEditTag2D.getAnnotation()) * pixelsForTierHeight) +*/
	                pixelsForTierHeightMargin) - verticalScrollOffset + dragEditTag2D.getY();
	            int he = pixelsForTierHeight - (2 * pixelsForTierHeightMargin);
	            paintDragEditTag2D(g2d, x, y, w, he);
	        }
	
	        paintSelectedAnnotations(g2d);
	        /*
	        if (timeRulerVisible && (bi != null)) {
	            g2d.setClip(0, 0, imageWidth, rulerHeight);
	            g2d.drawImage(bi, 0, 0, this);
	            g2d.setClip(null);
	            g2d.setColor(Constants.SELECTIONCOLOR);
	            g2d.drawLine(0, rulerHeight, imageWidth, rulerHeight);
	        }
	        */
			
	        if ((crossHairPos >= 0) && (crossHairPos <= imageWidth)) {
	            // prevents drawing outside the component on Mac
	            g2d.setColor(Constants.CROSSHAIRCOLOR);
	            g2d.drawLine(crossHairPos, 0, crossHairPos, h);
	        }
        //}// treelock
        //Toolkit.getDefaultToolkit().sync();
    }

    private void paintUnbuffered(Graphics2D g2d) {
    	// from paintBuffer
    	int h = getHeight();
    	int w = getWidth();
        // scrolling related fill
        g2d.setColor(Constants.DEFAULTBACKGROUNDCOLOR);
        g2d.fillRect(0, 0, w, h);
        // selection
        /*
        if (selectionBeginPos != selectionEndPos) {
            g2d.setColor(Constants.SELECTIONCOLOR);
            g2d.fillRect(selectionBeginPos, 0, 
            		(selectionEndPos - selectionBeginPos),
                h);
        }
        */
        // paint a slightly dif. background color for every other tier
        int y = rulerHeight - verticalScrollOffset;
        
        g2d.setColor(Constants.LIGHTBACKGROUNDCOLOR);
        for (int i = 0; i < visibleTiers.size(); i++) {
            if (i % 2 != 0) {
                g2d.fillRect(0, y + i * pixelsForTierHeight, w, pixelsForTierHeight);
            }
        }

        // mark the area beyond the media time
        /*
        if (intervalEndTime > getMediaDuration()) {
        	g2d.setColor(Color.LIGHT_GRAY);
            int xx = xAt(getMediaDuration());
            g2d.drawLine(xx, 0, xx, h);
            if (!SystemReporting.isMacOS()) {
            	g2d.setColor(UIManager.getColor("Panel.background"));
            	g2d.fillRect(xx + 1, 0, w - xx, h);
            }
        }
        */
        // mark the area beyond the media time
        long dur = getMediaDuration();
        int xx = xAt(dur);

        if (intervalEndTime > dur) {
        	if (xx >= 0 && xx <= w) {
        		g2d.setColor(Color.LIGHT_GRAY);         
        		g2d.drawLine(xx, 0, xx, h);
        	}
        	
        	if (xx <= w) {
        		xx = xx < 0 ? 0 : xx;
                if (!SystemReporting.isMacOS()) {
                	// this slows down Mac performance enormously, don't know why
                	g2d.setColor(UIManager.getColor("Panel.background"));
                } else {
                	//big2d.setColor(Constants.DEFAULTBACKGROUNDCOLOR);
                }
        		g2d.fillRect((xx < 0 ? 0 : xx + 1), 0, w - xx, h);
        	}
        }
        //paint selection
        if (selectionBeginPos != selectionEndPos) {
            g2d.setColor(Constants.SELECTIONCOLOR);
            g2d.setComposite(alpha04);
            g2d.fillRect(selectionBeginPos, 0, (selectionEndPos - selectionBeginPos), rulerHeight);
            g2d.setComposite(AlphaComposite.SrcOver);
            g2d.fillRect(selectionBeginPos, rulerHeight, (selectionEndPos - selectionBeginPos),
                w - rulerHeight);
        }
        /* Mod by Mark */
        /* Overlay the tier highlight colors */
        if(highlightColors != null) {
	        g2d.setComposite(alpha04);
	        for(int i = 0; i < visibleTiers.size(); i++) {
	        	Color highlightColor = highlightColors.get(((Tier2D)visibleTiers.get(i)).getName());
	        	if(highlightColor != null) {
	        		g2d.setColor(highlightColor);
	        		g2d.fillRect(0, y + (i * pixelsForTierHeight), w, pixelsForTierHeight);
	        	}
	        }
	        g2d.setComposite(AlphaComposite.SrcOver);
        }
        /* --- END --- */
        // translate horizontally
        int ax = timeToPixels(intervalBeginTime);
        g2d.translate(-ax, 0);
        int ht = pixelsForTierHeight - (2 * pixelsForTierHeightMargin);

        Tier2D tier2d;
        Tag2D tag2d;
        Font tf = null;

        synchronized (tierLock) {
            Iterator visIt = visibleTiers.iterator();
            
            	int count = 0;
            while (visIt.hasNext()) {
                tier2d = (Tier2D) visIt.next();
                tf = getFontForTier(tier2d.getTier());
                g2d.setFont(tf);               
                count++;               
                
                if (tier2d.isActive()) {    
                    g2d.setColor(Constants.ACTIVETIERCOLOR);
                    g2d.setComposite(alpha07);
                    g2d.fillRect(ax, y, imageWidth, pixelsForTierHeight);
                    g2d.setComposite(AlphaComposite.SrcOver);
                }

                Iterator tagIt = tier2d.getTags();

                while (tagIt.hasNext()) {
                    tag2d = (Tag2D) tagIt.next();

                    if (tag2d.getEndTime() < intervalBeginTime) {
                        continue; //don't paint
                    } else if (tag2d.getBeginTime() > intervalEndTime) {
                        break; // stop looping this tier
                    }

                    //x = timeToPixels(tag2d.getBeginTime());
                    //w = timeToPixels(tag2d.getEndTime()) - x;
                    //paint tag at this position
                    paintTag(g2d, tag2d, tag2d.getX(),
                        y + pixelsForTierHeightMargin, tag2d.getWidth(), ht);
                }

                y += pixelsForTierHeight;
            }
        }
        /*paint time ruler */
        if (timeRulerVisible) {
	        g2d.setColor(Constants.DEFAULTBACKGROUNDCOLOR);
	        g2d.fillRect(ax, 0, w, rulerHeight);
	        g2d.setColor(Constants.DEFAULTFOREGROUNDCOLOR);
	        ruler.paint(g2d, intervalBeginTime, w, msPerPixel,
	            SwingConstants.TOP);
        }
        
        g2d.setFont(font);
        // horizontally translate back
        g2d.translate(ax, 0);
        //paint selection over ruler
        if (selectionBeginPos != selectionEndPos && timeRulerVisible) {
            g2d.setColor(Constants.SELECTIONCOLOR);
            g2d.setComposite(alpha04);
            g2d.fillRect(selectionBeginPos, 0, (selectionEndPos - selectionBeginPos), rulerHeight);
            g2d.setComposite(AlphaComposite.SrcOver);
        }
        if (timeRulerVisible) {
	        g2d.setColor(Constants.SELECTIONCOLOR);
            g2d.drawLine(0, rulerHeight, w, rulerHeight);
        }
        // from paintComponent
        //g2d.setFont(font);
        if ((cursorTag2D != null) &&
                visibleTiers.contains(cursorTag2D.getTier2D())) {
            //int x = xAt(cursorTag2D.getBeginTime());
            //int w = xAt(cursorTag2D.getEndTime()) - x;
            int x = (int) ((cursorTag2D.getBeginTime() / msPerPixel) -
                (intervalBeginTime / msPerPixel));
            int ww = (int) ((cursorTag2D.getEndTime() -
                cursorTag2D.getBeginTime()) / msPerPixel);
            int yy = (rulerHeight + (cursorTierIndex * pixelsForTierHeight) +
                pixelsForTierHeightMargin) - verticalScrollOffset;
            int he = pixelsForTierHeight - (2 * pixelsForTierHeightMargin);
            paintCursorTag2D(g2d, cursorTag2D, x, yy, ww, he);
        }

        //paint empty slots
        if (showEmptySlots) {
            for (int i = 0; i < visibleTiers.size(); i++) {
                TierImpl ti = ((Tier2D) visibleTiers.get(i)).getTier();

                if (ti.getParentTier() == null) {
                    continue;
                } else {
                    if (!ti.isTimeAlignable()) {
                        int yy = (rulerHeight + (i * pixelsForTierHeight) +
                            pixelsForTierHeightMargin) - verticalScrollOffset;
                        int he = pixelsForTierHeight -
                            (2 * pixelsForTierHeightMargin);
                        paintEmptySlots(g2d, ti, yy, he);
                    }
                }
            }
        }

        // paint the dragEdit annotation
        if (dragEditTag2D != null) {
            //long newTime = pixelToTime(dragEditTag2D.getX());
            //int x = (int) ((dragEditTag2D.getBeginTime() / msPerPixel) -
            //	(intervalBeginTime / msPerPixel));
            int x = (int) (dragEditTag2D.getX() -
                (intervalBeginTime / msPerPixel));

            //int w = (int) ((dragEditTag2D.getEndTime() -
            //dragEditTag2D.getBeginTime()) / msPerPixel);
            int ww = dragEditTag2D.getWidth();
            int yy = (rulerHeight + (cursorTierIndex * pixelsForTierHeight) +
                
                /*(getTierIndexForAnnotation(dragEditTag2D.getAnnotation()) * pixelsForTierHeight) +*/
                pixelsForTierHeightMargin) - verticalScrollOffset + dragEditTag2D.getY();

            int he = pixelsForTierHeight - (2 * pixelsForTierHeightMargin);
            paintDragEditTag2D(g2d, x, yy, ww, he);
        }

        paintSelectedAnnotations(g2d);
        
        if ((crossHairPos >= 0) && (crossHairPos <= w)) {
            // prevents drawing outside the component on Mac
            g2d.setColor(Constants.CROSSHAIRCOLOR);
            g2d.drawLine(crossHairPos, 0, crossHairPos, h);
        }
    }
    
    /**
     * Paint the given Tag2D to the specified Graphics2D object using the
     * specified location and dimension.
     *
     * @param g2d the graphics object to paint to
     * @param tag2d the tag to paint
     * @param x the x postion of the tag
     * @param y the y position of the tag
     * @param width the width of the tag
     * @param height the height of the tag
     */
    private void paintTag(Graphics2D g2d, Tag2D tag2d, int x, int y, int width,
        int height) {
        if (tag2d.getAnnotation() instanceof AlignableAnnotation) {
            AlignableAnnotation a = (AlignableAnnotation) tag2d.getAnnotation();
            TimeSlot b = a.getBegin();
            TimeSlot e = a.getEnd();
            // preferred background color
            if (tag2d.getColor() != null) {
            	g2d.setColor(tag2d.getColor());
            	if (!reducedTierHeight) {
            		g2d.fillRect(x, (y + height / 2), width, (height / 2));
            	} else {
            		g2d.fillRect(x, y, width, height - rhDist);
            	}
            }
            //skip check cursor
            g2d.setColor(Constants.DEFAULTFOREGROUNDCOLOR);
            if (!reducedTierHeight) {
	            g2d.drawLine(x, y + (b.isTimeAligned() ? 0 : (height / 4)), x,
	                y + (b.isTimeAligned() ? height : ((height * 3) / 4)));
	            g2d.drawLine(x, y + (height / 2), x + width, y + (height / 2));
	            g2d.drawLine(x + width, y + (e.isTimeAligned() ? 0 : (height / 4)),
	                x + width, y +
	                (e.isTimeAligned() ? height : ((height * 3) / 4)));
	            if (tag2d.isTruncated()) {
	            	g2d.setColor(Constants.SHAREDCOLOR6);
	            	g2d.fillRect(x + width - 2, y + (height / 2) - 2, 2, 2);
	            }
            } else {
	            g2d.drawLine(x, y + (b.isTimeAligned() ? 0 : (height / 2 - rhDist)), x,
		                y + height);
		        g2d.drawLine(x, y + (height - rhDist), x + width, y + (height - rhDist));
		        g2d.drawLine(x + width, y + (e.isTimeAligned() ? 0 : (height / 2 - rhDist)),
		                x + width, y + height);
	            if (tag2d.isTruncated()) {
	            	g2d.setColor(Constants.SHAREDCOLOR6);
	            	g2d.fillRect(x + width - 2, y + height - rhDist - 2, 2, 2);
	            }
            }
        } else {
            if (tag2d.getColor() != null) {
            	g2d.setColor(tag2d.getColor());
            	if (!reducedTierHeight) {
            		g2d.fillRect(x, (y + height / 2), width, (height / 4));
            	} else {
            		g2d.fillRect(x, y, width, height - rhDist);
            	}
            	
            }
            //not alignable

            g2d.setColor(symAnnColor); // previously, this was a fixed color: Constants.SHAREDCOLOR1 
            if (!reducedTierHeight) {
	            g2d.drawLine(x, y + (height / 4), x, y + ((height * 3) / 4));
	            g2d.drawLine(x, y + (height / 2), x + width, y + (height / 2));
	            g2d.drawLine(x + width, y + (height / 4), x + width,
	                y + ((height * 3) / 4));
	            if (tag2d.isTruncated()) {
	            	g2d.setColor(Constants.SHAREDCOLOR6);
	            	g2d.fillRect(x + width - 2, y + (height / 2) - 2, 2, 2);
	            }
            } else {
	            g2d.drawLine(x, y + (height / 2 - rhDist), x, y + height);
	            g2d.drawLine(x, y + (height - rhDist), x + width, y + (height - rhDist));
	            g2d.drawLine(x + width, y + (height / 2 - rhDist), x + width,
	                y + height);
	            if (tag2d.isTruncated()) {
	            	g2d.setColor(Constants.SHAREDCOLOR6);
	            	g2d.fillRect(x + width - 2, y + height - rhDist - 2, 2, 2);
	            }
            }
        }

        g2d.setColor(Constants.DEFAULTFOREGROUNDCOLOR);

        int descent = g2d.getFontMetrics().getDescent();
        if (!reducedTierHeight) {
	        g2d.drawString(tag2d.getTruncatedValue(), (float) (x + 4),
	            (float) (y + ((height / 2) - descent + 1)));
        } else {
	        g2d.drawString(tag2d.getTruncatedValue(), (float) (x + 4),
		        (float) (y + ((height - rhDist) - descent + 1)));
        }
        /////
        /*
        if (tag2d.getAnnotation() instanceof AlignableAnnotation) {
            AlignableAnnotation aa = (AlignableAnnotation) tag2d.getAnnotation();
            String vv = aa.getBegin().getIndex() + " - " + aa.getEnd().getIndex();
            g2d.drawString(vv, (float) (x + 4),
                    (float) (y + ((height) - descent + 1)));
        }
        */
    }

    /**
     * Paint the mouseover highlight for a tag.
     *
     * @param g2d
     * @param tag2d
     * @param x
     * @param y
     * @param width
     * @param height
     */

    /*
       private void paintHoverTag2D(Graphics2D g2d, Tag2D tag2d, int x, int y,
           int width, int height) {
           g2d.setColor(Constants.SHAREDCOLOR3);
           g2d.drawRect(x, y, width, height);
           g2d.setColor(Constants.SHAREDCOLOR4);
           g2d.fillRect(x, y, width - 1, height - 1);
           g2d.setColor(Constants.DEFAULTFOREGROUNDCOLOR);
           g2d.drawString(tag2d.getTruncatedValue(), x + 4,
               (int) (y + ((height / 2) - 1)));
       }
     */

    /**
     * Paint the selected Tag.
     *
     * @param g2d
     * @param tag2d
     * @param x
     * @param y
     * @param width
     * @param height
     */
    private void paintCursorTag2D(Graphics2D g2d, Tag2D tag2d, int x, int y,
        int width, int height) {
        if (tag2d.getAnnotation() instanceof AlignableAnnotation) {
            AlignableAnnotation a = (AlignableAnnotation) tag2d.getAnnotation();
            TimeSlot b = a.getBegin();
            TimeSlot e = a.getEnd();
            g2d.setColor(Constants.ACTIVEANNOTATIONCOLOR);

            if (aaStrokeBold) {
                g2d.setStroke(stroke2);
            }

            int top = b.isTimeAligned() ? 0 : (height / 4);
            int bottom = b.isTimeAligned() ? height : ((height * 3) / 4);
            if (reducedTierHeight) {
            	top = (height / 2 - rhDist);
            	bottom = height;
            }
            g2d.drawLine(x, y + top, x, y + bottom);
            int mid = reducedTierHeight ? (height - rhDist) : height / 2;

            if (aaStrokeBold) {
                mid++;
            }

            //g2d.drawLine(x, y + top + 1, x + width, y + top + 1);
            g2d.drawLine(x, y + mid, x + width, y + mid);
            //top = e.isTimeAligned() ? 0 : (height / 4);
            //bottom = e.isTimeAligned() ? height : ((height * 3) / 4);
            g2d.drawLine(x + width, y + top, x + width, y + bottom);
            g2d.setStroke(stroke);
        } else {
            //not alignable
            g2d.setColor(Constants.ACTIVEANNOTATIONCOLOR);

            if (aaStrokeBold) {
                g2d.setStroke(stroke2);
            }
            if (!reducedTierHeight) {
	            g2d.drawLine(x, y + (height / 4), x, y + ((height * 3) / 4));
	            g2d.drawLine(x, y + (height / 2), x + width, y + (height / 2));
	            g2d.drawLine(x + width, y + (height / 4), x + width,
	                y + ((height * 3) / 4));
            } else {
                g2d.drawLine(x, y + (height / 2 - rhDist), x, y + height);
                g2d.drawLine(x, y + (height - rhDist), x + width, y + (height - rhDist));
                g2d.drawLine(x + width, y + (height / 2 - rhDist), x + width,
                    y + height);
            }
            g2d.setStroke(stroke);
        }
    }

    /**
     * Paints the tag is edited by dragging its boundaries, or by dragging  the
     * whole tag.
     *
     * @param g2d
     * @param x
     * @param y
     * @param width
     * @param height
     */
    private void paintDragEditTag2D(Graphics2D g2d, int x, int y, int width,
        int height) {
        g2d.setColor(dragEditColor);

        if (aaStrokeBold) {
            g2d.setStroke(stroke2);
        }

        int top = 0;
        int bottom = height;
        g2d.drawLine(x, y + top, x, y + bottom);
        top = reducedTierHeight ? height / 2 - rhDist : height / 2;

        if (aaStrokeBold) {
            top++;
        }

        //g2d.drawLine(x, y + top + 1, x + width, y + top + 1);
        g2d.drawLine(x, y + top, x + width, y + top);

        g2d.drawLine(x + width, y, x + width, y + bottom);
        if (dragEditTag2D != null && dragEditTag2D.isOverTargetTier) {// shouldn't be null
        	g2d.drawRect(x, y, width, height);
        	g2d.setColor(dropHighLlightColor);
        	g2d.fillRect(x, y, width, height);
        }
        g2d.setStroke(stroke);
    }

    /**
     * Paint empty slots on this tier.<br>
     * Iterate over the parent tags in the visible area and paint a tag when
     * it is not on the child.
     *
     * @param g2d the graphics context
     * @param ti the tier containing empty slots
     * @param y y coordinate for the tags
     * @param he height of the tags
     */
    private void paintEmptySlots(Graphics2D g2d, TierImpl ti, int y, int he) {
        try {
            TierImpl parent = (TierImpl) ti.getParentTier();
            Vector tags = parent.getAnnotations();
            Annotation a;
            int x;
            int wi;
            Iterator tagIt = tags.iterator();

            while (tagIt.hasNext()) {
                a = (Annotation) tagIt.next();

                if (a.getEndTimeBoundary() < intervalBeginTime) {
                    continue;
                }

                if (a.getBeginTimeBoundary() > intervalEndTime) {
                    break;
                }

                if (a.getChildrenOnTier(ti).size() == 0) {
                    x = (int) (((float)a.getBeginTimeBoundary() / msPerPixel) -
                        ((float)intervalBeginTime / msPerPixel));
                    wi = (int) ((a.getEndTimeBoundary() -
                        a.getBeginTimeBoundary()) / msPerPixel);
                    g2d.setColor(Constants.SHAREDCOLOR4);
                    g2d.fillRect(x, y, wi, he);
                    g2d.setColor(Constants.DEFAULTFOREGROUNDCOLOR);
                    g2d.drawRect(x, y, wi, he);
                }
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }
    
    private void paintSelectedAnnotations(Graphics2D g2d) {
    	if (selectedAnnotations.size() != 0) {
    		g2d.setColor(Constants.SHAREDCOLOR5);
    		int xOff = (int) (intervalBeginTime / msPerPixel);
    		for (Tag2D t2d : selectedAnnotations) {
    			if (visibleTiers.contains(t2d.getTier2D())) {
    				g2d.drawRect(t2d.getX() - xOff, 
    						rulerHeight + (getTierIndexForAnnotation(t2d.getAnnotation()) * pixelsForTierHeight) + 1 - verticalScrollOffset, 
    						t2d.getWidth(), pixelsForTierHeight - 2);
    			}
    		}
    	}
    }

    /**
     * Returns the x-ccordinate for a specific time. The coordinate is in the
     * component's coordinate system.
     *
     * @param t time
     *
     * @return int the x-coordinate for the specified time
     */
    public int xAt(long t) {
    	return (int) ((t / msPerPixel) - (intervalBeginTime / msPerPixel));
        //return (int) ((t - intervalBeginTime) / msPerPixel);
    }

    /**
     * Returns the time in ms at a given position in the current image. The
     * given x coordinate is in the component's ("this") coordinate system.
     * The interval begin time is included in the calculation of the time at
     * the given coordinate.
     *
     * @param x x-coordinate
     *
     * @return the mediatime corresponding to the specified position
     */
    public long timeAt(int x) {
        return intervalBeginTime + (int)(x * msPerPixel);
    }

    /**
     * Calculates the x coordinate in virtual image space.<br>
     * This virtual image would be an image of width <br>
     * media duration in ms / ms per pixel. Therefore the return value does
     * not correct for interval begin time and is not necessarily within the
     * bounds of this component.
     *
     * @param theTime the media time
     *
     * @return the x coordinate in the virtual image space
     */
    private int timeToPixels(long theTime) {
        return (int) (theTime / msPerPixel);
    }

    /**
     * Calculates the time corresponding to a pixel location in the virtual
     * image space.
     *
     * @param x the x coordinate in virtual image space
     *
     * @return the media time at the specified point
     */
    private long pixelToTime(int x) {
        return (long) (x * msPerPixel);
    }
    
    /**
     * Calculates the height for all tiers, based on the base font size.
     * 
     * @return the height in pixels
     */
    private int calcTierHeight() {
    	int th = (int)((font.getSize() * 2.5) + (36 / font.getSize()));
        if (reducedTierHeight) {
        	th = (th * 2) / 3;
        }
        return th;
    }

    /**
     * Implements updateTimeScale from TimeScaleBasedViewer to adjust the
     * TimeScale if needed and when in TimeScale connected mode.<br>
     * Checks the GlobalTimeScaleIntervalBeginTime and
     * GlobalTimeScaleMsPerPixel and adjusts the interval and resolution of
     * this viewer when they differ from the global values.<br>
     * For the time being assume that the viewer is notified only once when
     * the resolution or the interval begintime has changed.
     */
    public void updateTimeScale() {
        if (timeScaleConnected) {
            //if the resolution is changed recalculate the begin time
            if (getGlobalTimeScaleMsPerPixel() != msPerPixel) {
                setLocalTimeScaleMsPerPixel(getGlobalTimeScaleMsPerPixel());
            } else if (getGlobalTimeScaleIntervalBeginTime() != intervalBeginTime) {
                //assume the resolution has not been changed
                setLocalTimeScaleIntervalBeginTime(getGlobalTimeScaleIntervalBeginTime());

                //System.out.println("update begin time in TimeLineViewer called");
            }
        }
    }

    /**
     * Sets whether or not this viewer listens to global time scale updates.
     *
     * @param connected the new timescale connected value
     */
    public void setTimeScaleConnected(boolean connected) {
        timeScaleConnected = connected;

        if (timeScaleConnected) {
            if (msPerPixel != getGlobalTimeScaleMsPerPixel()) {
                setLocalTimeScaleMsPerPixel(getGlobalTimeScaleMsPerPixel());
            }

            if (intervalBeginTime != getGlobalTimeScaleIntervalBeginTime()) {
                setLocalTimeScaleIntervalBeginTime(getGlobalTimeScaleIntervalBeginTime());
            }
        }
    }

    /**
     * Gets whether this viewer listens to time scale updates from other
     * viewers.
     *
     * @return true when connected to global time scale values, false otherwise
     */
    public boolean getTimeScaleConnected() {
        return timeScaleConnected;
    }

    /**
     * Checks whether this viewer is TimeScale connected and changes the
     * milliseconds per pixel value globally or locally.
     *
     * @param mspp the new milliseconds per pixel value
     */
    public void setMsPerPixel(float mspp) {
        if (timeScaleConnected) {
            setGlobalTimeScaleMsPerPixel(mspp);
            setGlobalTimeScaleIntervalBeginTime(intervalBeginTime);
            setGlobalTimeScaleIntervalEndTime(intervalEndTime);
        } else {
            setLocalTimeScaleMsPerPixel(mspp);
        }
    }

    /**
     * Change the horizontal resolution or zoomlevel locally. The msPerPixel
     * denotes the number of milliseconds of which the sound samples should be
     * merged to one value. It corresponds to one pixel in image space (a
     * pixel is the smallest unit in image space).<br>
     * The position on the screen of crosshair cursor should change as little
     * as possible.<br>
     * This is calculated as follows:<br>
     * The absolute x coordinate in image space is the current media time
     * divided by the new msPerPixel.<br>
     * <pre>
     * |----------|----------|-------x--|-- <br>
     * |imagesize |                  | absolute x coordinate of media time<br>
     * |    1     |    2     |    3     |
     * </pre>
     * Calculate the number of screen images that fit within the absolute x
     * coordinate. The new position on the screen would then be the absolute x
     * coordinate minus the number of screen images multiplied by the image
     * width. The difference between the old x value and the new x value is
     * then used to calculte the new interval start time.<br>
     * The new start time = (number of screen images  image width -
     * difference)  msPerPixel.
     *
     * @param step the new horizontal zoomlevel
     */
    private void setLocalTimeScaleMsPerPixel(float step) {
        if (msPerPixel == step) {
            return;
        }

        if (step >= TimeScaleBasedViewer.MIN_MSPP) {
            msPerPixel = step;
        } else {
            msPerPixel = TimeScaleBasedViewer.MIN_MSPP;
        }

        resolution = (int) (1000f / msPerPixel);

        /*stop the player if necessary*/
        boolean playing = playerIsPlaying();

        if (playing) {
            stopPlayer();
        }

        long mediaTime = getMediaTime();
        int oldScreenPos = crossHairPos;
        int newMediaX = (int) (mediaTime / msPerPixel);
        int numScreens;

        if (imageWidth > 0) {
            numScreens = (int) (mediaTime / (imageWidth * msPerPixel));
        } else {
            numScreens = 0;
        }

        int newScreenPos = newMediaX - (numScreens * imageWidth);
        int diff = oldScreenPos - newScreenPos;

        //new values
        intervalBeginTime = (long) (((numScreens * imageWidth) - diff) * msPerPixel);

        if (intervalBeginTime < 0) {
            intervalBeginTime = 0;
        }

        intervalEndTime = intervalBeginTime + (long) (imageWidth * msPerPixel);
        recalculateTagSizes();
        crossHairPos = xAt(mediaTime);
        selectionBeginPos = xAt(getSelectionBeginTime());
        selectionEndPos = xAt(getSelectionEndTime());
        updateHorScrollBar();
        paintBuffer();

        if (playing) {
            startPlayer();
        }

        int zoom = (int) (100f * (10f / msPerPixel));

        if (zoom <= 0) {
            zoom = 100;
        }

        updateZoomPopup(zoom);

        if (zoomSlider != null) {
        	if (zoom <= 100) {
        		zoomSlider.setValue(zoom);
        	} else {
        		// recalculate for > 100 values.
        		float factor = (zoom - 100) / (float) 900;
        		int zm = 100 + (int) ( factor * (zoomSlider.getMaximum() - 100) );
        		if (zm != zoomSlider.getValue()) {
        			zoomSlider.removeChangeListener(this);
        			zoomSlider.setValue(zm);
        			zoomSlider.addChangeListener(this);
        			zoomSlider.setToolTipText(String.valueOf(zoom));
        		}
        	}
        }
        //repaint();
    }

    /**
     * Returns the current msPerPixel.
     *
     * @return msPerPixel
     */
    public float getMsPerPixel() {
        return msPerPixel;
    }

    /**
     * Calls #setMsPerPixel with the appropriate value. In setMsPerPixel the
     * value of this.resolution is actually set. msPerPixel = 1000 / resolution<br>
     * resolution = 1000 / msPerPixel
     *
     * @param resolution the new resolution
     */
    public void setResolution(int resolution) {
        if (resolution < 1) {
            this.resolution = 1;
        } else {
            this.resolution = resolution;
        }

        float mspp = (1000f / resolution);
        setMsPerPixel(mspp);
    }

    /**
     * Sets the resolution by providing a factor the default PIXELS_FOR_SECOND
     * should be multiplied with.<br>
     * resolution = factor * PIXELS_FOR_SECOND.<br>
     * <b>Note:</b><br>
     * The factor = 100 / resolution_menu_percentage !
     *
     * @param factor the multiplication factor
     */
    public void setResolutionFactor(float factor) {
        int res = (int) (PIXELS_FOR_SECOND * factor);
        setResolution(res);
    }

    /**
     * Gets the current resolution
     *
     * @return the current resolution
     */
    public int getResolution() {
        return resolution;
    }

    /**
     * Find the tag at the given location.
     *
     * @param p the location
     * @param tierIndex the tier the tag should be found in
     *
     * @return the tag
     */
    private Tag2D getTagAt(Point2D p, int tierIndex) {
        if ((tierIndex < 0) || (tierIndex > (visibleTiers.size() - 1))) {
            return null;
        }

        long pTime = pixelToTime((int) p.getX());
        Tag2D t2d;
        Iterator it = ((Tier2D) visibleTiers.get(tierIndex)).getTags();

        while (it.hasNext()) {
            t2d = (Tag2D) it.next();

            if ((pTime >= t2d.getBeginTime()) && (pTime <= t2d.getEndTime())) {
                return t2d;
            }
        }

        return null;
    }

    /**
     * Calculate the index in the visible tiers array for the given y
     * coordinate.
     *
     * @param p DOCUMENT ME!
     *
     * @return the index of the Tier2D  or -1 when not found
     */
    private int getTierIndexForPoint(Point2D p) {
        int y = (int) p.getY() - rulerHeight;

        if ((y < 0) || (y > (visibleTiers.size() * pixelsForTierHeight))) {
            return -1;
        } else {
            return y / pixelsForTierHeight;
        }
    }

    /**
     * Calculate the index in the visible tiers array for the given annotation.
     *
     * @param annotation DOCUMENT ME!
     *
     * @return the index of the Tier2D or -1 when not found
     */
    private int getTierIndexForAnnotation(Annotation annotation) {
        Tier tier = annotation.getTier();
        int index = -1;

        for (int i = 0; i < visibleTiers.size(); i++) {
            if (((Tier2D) visibleTiers.get(i)).getTier() == tier) {
                index = i;

                break;
            }
        }

        return index;
    }

    /**
     * Inverts the point and finds the Tag2D at that point.
     *
     * @param p point in component space
     *
     * @return a tag2d or null
     */
    private Tag2D getHoverTag(Point p) {
        p.x += timeToPixels(intervalBeginTime);
        p.y += verticalScrollOffset;
        hoverTierIndex = getTierIndexForPoint(p);

        Tag2D hover = getTagAt(p, hoverTierIndex);

        return hover;
    }

    /**
     * Update the dragedit tag2d while dragging. <br>
     * Checks on the parent's boundaries (if any).
     *
     * @param dragEndPoint the position of the mouse pointer
     */
    private void updateDragEditTag(Point dragEndPoint) {
        if (dragEditTag2D == null) {
            return;
        }
        
        dragEditTag2D.move(dragEndPoint.x - dragStartPoint.x, dragEndPoint.y - dragStartPoint.y);
        int diff = dragEndPoint.x - dragStartPoint.x;
        int diffy = dragEndPoint.y - dragStartPoint.y;

        //if (Math.abs(dragEditTag2D.getDy()) > Math.abs(dragEditTag2D.getDx())) {
        if (dragEditMode == DRAG_EDIT_CENTER && (Math.abs(dragEditTag2D.getDy()) > Math.abs(dragEditTag2D.getDx()) || 
        		Math.abs(dragEditTag2D.getDy()) >= pixelsForTierHeight)) {
        	// vertical dragging
        	dragEditTag2D.setY(dragEditTag2D.getY() + diffy);
        	dragEditTag2D.resetX();
            dragStartPoint = dragEndPoint;
            int index = getTierIndexForPoint(new Point(dragEndPoint.x, dragEndPoint.y + verticalScrollOffset));
            if (index > -1 && index < visibleTiers.size()) {
            	Tier2D t2d = visibleTiers.get(index);
            	if (t2d != dragEditTag2D.getTier2D() && t2d.getTier().getParentTier() == null) {
            		dragEditTag2D.isOverTargetTier = true;
            	} else {
            		dragEditTag2D.isOverTargetTier = false;
            	}
            }
        } else {
        	// horizontal dragging
	        switch (dragEditMode) {
	        case DRAG_EDIT_CENTER:
	        	dragEditTag2D.resetY();
	            if (dragParentBegin == -1) {
	                dragEditTag2D.setX(dragEditTag2D.getX() + diff);
	                dragStartPoint = dragEndPoint;
	            } else {
	                long bt = pixelToTime(dragEditTag2D.getX() + diff);
	
	                if (diff < 0) {
	                    if (bt < dragParentBegin) {
	                        bt = dragParentBegin;
	
	                        int nx = timeToPixels(bt);
	                        dragEditTag2D.setX(nx);
	                    } else {
	                        dragEditTag2D.setX(dragEditTag2D.getX() + diff);
	                        dragStartPoint = dragEndPoint;
	                    }
	                } else {
	                    long et = pixelToTime(dragEditTag2D.getX() +
	                            dragEditTag2D.getWidth() + diff);
	
	                    if (et > dragParentEnd) {
	                        et = dragParentEnd;
	                        bt = et - pixelToTime(dragEditTag2D.getWidth());
	
	                        dragEditTag2D.setX(timeToPixels(bt));
	                    } else {
	                        dragEditTag2D.setX(dragEditTag2D.getX() + diff);
	                        dragStartPoint = dragEndPoint;
	                    }
	                }
	            }
	
	            setMediaTime(pixelToTime(dragEditTag2D.getX()));
	
	            break;
	
	        case DRAG_EDIT_LEFT:
	
	            if ((dragEditTag2D.getX() + diff) < ((dragEditTag2D.getX() +
	                    dragEditTag2D.getWidth()) - 1)) {
	                if ((dragParentBegin == -1) || (diff > 0)) {
	                    dragEditTag2D.setX(dragEditTag2D.getX() + diff);
	                    dragEditTag2D.setWidth(dragEditTag2D.getWidth() - diff);
	                    dragStartPoint = dragEndPoint;
	                } else if ((dragParentBegin > -1) && (diff < 0)) {
	                    long bt = pixelToTime(dragEditTag2D.getX() + diff);
	
	                    if (bt < dragParentBegin) {
	                        bt = dragParentBegin;
	
	                        int nx = timeToPixels(bt);
	                        dragEditTag2D.setX(nx);
	                        dragEditTag2D.setWidth(timeToPixels(dragEditTag2D.getEndTime() -
	                                bt));
	                    } else {
	                        dragEditTag2D.setX(dragEditTag2D.getX() + diff);
	                        dragEditTag2D.setWidth(dragEditTag2D.getWidth() - diff);
	                        dragStartPoint = dragEndPoint;
	                    }
	                }
	
	                setMediaTime(pixelToTime(dragEditTag2D.getX()));
	            }
	
	            break;
	
	        case DRAG_EDIT_RIGHT:
	
	            if ((dragEditTag2D.getWidth() + diff) > 1) {
	                if ((dragParentEnd == -1) || (diff < 0)) {
	                    dragEditTag2D.setWidth(dragEditTag2D.getWidth() + diff);
	                    dragStartPoint = dragEndPoint;
	                } else if ((dragParentEnd > -1) && (diff > 0)) {
	                    long et = pixelToTime(dragEditTag2D.getX() +
	                            dragEditTag2D.getWidth() + diff);
	
	                    if (et > dragParentEnd) {
	                        et = dragParentEnd;
	                        dragEditTag2D.setWidth(timeToPixels(et) -
	                            dragEditTag2D.getX());
	                    } else {
	                        dragEditTag2D.setWidth(dragEditTag2D.getWidth() + diff);
	                        dragStartPoint = dragEndPoint;
	                    }
	                }
	
	                setMediaTime(pixelToTime(dragEditTag2D.getX() +
	                        dragEditTag2D.getWidth()));
	            }
	
	            break;
	        }
        }

        repaint();
    }

    /**
     * DOCUMENT ME!
     *
     * @return the current interval begin time
     */
    public long getIntervalBeginTime() {
        return intervalBeginTime;
    }

    /**
     * DOCUMENT ME!
     *
     * @return the current interval end time
     */
    public long getIntervalEndTime() {
        return intervalEndTime;
    }

    /**
     * Checks whether this viewer is TimeScale connected and changes the
     * interval begin time globally or locally.
     *
     * @param begin the new interval begin time
     */
    public void setIntervalBeginTime(long begin) {
        if (timeScaleConnected) {
            setGlobalTimeScaleIntervalBeginTime(begin);
            setGlobalTimeScaleIntervalEndTime(intervalEndTime);
        } else {
            setLocalTimeScaleIntervalBeginTime(begin);
        }
    }

    /**
     * Calculates the new interval begin and/or end time.<br>
     * There are two special cases taken into account:<br>
     * 
     * <ul>
     * <li>
     * when the player is playing attempts are made to shift the interval
     * <i>n</i> times the interval size to the left or to the right, until the
     * new interval contains the new mediatime.
     * </li>
     * <li>
     * when the player is not playing and the new interval begin time coincides
     * with the selection begin time, the interval is shifted a certain offset
     * away from the image edge. Same thing when the interval end time
     * coincides with the selection end time.
     * </li>
     * </ul>
     * 
     *
     * @param mediaTime
     */
    private void recalculateInterval(final long mediaTime) {
        long newBeginTime = intervalBeginTime;
        long newEndTime = intervalEndTime;

        if (playerIsPlaying()) {
            // we might be in a selection outside the new interval
            // shift the interval n * intervalsize to the left or right
            if (mediaTime > intervalEndTime) {
                newBeginTime = intervalEndTime;
                newEndTime = newBeginTime + (int) (imageWidth * msPerPixel);

                while ((newEndTime += (imageWidth + msPerPixel)) < mediaTime) {
                    newBeginTime += (imageWidth * msPerPixel);
                }
            } else if (mediaTime < intervalBeginTime) {
                newEndTime = intervalBeginTime;
                newBeginTime = newEndTime - (int) (imageWidth * msPerPixel);

                while ((newEndTime -= (imageWidth * msPerPixel)) > mediaTime) {
                    newBeginTime -= (imageWidth * msPerPixel);
                }

                if (newBeginTime < 0) {
                    newBeginTime = 0;
                    newEndTime = (long) (imageWidth * msPerPixel);
                }
            } else {
                // the new time appears to be in the current interval after all
                return;
            }
        } else { //player is not playing

            // is the new media time to the left or to the right of the current interval
            if (mediaTime < intervalBeginTime) {
                newBeginTime = mediaTime - (int) (SCROLL_OFFSET * msPerPixel);

                if (newBeginTime < 0) {
                    newBeginTime = 0;
                }

                newEndTime = newBeginTime + (int) (imageWidth * msPerPixel);
            } else if (mediaTime > intervalEndTime) {
                newEndTime = mediaTime + (int) (SCROLL_OFFSET * msPerPixel);
                newBeginTime = newEndTime - (int) (imageWidth * msPerPixel);

                if (newBeginTime < 0) { // something would be wrong??
                    newBeginTime = 0;
                    newEndTime = newBeginTime + (int) (imageWidth * msPerPixel);
                }
            }

            if ((newBeginTime == getSelectionBeginTime()) &&
                    (newBeginTime > (SCROLL_OFFSET * msPerPixel))) {
                newBeginTime -= (SCROLL_OFFSET * msPerPixel);
                newEndTime = newBeginTime + (int) (imageWidth * msPerPixel);
            }

            if (newEndTime == getSelectionEndTime()) {
                newEndTime += (SCROLL_OFFSET * msPerPixel);
                newBeginTime = newEndTime - (int) (imageWidth * msPerPixel);

                if (newBeginTime < 0) { // something would be wrong??
                    newBeginTime = 0;
                    newEndTime = newBeginTime + (int) (imageWidth * msPerPixel);
                }
            }

            // try to position the whole selection in the view
            if ((mediaTime == getSelectionBeginTime()) &&
                    (getSelectionEndTime() > (newEndTime -
                    (SCROLL_OFFSET * msPerPixel))) && !panMode) {
                newEndTime = getSelectionEndTime() +
                    (int) (SCROLL_OFFSET * msPerPixel);
                newBeginTime = newEndTime - (int) (imageWidth * msPerPixel);

                if ((newBeginTime > mediaTime) &&
                        (mediaTime > (SCROLL_OFFSET * msPerPixel))) {
                    newBeginTime = mediaTime - (int) (SCROLL_OFFSET * msPerPixel);
                    newEndTime = newBeginTime + (int) (imageWidth * msPerPixel);
                } else if (newBeginTime > mediaTime) {
                    newBeginTime = 0;
                    newEndTime = (long) (imageWidth * msPerPixel);
                }
            }
        }

        if (timeScaleConnected) {
            //System.out.println("TLV new begin time: " + newBeginTime);
            //System.out.println("TLV new end time: " + newEndTime);
            setGlobalTimeScaleIntervalBeginTime(newBeginTime);
            setGlobalTimeScaleIntervalEndTime(newEndTime);
        } else {
            setLocalTimeScaleIntervalBeginTime(newBeginTime);
        }
    }

    /**
     * Changes the interval begin time locally.
     *
     * @param begin the new local interval begin time
     */
    private void setLocalTimeScaleIntervalBeginTime(long begin) {
        if (begin == intervalBeginTime) {
            return;
        }

        intervalBeginTime = begin;
        intervalEndTime = intervalBeginTime + (int) (imageWidth * msPerPixel);

        //
        if (editBox.isVisible()) {
            if (getActiveAnnotation() != null) {
                int x = xAt(getActiveAnnotation().getBeginTimeBoundary());
                editBox.setLocation(x, editBox.getY());
            } else {
                dismissEditBox();
            }

            /*
               if (x < 0 || x > imageWidth) {
                   dismissEditBox();
               } else {
                   editBox.setLocation(x, editBox.getY());
               }
             */
        }

        //
        crossHairPos = xAt(crossHairTime);
        selectionBeginPos = xAt(getSelectionBeginTime());
        selectionEndPos = xAt(getSelectionEndTime());
        updateHorScrollBar();
        paintBuffer();
    }

    /**
     * DOCUMENT ME!
     *
     * @return the vertical scroll offset
     */
    public int getVerticalScrollOffset() {
        return verticalScrollOffset;
    }

    /**
     * Sets the vertical scroll offset of the tags on this component. <b>Note:</b><br>
     * There should be some kind of synchronization with other viewers:
     * pending..
     *
     * @param offset the new vertical scroll offset
     */
    public void setVerticalScrollOffset(int offset) {
        verticalScrollOffset = offset;
//        repaint();
        paintBuffer();
    }

    /**
     * Scrolls the viewport vertically to ensure the cursorTag is visible.
     */
    private void ensureVerticalVisibilityOfActiveAnnotation() {
        if (cursorTag2D == null) {
            return;
        }

        int cy = cursorTierIndex * pixelsForTierHeight;

        if (cy < verticalScrollOffset) {
            scrollBar.setValue(cy);
        } else if (((cy + pixelsForTierHeight) - verticalScrollOffset) > (getHeight() -
                rulerHeight)) {
            scrollBar.setValue((cy + pixelsForTierHeight + rulerHeight) -
                getHeight());
        }
    }

    /**
     * Update the values of the scrollbar.<br>
     * Called after a change in the number of visible tiers.
     */
    private void updateScrollBar() {
        int value = scrollBar.getValue();
        int max = (visibleTiers.size() * pixelsForTierHeight) + rulerHeight;

        // before changing scrollbar values do a setValue(0), otherwise
        // setMaximum and/or setVisibleAmount will not be accurate
        scrollBar.setValue(0);
        scrollBar.setMaximum(max);
        if (hScrollBarVisible) {
            scrollBar.setVisibleAmount(getHeight() - defBarWidth);
        } else {
        scrollBar.setVisibleAmount(getHeight());
        }

        if ((value + getHeight()) > max) {
            value = max - getHeight();
        }

        scrollBar.setValue(value);
        scrollBar.revalidate();
    }

    /**
     * Updates the values of the horizontal scrollbar. Called when the interval begin time, the 
     * resolution (msPerPixel), the viewer's width or the master media duration has changed. 
     */
    private void updateHorScrollBar() {
        if (!hScrollBarVisible) {
            return;
        }
        int value = hScrollBar.getValue();
        if (value != (int)(intervalBeginTime / msPerPixel)) {
            value = (int)(intervalBeginTime / msPerPixel);
        }
        int max = (int) (getMediaDuration() / msPerPixel + DEFAULT_MS_PER_PIXEL);
        hScrollBar.removeAdjustmentListener(this);
        
        hScrollBar.setValue(0);
        hScrollBar.setMaximum(max);
        hScrollBar.setVisibleAmount(getWidth() - defBarWidth);
        hScrollBar.setBlockIncrement(getWidth() - defBarWidth);
        if (value != hScrollBar.getValue()) {
            hScrollBar.setValue(value);
        }
        hScrollBar.revalidate();
        hScrollBar.addAdjustmentListener(this);
    }

    /**
     * Calculate the y positions of the vertical middle of all visible tiers
     * and pass them to the MultiViewerController.
     */
    private void notifyMultiTierControlPanel() {
        if (multiTierControlPanel == null) {
            return;
        }

        if (tierYPositions.length != visibleTiers.size()) {
        	tierYPositions = new int[visibleTiers.size()];
        }
        if (tierYPositions.length > 0) {
        	int bh = timeRulerVisible ? rulerHeight : 0;
            tierYPositions[0] = (bh + (pixelsForTierHeight / 2)) -
                verticalScrollOffset;

            for (int i = 1; i < visibleTiers.size(); i++) {
                tierYPositions[i] = tierYPositions[0] +
                    (i * pixelsForTierHeight);
            }
        }

        multiTierControlPanel.setTierPositions(tierYPositions);
    }

    /**
     * Returns the actual size of the viewer (viewable area), i.e. the size of
     * the component minus the size of the scrollbar.<br>
     * Needed for the accurate alignment with other viewers.
     *
     * @return the actual size of the viewer
     */
    public Dimension getViewerSize() {
        return new Dimension(imageWidth, imageHeight);
    }

    /**
     * Layout information, gives the nr of pixels at the left of the viewer
     * panel that contains no time line information
     *
     * @return the nr of pixels at the left that contain no time line related
     *         data
     */
    public int getLeftMargin() {
        return 0;
    }

    /**
     * Layout information, gives the nr of pixels at the right of the viewer
     * panel that contains no time line information
     *
     * @return the nr of pixels at the right that contain no time line related
     *         data
     */
    public int getRightMargin() {
        return scrollBar.getWidth();
    }

    /**
     * Create a popup menu to enable the manipulation of some settings for this
     * viewer.
     */
    private void createPopupMenu() {
        popup = new JPopupMenu("TimeLine Viewer");
        zoomMI = new JMenu(ElanLocale.getString("TimeScaleBasedViewer.Zoom"));
        zoomBG = new ButtonGroup();
        zoomSelectionMI = new JMenuItem(
        		ElanLocale.getString("TimeScaleBasedViewer.Zoom.Selection"));
        zoomSelectionMI.addActionListener(this);
        zoomSelectionMI.setActionCommand("zoomSel");
        zoomMI.add(zoomSelectionMI);
        customZoomMI = new JRadioButtonMenuItem(
        		ElanLocale.getString("TimeScaleBasedViewer.Zoom.Custom"));
        customZoomMI.setEnabled(false);
        zoomBG.add(customZoomMI);
        zoomMI.add(customZoomMI);
        zoomMI.addSeparator();
        //
        JRadioButtonMenuItem zoomRB;

        for (int i = 0; i < ZOOMLEVELS.length; i++) {
            zoomRB = new JRadioButtonMenuItem(ZOOMLEVELS[i] + "%");
            zoomRB.setActionCommand(String.valueOf(ZOOMLEVELS[i]));
            zoomRB.addActionListener(this);
            zoomBG.add(zoomRB);
            zoomMI.add(zoomRB);

            if (ZOOMLEVELS[i] == 100) {
                zoomRB.setSelected(true);
            }
        }

        popup.add(zoomMI);

        // font size items		
        int fontSize = getFont().getSize();
        fontSizeBG = new ButtonGroup();
        fontMenu = new JMenu(ElanLocale.getString("Menu.View.FontSize"));
        
		JRadioButtonMenuItem fontRB;
		
		for (int i = 0; i < Constants.FONT_SIZES.length; i++) {
			fontRB = new JRadioButtonMenuItem(String.valueOf(Constants.FONT_SIZES[i]));
			fontRB.setActionCommand("font" + Constants.FONT_SIZES[i]);
			if (fontSize == Constants.FONT_SIZES[i]) {
				fontRB.setSelected(true);
			}
			fontRB.addActionListener(this);
			fontSizeBG.add(fontRB);
			fontMenu.add(fontRB);
		}

        popup.add(fontMenu);

        activeAnnStrokeBoldMI = new JCheckBoxMenuItem(ElanLocale.getString(
                    "TimeLineViewer.ActiveAnnotationBold"));
        activeAnnStrokeBoldMI.setSelected(aaStrokeBold);
        activeAnnStrokeBoldMI.setActionCommand("aastroke");
        activeAnnStrokeBoldMI.addActionListener(this);
        popup.add(activeAnnStrokeBoldMI);
        
        reducedTierHeightMI = new JCheckBoxMenuItem(ElanLocale.getString(
        		"TimeLineViewer.ReducedTierHeight"));
        reducedTierHeightMI.setSelected(reducedTierHeight);
        reducedTierHeightMI.setActionCommand("redTH");
        reducedTierHeightMI.addActionListener(this);
        popup.add(reducedTierHeightMI);

        hScrollBarVisMI = new JCheckBoxMenuItem(ElanLocale.getString(
                "TimeLineViewer.Menu.HScrollBar"));
        hScrollBarVisMI.setSelected(hScrollBarVisible);
        hScrollBarVisMI.setActionCommand("hsVis");
        hScrollBarVisMI.addActionListener(this);
        popup.add(hScrollBarVisMI);
        timeRulerVisMI = new JCheckBoxMenuItem(ElanLocale.getString(
        		"TimeScaleBasedViewer.TimeRuler.Visible"));
        timeRulerVisMI.setSelected(timeRulerVisible);
        timeRulerVisMI.addActionListener(this);
        popup.add(timeRulerVisMI);
        popup.addSeparator();

        timeScaleConMI = new JCheckBoxMenuItem(ElanLocale.getString(
                    "TimeScaleBasedViewer.Connected"), timeScaleConnected);
        timeScaleConMI.setActionCommand("connect");
        timeScaleConMI.addActionListener(this);
        popup.add(timeScaleConMI);
        tickerModeMI = new JCheckBoxMenuItem(ElanLocale.getString(
        		"TimeScaleBasedViewer.TickerMode"));
        tickerModeMI.setSelected(tickerMode);
        tickerModeMI.addActionListener(this);
        popup.add(tickerModeMI);

        popup.addSeparator();

        // tier menu items
        activeTierMI = new JMenuItem(ElanLocale.getString(
                    "Menu.Tier.ActiveTier"));
        activeTierMI.setActionCommand("activeTier");
        activeTierMI.addActionListener(this);
        popup.add(activeTierMI);

        deleteTierMI = new JMenuItem(ElanLocale.getString(
                    "Menu.Tier.DeleteTier"));
        deleteTierMI.setActionCommand("deleteTier");
        deleteTierMI.addActionListener(this);
        popup.add(deleteTierMI);

        changeTierMI = new JMenuItem(ElanLocale.getString(
                    "Menu.Tier.ChangeTier"));

        //changeTierMI = new JMenuItem(ELANCommandFactory.CHANGE_TIER_ATTR);
        changeTierMI.setActionCommand("changeTier");
        changeTierMI.addActionListener(this);
        popup.add(changeTierMI);

        popup.addSeparator();

        // annotation menu items
        newAnnoMI = new JMenuItem(ElanLocale.getString(
                    "Menu.Annotation.NewAnnotation"));
        newAnnoMI.setActionCommand("newAnn");
        newAnnoMI.addActionListener(this);
        popup.add(newAnnoMI);

        newAnnoBeforeMI = new JMenuItem(ELANCommandFactory.getCommandAction(
        		transcription, ELANCommandFactory.NEW_ANNOTATION_BEFORE));
        /* remove after testing
        newAnnoBeforeMI = new JMenuItem(ElanLocale.getString(
                    "Menu.Annotation.NewAnnotationBefore"));
        newAnnoBeforeMI.setActionCommand("annBefore");
        newAnnoBeforeMI.addActionListener(this);
        */
        popup.add(newAnnoBeforeMI);

        newAnnoAfterMI = new JMenuItem(ELANCommandFactory.getCommandAction(
        		transcription, ELANCommandFactory.NEW_ANNOTATION_AFTER));
        /* remove after testing
        newAnnoAfterMI = new JMenuItem(ElanLocale.getString(
                    "Menu.Annotation.NewAnnotationAfter"));
        newAnnoAfterMI.setActionCommand("annAfter");
        newAnnoAfterMI.addActionListener(this);
        */
        popup.add(newAnnoAfterMI);

        modifyAnnoMI = new JMenuItem(ElanLocale.getString(
                    "Menu.Annotation.ModifyAnnotation"));
        modifyAnnoMI.setActionCommand("modifyAnn");
        modifyAnnoMI.addActionListener(this);
        popup.add(modifyAnnoMI);

        modifyAnnoDCMI = new JMenuItem(ELANCommandFactory.getCommandAction(
        		transcription, ELANCommandFactory.MODIFY_ANNOTATION_DC_DLG));
        popup.add(modifyAnnoDCMI);
        mergeAnnoNextMI = new JMenuItem(ELANCommandFactory.getCommandAction(transcription, 
        		ELANCommandFactory.MERGE_ANNOTATION_WN));
        popup.add(mergeAnnoNextMI);
        mergeAnnoBeforeMI = new JMenuItem(ELANCommandFactory.getCommandAction(transcription, 
        		ELANCommandFactory.MERGE_ANNOTATION_WB));
        popup.add(mergeAnnoBeforeMI);
        splitAnnotationMI = new JMenuItem(ElanLocale.getString(ELANCommandFactory.SPLIT_ANNOTATION));
        splitAnnotationMI.setEnabled(false);
        splitAnnotationMI.addActionListener(this);
        popup.add(splitAnnotationMI);
        deleteAnnoValueMI = new JMenuItem(ELANCommandFactory.getCommandAction(
        		transcription, ELANCommandFactory.REMOVE_ANNOTATION_VALUE));
        popup.add(deleteAnnoValueMI);
        
        modifyAnnoTimeMI = new JMenuItem(ELANCommandFactory.getCommandAction(
                    transcription, ELANCommandFactory.MODIFY_ANNOTATION_TIME));
        popup.add(modifyAnnoTimeMI);
        
        deleteAnnoMI = new JMenuItem(ElanLocale.getString(
                    "Menu.Annotation.DeleteAnnotation"));
        deleteAnnoMI.setActionCommand("deleteAnn");
        deleteAnnoMI.addActionListener(this);
        popup.add(deleteAnnoMI);
        
        deleteSelAnnosMI = new JMenuItem(ElanLocale.getString("Menu.Annotation.DeleteSelectedAnnotations"));
        deleteSelAnnosMI.addActionListener(this);       
        popup.add(deleteSelAnnosMI);
        // copy and paste
        popup.addSeparator();
        copyAnnoMI = new JMenuItem(ElanLocale.getString(
        		"Menu.Annotation.CopyAnnotation"));
        copyAnnoMI.addActionListener(this);
        popup.add(copyAnnoMI);
        copyAnnoTreeMI = new JMenuItem(ElanLocale.getString(
        		"Menu.Annotation.CopyAnnotationTree"));
        copyAnnoTreeMI.addActionListener(this);
        popup.add(copyAnnoTreeMI);
        pasteAnnoHereMI = new JMenuItem(ElanLocale.getString(
        		"Menu.Annotation.PasteAnnotationHere"));
        pasteAnnoHereMI.addActionListener(this);
        popup.add(pasteAnnoHereMI);
        pasteAnnoTreeHereMI = new JMenuItem(ElanLocale.getString(
        		"Menu.Annotation.PasteAnnotationTreeHere"));
        pasteAnnoTreeHereMI.addActionListener(this);
        popup.add(pasteAnnoTreeHereMI);
        popup.addSeparator();
        shiftActiveAnnMI = new JMenuItem(ELANCommandFactory.getCommandAction(
        		transcription, ELANCommandFactory.SHIFT_ACTIVE_ANNOTATION));
        shiftActiveAnnMI.setText(ElanLocale.getString("Menu.Annotation.Shift") + " "
        		+ ElanLocale.getString("Menu.Annotation.ShiftActiveAnnotation"));
        popup.add(shiftActiveAnnMI);
        
        JPopupMenu.setDefaultLightWeightPopupEnabled(false);

        int zoom = (int) (100f * (10f / msPerPixel));

        if (zoom <= 0) {
            zoom = 100;
        }

        updateZoomPopup(zoom);
    }

    /**
     * Updates the "zoom" menu item. Needed, when timeScaleConnected, after a
     * change of the zoomlevel in some other connected viewer.
     *
     * @param zoom the zoom level
     */
    private void updateZoomPopup(int zoom) {
    	// 76 == 75%, 166 == 150%
        // first find the closest match (there can be rounding issues)
        int zoomMenuIndex = -1;
//        if (zoom == 76) {
//        	zoom = 75;
//        } else if (zoom == 166) {
//        	zoom = 150;
//        }
        //int diff = Integer.MAX_VALUE;

        for (int i = 0; i < ZOOMLEVELS.length; i++) {
        	/*
            int d = Math.abs(ZOOMLEVELS[i] - zoom);

            if (d < diff) {
                diff = d;
                zoomMenuIndex = i;
            }
            */
        	// compensate for rounding effects
        	if (zoom - ZOOMLEVELS[i] >= -1 && zoom - ZOOMLEVELS[i] <= 1) {
        		zoomMenuIndex = i;
        		break;
        	}
        }

        if (popup != null) {
            java.util.Enumeration en = zoomBG.getElements();
            int counter = 0;

            while (en.hasMoreElements()) {
                JRadioButtonMenuItem rbmi = (JRadioButtonMenuItem) en.nextElement();
                // +1 cause of the "custom" menu item
                if (counter == zoomMenuIndex + 1) { //rbmi.getActionCommand().equals(zoomLevel)
                    rbmi.setSelected(true);
                    
                    break;
                } else {
                    rbmi.setSelected(false);
                }

                counter++;
            }
            if (zoomMenuIndex == -1) {
            	customZoomMI.setSelected(true);
            	customZoomMI.setText(ElanLocale.getString("TimeScaleBasedViewer.Zoom.Custom") + " - " + zoom + "%");
            } else {
            	customZoomMI.setText(ElanLocale.getString("TimeScaleBasedViewer.Zoom.Custom"));
            }
        }
    }
    
    /**
     * Enables / Disables the split annotation option in the 
     * popup
     */
    
    private void checkForSplitAnnotation(){
    	splitAnnotationMI.setEnabled(false);
    	Annotation activeAnnotation = getActiveAnnotation();
    	if(activeAnnotation != null){
    		TierImpl tier = (TierImpl) activeAnnotation.getTier();	    	   
 	  	   	
 	  	   	if(tier.isTimeAlignable() && !tier.hasParentTier()){ 	
 	  	   		Vector childTiers = tier.getChildTiers();
 	  	   		if(childTiers!= null){
 	  	   			boolean valid = false;
 	  	   			for(int i=0; i <childTiers.size(); i++){
 	  	   				TierImpl childTier = (TierImpl)childTiers.get(i);
 	  	   				if(childTier.getLinguisticType().getConstraints().getStereoType() != Constraint.SYMBOLIC_ASSOCIATION){
 	  	   					valid = false;
 	  	   					break;
 	  	   				}
 	  	   				valid = true;
 	  	   			}
 	  	   			splitAnnotationMI.setEnabled(valid);
 	  	   			
 	  	   			if(!valid){
 	  	   				Vector dependentTiers = tier.getDependentTiers();
 	  	   				Vector dependingAnnotations = new Vector();
 	  	   				for(int i=0;i < dependentTiers.size();i++){
 	  	   					if( ((TierImpl) dependentTiers.get(i)).getLinguisticType().getConstraints().getStereoType() == Constraint.SYMBOLIC_ASSOCIATION){
 	  	   						continue; 
 	  	   					}
 	  	   					dependingAnnotations.addAll(activeAnnotation.getChildrenOnTier((Tier) dependentTiers.get(i)));
 	  	   				}	
 	  	   				if(dependingAnnotations.size() == 0){	  			 
 	  	   					splitAnnotationMI.setEnabled(true);
 	  	   				} 
 	  	   			}
 	  	   		}
 	  	   	}
    	}
	 }

    /**
     * Zooms in to the next level of predefined zoomlevels.
     * Note: has to be adapted once custom zoomlevels are implemented.
     */
    private void zoomIn() {
    	float zoom = 100 / ((float) msPerPixel / 10);
    	// temp
    	float nz = zoom + 10;
    	float nm = ((100f / nz) * 10);
    	setMsPerPixel(nm);
    	if (1+1==2) {
    		return;
    	}
    	
        // first find the closest match (there can be rounding issues)
        int zoomMenuIndex = -1;
        int diff = Integer.MAX_VALUE;

        for (int i = 0; i < ZOOMLEVELS.length; i++) {
            int d = Math.abs(ZOOMLEVELS[i] - (int) zoom);

            if (d < diff) {
                diff = d;
                zoomMenuIndex = i;
            }
        }
    	
    	if (zoomMenuIndex > -1 && zoomMenuIndex < ZOOMLEVELS.length - 1) {
    		int nextZoom = ZOOMLEVELS[zoomMenuIndex + 1];
    		float nextMsPerPixel = ((100f / nextZoom) * 10);
    	    setMsPerPixel(nextMsPerPixel);
    		updateZoomPopup(nextZoom);
    	}
    }
   
    /**
     * Zooms in to the next level of predefined zoomlevels.
     * Note: has to be adapted once custom zoomlevels are implemented.
     */
    private void zoomOut() {
    	float zoom = 100 / ((float) msPerPixel / 10);
    	
    	// temp
    	float nz = zoom - 10;
    	if (nz < 10) {
    		nz = 10;
    	}
    	float nm = ((100f / nz) * 10);
    	setMsPerPixel(nm);
    	if (1+1==2) {
    		return;
    	}
        // first find the closest match (there can be rounding issues)
        int zoomMenuIndex = -1;
        int diff = Integer.MAX_VALUE;

        for (int i = 0; i < ZOOMLEVELS.length; i++) {
            int d = Math.abs(ZOOMLEVELS[i] - (int) zoom);

            if (d < diff) {
                diff = d;
                zoomMenuIndex = i;
            }
        }
    	
    	if (zoomMenuIndex > 0) {
    		int nextZoom = ZOOMLEVELS[zoomMenuIndex - 1];
    		float nextMsPerPixel = ((100f / nextZoom) * 10);
    	    setMsPerPixel(nextMsPerPixel);
    		updateZoomPopup(nextZoom);
    	}
    }
    
    private void zoomToSelection() {
    	long selInterval = getSelectionEndTime() - getSelectionBeginTime();
    	if (selInterval < 150) {
    		selInterval = 150;
    	}
    	int sw = imageWidth != 0 ? imageWidth - (2 * SCROLL_OFFSET) : getWidth() - defBarWidth - (2 * SCROLL_OFFSET);
    	float nextMsPP = selInterval / (float) sw;
    	//System.out.println("interval: " + selInterval + " mspp: " + nextMsPP);
    	// set a limit of zoom = 5% or mspp = 200
    	if (nextMsPP > 200) {
    		nextMsPP = 200;
    	}
    	setMsPerPixel(nextMsPP);
    	//customZoomMI.setSelected(true);
    	//customZoomMI.setText(ElanLocale.getString("TimeScaleBasedViewer.Zoom.Custom") + " - " + (int)(100 / ((float) msPerPixel / 10)) + "%");
    	if (!playerIsPlaying()) {
    		long ibt = getSelectionBeginTime() - (long)(SCROLL_OFFSET * msPerPixel);
    		if (ibt < 0) {
    			ibt = 0;
    		}
    		setIntervalBeginTime(ibt);
    	}
        setPreference("TimeLineViewer.ZoomLevel", new Float(100f * (10f / msPerPixel)), 
        		transcription);
    }
    
    /**
     * Enables / disables annotation and tier specific menuitems, depending on
     * the mouse click position.<br>
     * <b>Note: </b> this might need to be changed once usage of Action and
     * Command objects is implemented.
     *
     * @param p the position of the mouse click
     */
    private void updatePopup(Point p) {
        //disable all first
        newAnnoMI.setEnabled(false);
        newAnnoBeforeMI.setEnabled(false);
        newAnnoAfterMI.setEnabled(false);
        modifyAnnoMI.setEnabled(false);
        modifyAnnoDCMI.setEnabled(false);
        modifyAnnoTimeMI.setEnabled(false);
        deleteAnnoValueMI.setEnabled(false);
        deleteAnnoMI.setEnabled(false);
        deleteSelAnnosMI.setEnabled(false);
        activeTierMI.setEnabled(false);
        deleteTierMI.setEnabled(false);
        changeTierMI.setEnabled(false);
        copyAnnoMI.setEnabled(false);
        copyAnnoTreeMI.setEnabled(false);
        pasteAnnoHereMI.setEnabled(false);
        pasteAnnoTreeHereMI.setEnabled(false); 
//        mergeAnnoNextMI.setEnabled(false); 
//        mergeAnnoBeforeMI.setEnabled(false); 
        splitAnnotationMI.setEnabled(false);
        
        splitTime =  0;

        if (timeRulerVisible && (p.y < rulerHeight)) {
            return;
        } else {
            Point inverse = new Point(p);

            //compensate for the intervalBeginTime
            inverse.x += timeToPixels(intervalBeginTime);
            inverse.y += verticalScrollOffset;
            rightClickTime = pixelToTime(inverse.x);

            if (rightClickTime > getMediaDuration()) {
                return;
            }

            boolean supportsInsertion = false;

            TierImpl tier;

            Tag2D tag2d;
            int tierIndex = getTierIndexForPoint(inverse);

            if (tierIndex < 0) {
                return;
            }

            rightClickTier = (Tier2D) visibleTiers.get(tierIndex);
            tier = rightClickTier.getTier();

            if (tier == null) {
                return;
            }

            try {
                LinguisticType lt = tier.getLinguisticType();
                Constraint c = null;

                if (lt != null) {
                    c = lt.getConstraints();
                }

                if (c != null) {
                    supportsInsertion = c.supportsInsertion();
                }
                
                checkForSplitAnnotation(); 
                splitTime = rightClickTime;

                deleteTierMI.setEnabled(true);
                changeTierMI.setEnabled(true);

                if (!rightClickTier.isActive()) {
                    activeTierMI.setEnabled(true);
                }
                if (selectedAnnotations.size() > 0) {
                	deleteSelAnnosMI.setEnabled(true);
                }

                tag2d = getTagAt(inverse, tierIndex);

                if ((tag2d != null) && (tag2d == cursorTag2D)) {
                    modifyAnnoMI.setEnabled(true);
                    modifyAnnoDCMI.setEnabled(true);
                    deleteAnnoValueMI.setEnabled(true);
                    deleteAnnoMI.setEnabled(true);
                    copyAnnoMI.setEnabled(true);
                    copyAnnoTreeMI.setEnabled(true);

                    if (supportsInsertion) {
                        newAnnoAfterMI.setEnabled(true);
                        newAnnoBeforeMI.setEnabled(true);
                    }

                    // removed else here...
                    if (tier.isTimeAlignable()) {
                        newAnnoMI.setEnabled(true); //replace an existing annotation??

                        if (getSelectionBeginTime() != getSelectionEndTime()) {
                            modifyAnnoTimeMI.setEnabled(true);
                        }
                    }
                } else {
                    // check this, much to complicated
                    // this should be based on the Constraints object..
                    if ((getSelectionBeginTime() != getSelectionEndTime()) &&
                            (getSelectionBeginTime() <= rightClickTime) &&
                            (getSelectionEndTime() >= rightClickTime)) {
                        if (tier.isTimeAlignable()) {
                            newAnnoMI.setEnabled(true);
                        } else {
                            if ((tier.getParentTier() != null) &&
                                    ((c.getStereoType() == Constraint.SYMBOLIC_ASSOCIATION) ||
                                    (c.getStereoType() == Constraint.SYMBOLIC_SUBDIVISION))) {
                                Annotation parentA = ((TierImpl) tier.getParentTier()).getAnnotationAtTime(rightClickTime);
                                Annotation refA = tier.getAnnotationAtTime(rightClickTime);

                                if ((parentA != null) && (refA == null)) {
                                    newAnnoMI.setEnabled(true);
                                }
                            }
                        }
                    }
                    // paste items
                    if (AnnotationTransfer.validContentsOnClipboard()) {
                    	if (tier.getParentTier() == null) {
                    		pasteAnnoHereMI.setEnabled(true);
                    		pasteAnnoTreeHereMI.setEnabled(true);
                    	} else {
                            Annotation parentA = ((TierImpl) tier.getParentTier()).getAnnotationAtTime(rightClickTime);
                            Annotation refA = tier.getAnnotationAtTime(rightClickTime);

                            if ((parentA != null) && (refA == null)) {
                            	pasteAnnoHereMI.setEnabled(true);
                        		pasteAnnoTreeHereMI.setEnabled(true);
                            }	
                    	}
                    }
                }
            } catch (Exception rex) {
                rex.printStackTrace();
            }
        }
        // update zoom to selection mi
        zoomSelectionMI.setEnabled(getSelectionBeginTime() != getSelectionEndTime());
    }

    /**
     * Set the inline edit box to invisible by canceling the edit.
     */
    private void dismissEditBox() {
        if (editBox.isVisible()) {
        	   if (deselectCommits) {
               editBox.commitEdit();
        	   } else {
               editBox.cancelEdit();
        	   }
        }
    }
    
    public void setKeyStrokesNotToBeConsumed(List<KeyStroke> ksList){
    	editBox.setKeyStrokesNotToBeConsumed(ksList);
    }
    
    /**
     * Display the edit box for the specified Tag2D
     *
     * @param tag2d the tag to edit
     */
    protected void showEditBoxForTag(Tag2D tag2d) {
        if (tag2d.getAnnotation() == null) {
            return;
        }

        editBox.setAnnotation(tag2d.getAnnotation(),
            forceOpenControlledVocabulary);

        int tierIndex = getTierIndexForAnnotation(tag2d.getAnnotation());
        int y = (rulerHeight + (tierIndex * pixelsForTierHeight)) -
            verticalScrollOffset;
        int x = xAt(tag2d.getBeginTime());

        /* //scroll to the beginning of the tag
           if (x < 0) {
               x = 0;
               setIntervalBeginTime(tag2d.getBeginTime());
           }
         */
        Font f = getFontForTier(tag2d.getTier2D().getTier());
        if (f != null) {
        	editBox.setFont(f);
        }
        editBox.setLocation(x, y);
        editBox.configureEditor(JPanel.class, null,
            new Dimension(tag2d.getWidth() + 2, pixelsForTierHeight));
        editBox.startEdit();

        forceOpenControlledVocabulary = false;
    }

    public void showEditBoxForAnnotation(Annotation ann) {
        if (ann == null) {
            return;
        }

        editBox.setAnnotation(ann);

        int tierIndex = getTierIndexForAnnotation(ann);
        int y = (rulerHeight + (tierIndex * pixelsForTierHeight)) -
            verticalScrollOffset;
        int x = xAt(ann.getBeginTimeBoundary());
        int w = xAt(ann.getEndTimeBoundary()) - x;

        /* //scroll to the beginning of the tag
           if (x < 0) {
               x = 0;
               setIntervalBeginTime(ann.getBeginTimeBoundary());
           }
         */
        // make sure the begin of the edit box is not outside the viewer's area
        if (x < 0) {
        	w = w - (-x) + 2;
        	x = 2;
        }
        //System.out.println("x: " + x + "w: " + w);
        Font f = getFontForTier(ann.getTier());
        if (f != null) {
        	editBox.setFont(f);
        }
        editBox.setLocation(x, y);
        editBox.configureEditor(JPanel.class, null,
            new Dimension(w + 2, pixelsForTierHeight));
        editBox.startEdit();
        
    }

    // ***** editing and data changed methods **************************************//

    /**
     * Change a single annotation.
     *
     * @param tier the tier the annotation is part of
     * @param ann the annotation that has been changed
     */
    private void annotationChanged(TierImpl tier, Annotation ann) {
        Iterator allTierIt = allTiers.iterator();
        dismissEditBox();
alltierloop: 
        while (allTierIt.hasNext()) {
            Tier2D t2d = (Tier2D) allTierIt.next();

            if (t2d.getTier() == tier) {
                ArrayList tagList = t2d.getTagsList();
alltagloop: 
                for (int i = 0; i < tagList.size(); i++) {
                    Tag2D tag2d = (Tag2D) tagList.get(i);

                    // check equality with ==
                    if (tag2d.getAnnotation() == ann) {
                        tag2d.setTruncatedValue(truncateString(ann.getValue(),
                                tag2d.getWidth(), metrics));
                        // check CV entry color
                        ControlledVocabulary cv = transcription.getControlledVocabulary(
                        		tier.getLinguisticType().getControlledVocabylaryName());
                        if (cv != null) {
                        	CVEntry[] entries = cv.getEntries();
                            if (entries != null) {
                            	for (CVEntry e : entries) {
                            		if (e.getValue().equals(ann.getValue())) {
                            			tag2d.setColor(e.getPrefColor());
                            			
                            			// check a parent annotation for which this annotation could be a color marker
                            			TierImpl parTier = (TierImpl) tier.getParentTier();
                            			if (parTier != null && tier == TierAssociation.findMarkerTierFor(transcription, parTier)) {
                            				parentTierLoop:
                            				for (Tier2D pt : allTiers) {
                            					if (pt.getTier() == parTier) {
                            						Tag2D pa2d;
                            						Iterator tagIt = pt.getTags();
                            						while (tagIt.hasNext()) {
                            							pa2d = (Tag2D) tagIt.next();
                            							if (pa2d.getBeginTime() == tag2d.getBeginTime() && 
                            									pa2d.getEndTime() == tag2d.getEndTime()) {
                            								pa2d.setColor(e.getPrefColor());
                            								break;
                            							}
                            							if (pa2d.getBeginTime() > tag2d.getBeginTime()) {
                            								break;
                            							}
                            						}
                            						break parentTierLoop;
                            					}
                            				}
                            			}
                            			break;
                            		}
                            	}
                            }
                        }
                        
                        break alltierloop;
                    }
                }
            }
        }

        paintBuffer();
    }

    /**
     * Add a new Tier to the existing list of tiers.<br>
     * This method is private because it does not check whether the specified
     * Tier already is present in the transcription.
     *
     * @param tier the new Tier
     */
    private void tierAdded(TierImpl tier) {
        Tag2D tag2d;
        int xPos;
        int tagWidth;

        Tier2D tier2d = new Tier2D(tier);
        Iterator annotIter = tier.getAnnotations().iterator();

        while (annotIter.hasNext()) {
            Annotation a = (Annotation) annotIter.next();

            //System.out.println("Annotation: " + a);
            tag2d = new Tag2D(a);
            xPos = timeToPixels(a.getBeginTimeBoundary());
            tag2d.setX(xPos);
            tagWidth = timeToPixels(a.getEndTimeBoundary()) - xPos;
            tag2d.setWidth(tagWidth);
            tag2d.setTruncatedValue(truncateString(a.getValue(), tagWidth,
                    metrics));
            tier2d.addTag(tag2d);
        }

        allTiers.add(tier2d);
        tierYPositions = new int[allTiers.size()];

        //wait for a call to setVisibleTiers to show the tier
        //visibleTiers.add(tier2d);
        //paintBuffer();
        //System.out.println("new tier: " + tier2d.getName());
    }

    /**
     * Remove a Tier from the list of tiers.
     *
     * @param tier the Tier to remove
     */
    private void tierRemoved(TierImpl tier) {
        dismissEditBox();

        for (int i = 0; i < allTiers.size(); i++) {
            Tier2D tier2d = (Tier2D) allTiers.get(i);

            if (tier2d.getTier() == tier) {
                allTiers.remove(i);
                prefTierFonts.remove(tier.getName());

                //wait for a call to setVisibleTiers
                if ((cursorTag2D != null) &&
                        (cursorTag2D.getTier2D() == tier2d)) {
                    cursorTag2D = null;
                    setActiveAnnotation(null);
                }

                break;
            }
        }

        //repaint();
    }

    /**
     * Check the name of the Tier2D object of the specified Tier.<br>
     * Other changes to the tier such as linguistic type or parent are
     * expected to become manifest through (a series of) changes in
     * annotations.
     *
     * @param tier the Tier that has been changed
     */
    private void tierChanged(TierImpl tier) {
        Vector depTiers = new Vector();

        if (tier != null) {
            depTiers = tier.getDependentTiers();
        }

        for (int i = 0; i < allTiers.size(); i++) {
            Tier2D tier2d = (Tier2D) allTiers.get(i);

            if (tier2d.getTier() == tier) {
            	if (!tier2d.getName().equals(tier.getName())) {
            		Object opf = prefTierFonts.remove(tier2d.getName());
            		if (opf != null) {
            			prefTierFonts.put(tier.getName(), opf);
            		}
            	}
                tier2d.updateName();               
            }

            if ((tier2d.getTier() == tier) ||
                    depTiers.contains(tier2d.getTier())) {
                reextractTagsForTier(tier2d);
            }
        }
    }

    /**
     * Create a Tag2D for a new Annotation on the Tier2D of the specified TierImpl.<br>
     * If the Transcription is in Bulldozer mode, reextract the Tier2D.
     * Correction: transcription does not have a bulldozer mode (yet), just
     * reextract the tier...
     *
     * @param tiers the Tier the annotation belongs to
     */

    /*
       private void annotationAdded(TierImpl tier, Annotation annotation) {
           dismissEditBox();
    
               for (int i = 0; i < allTiers.size(); i++) {
                   Tier2D tier2d = (Tier2D) allTiers.get(i);
    
                   if (tier2d.getTier() == tier) {
                       reextractTagsForTier(tier2d);
    
                       paintBuffer();
    
                       break;
                   }
               }
           }
     */

    /**
     * Called when an annotation has been added before or after another
     * annotation, effecting more than one or two annotations.
     *
     * @param tiers a Vector of tiers that have been changed
     */
    private void annotationsAdded(Vector tiers) {
        int mode = transcription.getTimeChangePropagationMode();

        if (mode != Transcription.SHIFT) {
            Tier2D tier2d;
            dismissEditBox();

            for (int i = 0; i < allTiers.size(); i++) {
                tier2d = (Tier2D) allTiers.get(i);

                if (tiers.contains(tier2d.getTier())) {
                    reextractTagsForTier(tier2d);
                }
            }

            paintBuffer();
        } else {
            transcriptionChanged();
        }
    }

    /**
     * This method inserts a new annotation if there is an empty slot at the
     * specified point on the tier for the specified index.
     *
     * @param p the location of a doubleclick
     * @param tierIndex tier index
     */
    private void autoInsertAnnotation(Point p, int tierIndex) {
        if ((tierIndex < 0) || (tierIndex > (visibleTiers.size() - 1))) {
            return;
        }

        TierImpl child = ((Tier2D) visibleTiers.get(tierIndex)).getTier();

        if (child == null) {
            return;
        }

        long clickTime = pixelToTime((int) p.getX());

        if (child.isTimeAlignable() || !child.hasParentTier()) {
            if ((clickTime >= getSelectionBeginTime()) &&
                    (clickTime <= getSelectionEndTime())) {
            	Command c = null;
                Object val = Preferences.get("CreateDependingAnnotations", null);                
                if (val instanceof Boolean) {
                	 if(((Boolean) val).booleanValue()){
                		 c = ELANCommandFactory.createCommand(transcription,
                                 ELANCommandFactory.NEW_ANNOTATION_REC);  
                		 
                	 }else{
                		 c = ELANCommandFactory.createCommand(transcription,
                                 ELANCommandFactory.NEW_ANNOTATION);                		 
                	 }
                } else {
                	 c = ELANCommandFactory.createCommand(transcription,
                             ELANCommandFactory.NEW_ANNOTATION); 
                }
                Object[] args = new Object[] {
                        new Long(getSelectionBeginTime()),
                        new Long(getSelectionEndTime())
                    };
                c.execute(child, args);
            }
        } else {
            TierImpl parent = (TierImpl) child.getParentTier();

            Annotation ann = parent.getAnnotationAtTime(clickTime);

            if (ann != null) {
            	Command c = null;
                Object val = Preferences.get("CreateDependingAnnotations", null);                
                if (val instanceof Boolean) {
                	 if(((Boolean) val).booleanValue()){
                		 c = ELANCommandFactory.createCommand(transcription,
                                 ELANCommandFactory.NEW_ANNOTATION_REC);  
                		 
                	 }else{
                		 c = ELANCommandFactory.createCommand(transcription,
                                 ELANCommandFactory.NEW_ANNOTATION);                		 
                	 }
                } else {
                	c = ELANCommandFactory.createCommand(transcription,
                         ELANCommandFactory.NEW_ANNOTATION); 
                }
                Object[] args = new Object[] {
                        new Long(clickTime),
                        new Long(clickTime)
                    };
                c.execute(child, args);
            }
        }
    }

    /**
     * Remove the Tag2D from the Tier2D corresponding to the respective
     * Annotation and TierImpl.
     */

    /*
       private void annotationRemoved(TierImpl tier, Annotation annotation) {
           Tier2D tier2d;
           Tag2D tag2d;
    
               for (int i = 0; i < allTiers.size(); i++) {
                   tier2d = (Tier2D) allTiers.get(i);
    
                   if (tier2d.getTier() == tier) {
                       Iterator tagIt = tier2d.getTags();
    
                       while (tagIt.hasNext()) {
                           tag2d = (Tag2D) tagIt.next();
    
                           if (tag2d.getAnnotation() == annotation) {
                               dismissEditBox();
                               tier2d.removeTag(tag2d);
    
                               return;
                           }
                       }
                   }
               }
           }
     */

    /**
     * This is called when an ACMEditEvent is received with operation
     * REMOVE_ANNOTATION and the transcription as invalidated object.<br>
     * It is undefined which tiers and annotations have been effected, so the
     * transcription is simply re-processed. Store state as much as possible.
     * Assume no tiers have been deleted or added.
     */
    private void annotationsRemoved() {
        transcriptionChanged();
    }

    /**
     * Called when begin and/or end time of an alignable annotation  has been
     * changed. In shift time propagation mode all tiers  are reextracted, in
     * other modes only the tiers that can be effected  are reextracted.
     *
     * @param tiers the vector of tiers that could be effected by the change
     */
    private void annotationTimeChanged(Vector tiers) {
        int mode = transcription.getTimeChangePropagationMode();

        if (mode != Transcription.SHIFT) {
            Tier2D tier2d;

            for (int i = 0; i < allTiers.size(); i++) {
                tier2d = (Tier2D) allTiers.get(i);

                if (tiers.contains(tier2d.getTier())) {
                    reextractTagsForTier(tier2d);
                }
            }

            paintBuffer();
        } else {
            transcriptionChanged();
        }
    }

    /**
     * This is called when an ACMEditEvent is received with an operation that
     * could influence all tiers in the transcription or with the
     * transcription as invalidated object.<br>
     * Examples are annotations_removed or annotation_added and
     * annotation_time_changed  in shift mode. The transcription is simply
     * re-processed. Store state as much as possible. Assume no tiers have
     * been deleted or added.
     */
    private void transcriptionChanged() {
        cursorTag2D = null;

        Tier2D tier2d;
        dismissEditBox();

        for (int i = 0; i < allTiers.size(); i++) {
            tier2d = (Tier2D) allTiers.get(i);
            reextractTagsForTier(tier2d);
        }

        if (cursorTag2D != null) {
            cursorTierIndex = visibleTiers.indexOf(cursorTag2D.getTier2D());
        }

        paintBuffer();
    }

    /**
     * Tries to retrieve the default Font size for tooltips.<br>
     * This can then be used when the changing the Font the tooltip has to
     * use.
     *
     * @return the default font size or 12 when not found
     */
    private int getDefaultTooltipFontSize() {
        Object value = UIManager.getDefaults().get("ToolTip.font");

        if ((value != null) && value instanceof Font) {
            return ((Font) value).getSize();
        }

        return 12;
    }
    
    /**
     * Returns the user defined preferred font for the tier, if there is one.
     *  
     * @param tier the tier the font is to be used for
     * @return the preferred font, if there is one, otherwise the default font
     */
    private Font getFontForTier(Tier tier) {
    	if (tier != null) {
    		Font fo = (Font) prefTierFonts.get(tier.getName());
    		if (fo != null) {
    			return fo;
    		}
    	}
    	
    	return font;
    }

    /**
     * Override create tooltip to be able to set the (Unicode) Font for the
     * tip.
     *
     * @return DOCUMENT ME!
     */
    public JToolTip createToolTip() {
        JToolTip tip = new JToolTip();
        tip.setFont(tooltipFont);

        //tip.setFont(tooltipFont != null ? tooltipFont : this.getFont().deriveFont((float)tooltipFontSize));
        tip.setComponent(this);

        return tip;
    }

    /**
     * Set a new Transcription for this viewer.<br>
     * We should receive a setVisibleTiers() and a setActiveTier() call after
     * this but are faking it now.
     *
     * @param transcription the new transcription.
     */
    public void setTranscription(Transcription transcription) {
        this.transcription = (TranscriptionImpl) transcription;
        hoverTag2D = null;
        hoverTierIndex = 0;
        cursorTag2D = null;
        cursorTierIndex = 0;

        //
        Vector oldVisibles = new Vector(visibleTiers.size());

        for (int i = 0; i < visibleTiers.size(); i++) {
            Tier2D tier2d = (Tier2D) visibleTiers.get(i);
            oldVisibles.add(tier2d.getTier());
        }

        String activeTierName = "";

        for (int i = 0; i < allTiers.size(); i++) {
            Tier2D tier2d = (Tier2D) allTiers.get(i);

            if (tier2d.isActive()) {
                activeTierName = tier2d.getName();

                break;
            }
        }

        //
        initTiers();

        //
        for (int i = 0; i < allTiers.size(); i++) {
            Tier2D tier2d = (Tier2D) allTiers.get(i);

            if (tier2d.getName().equals(activeTierName)) {
                tier2d.setActive(true);

                break;
            }
        }

        setVisibleTiers(oldVisibles);
    }

    //***** end of initial editing and data changed methods **************************************//
    //**************************************************************************************//

    /* implement ControllerListener */
    /* (non-Javadoc)
     * @see mpi.eudico.client.annotator.ControllerListener#controllerUpdate(mpi.eudico.client.annotator.ControllerEvent)
     */
    public void controllerUpdate(ControllerEvent event) {
        if (event instanceof TimeEvent || event instanceof StopEvent) {
            crossHairTime = getMediaTime();

            /*
               //System.out.println("TimeLineViewer time: " + crossHairTime);
               if (crossHairTime < intervalBeginTime || crossHairTime > intervalEndTime) {
                   if (playerIsPlaying()) {
                       // we might be in a selection outside the new interval
                       long newBeginTime = intervalEndTime;
                       long newEndTime = newBeginTime + (imageWidth * msPerPixel);
                       if (crossHairTime > newEndTime) {
                           while ((newEndTime += imageWidth + msPerPixel) < crossHairTime) {
                               newBeginTime += imageWidth * msPerPixel;
                           }
                       } else if (crossHairTime < newBeginTime) {
                           while ((newEndTime -= imageWidth * msPerPixel) > crossHairTime) {
                               newBeginTime -= imageWidth * msPerPixel;
                           }
                       }
                       setIntervalBeginTime(newBeginTime);
                   } else {
                       setIntervalBeginTime(crossHairTime);
                   }
                   crossHairPos = xAt(crossHairTime);
             */
            /*// replaced 07 apr 2005 better positioning of search result annotations
               if ((crossHairTime == intervalEndTime) && !playerIsPlaying()) {
                   recalculateInterval(crossHairTime);
               } else if ((crossHairTime < intervalBeginTime) ||
                       (crossHairTime > intervalEndTime)) {
                   //dismissEditBox();
                   recalculateInterval(crossHairTime);
               } */
            if (!playerIsPlaying()) {
                if (scroller == null) {
                    recalculateInterval(crossHairTime);
                    crossHairPos = xAt(crossHairTime);
                    repaint();
                } else {
                    recalculateInterval(crossHairTime);
                }
            } else {
            	if (tickerMode) {
                    long intervalMidTime = (intervalBeginTime + intervalEndTime) / 2;

                    if (crossHairTime > (intervalMidTime + (1 * msPerPixel))) {
                        setIntervalBeginTime(intervalBeginTime +
                            (crossHairTime - intervalMidTime));
                    } else if (crossHairTime < intervalMidTime) {
                    	if (crossHairTime < intervalBeginTime) {
                    		setIntervalBeginTime(Math.max(0, crossHairTime - 
                    				(intervalMidTime - intervalBeginTime)));
                    	} else {
	                        int oldPos = crossHairPos;
	                        crossHairPos = xAt(crossHairTime);
	
	                        if (crossHairPos >= oldPos) {
	                            repaint(oldPos - 2, 0, crossHairPos - oldPos + 4,
	                                getHeight());
	                        } else {
	                            repaint(crossHairPos - 2, 0, oldPos - crossHairPos + 4,
	                                getHeight());
	                        }
                    	}
                    } else {
                        repaint();
                    }
            	} else if ((crossHairTime < intervalBeginTime) ||
                        (crossHairTime > intervalEndTime)) {
                    //dismissEditBox();
                    recalculateInterval(crossHairTime);
                } else {
                    // repaint a part of the viewer
                    int oldPos = crossHairPos;
                    crossHairPos = xAt(crossHairTime);

                    int newPos = crossHairPos;

                    if (newPos >= oldPos) {
                        repaint(oldPos - 2, 0, newPos - oldPos + 4, getHeight());

                        //repaint();
                    } else {
                        repaint(newPos - 2, 0, oldPos - newPos + 4, getHeight());

                        //repaint();
                    }
                }
            }
            if (event instanceof StopEvent) {
            	isPlaying = false;
            	paintBuffer();
            }
        } else if (event instanceof StartEvent) {
        	isPlaying = true;
        	if (!useBufferedImage) {
        		paintBuffer();	
        	}     	
        }
    }

    /**
     * Update method from ActiveAnnotationUser.<br>
     * The application wide active annotation corresponds to the cursorTag2D
     * in this viewer. If the Tier that the cursorTag2D belongs to is
     * invisible the cursorTag2D is <i>not</i> set to <code>null</code>.
     */
    public void updateActiveAnnotation() {
        dismissEditBox();

        Annotation anno = getActiveAnnotation();

        if (anno != null) {
            //look for the annotation
            Tier2D tier2d;
            Tag2D tag2d;
            Iterator allIter = allTiers.iterator();

            while (allIter.hasNext()) {
                tier2d = (Tier2D) allIter.next();

                if (tier2d.getTier() == anno.getTier()) {
                    Iterator tagIter = tier2d.getTags();

                    while (tagIter.hasNext()) {
                        tag2d = (Tag2D) tagIter.next();

                        if (tag2d.getAnnotation() == anno) {
                            cursorTag2D = tag2d;
                            cursorTierIndex = visibleTiers.indexOf(cursorTag2D.getTier2D());
                            ensureVerticalVisibilityOfActiveAnnotation();

                            break;
                        }
                    }
                }
            }
            
            long beginTime = anno.getBeginTimeBoundary();
            long endTime = anno.getEndTimeBoundary();

            // update interval //
            if (!playerIsPlaying()) {
            	long newBeginTime = intervalBeginTime;
                long newEndTime = intervalEndTime;
                
                boolean updateInterval = false;
                
                if(centerAnnotation){
                	// always center the active annotation
                	long intMid = (long) ((imageWidth * msPerPixel) / 2);
                    long annMid = (endTime + beginTime) / 2;
                    newBeginTime = annMid - intMid;
                    newEndTime = annMid + intMid;                    
                    if (newBeginTime < 0) {
                        newBeginTime = 0;
                        newEndTime = (long) (imageWidth * msPerPixel);
                    }                    
                    updateInterval = true;
                } else if ((beginTime < intervalBeginTime) ||
                        (endTime > intervalEndTime)) {
                	// if the next annotation is not seen in the current interval, 
                	// update the interval
                        
                    if ((beginTime < newBeginTime) &&
                            (beginTime > (SCROLL_OFFSET * msPerPixel))) {
                        newBeginTime = beginTime -
                            (int) (SCROLL_OFFSET * msPerPixel);
                        newEndTime = newBeginTime + (int) (imageWidth * msPerPixel);
                    } else if (endTime > newEndTime) {
                        newEndTime = endTime +
                            (int) (SCROLL_OFFSET * msPerPixel);  
                        newBeginTime = newEndTime - (int) (imageWidth * msPerPixel);

                        if ((newBeginTime > beginTime) &&
                                (beginTime > (SCROLL_OFFSET * msPerPixel))) {
                            newBeginTime = beginTime -
                                (int) (SCROLL_OFFSET * msPerPixel);
                            newEndTime = newBeginTime +
                                (int) (imageWidth * msPerPixel);
                        } else if (newBeginTime > beginTime) {
                            newBeginTime = 0;
                            newEndTime = (long) (imageWidth * msPerPixel);
                        }                        
                        updateInterval = true;
                    }
                }                 
                if (timeScaleConnected && updateInterval) {                    	
                    setGlobalTimeScaleIntervalBeginTime(newBeginTime);
                    setGlobalTimeScaleIntervalEndTime(newEndTime);
                } else {
                    setLocalTimeScaleIntervalBeginTime(newBeginTime);
                }
            }

            // end update interval //
        } else {
            cursorTag2D = null;
        }
        repaint();
    }

    /**
     * Implements ACMEditListener.<br>
     * The ACMEditEvent that is received contains information about the kind
     * of modification and the objects effected by that modification.
     *
     * @param e the event object
     *
     * @see ACMEditEvent
     */
    public void ACMEdited(ACMEditEvent e) {
        //System.out.println("ACMEdited:: operation: " + e.getOperation() + ", invalidated: " + e.getInvalidatedObject());
        //System.out.println("\tmodification: " + e.getModification() + ", source: " + e.getSource());
        switch (e.getOperation()) {
        case ACMEditEvent.ADD_TIER:

            if (e.getModification() instanceof TierImpl) {
                tierAdded((TierImpl) e.getModification());

                if (multiTierControlPanel != null) {
                    multiTierControlPanel.tierAdded((TierImpl) e.getModification());
                }
            }
            break;

        case ACMEditEvent.REMOVE_TIER:

            if (e.getModification() instanceof TierImpl) {
                tierRemoved((TierImpl) e.getModification());

                if (multiTierControlPanel != null) {
                    multiTierControlPanel.tierRemoved((TierImpl) e.getModification());
                }
            }

            break;

        case ACMEditEvent.CHANGE_TIER:

            if (e.getInvalidatedObject() instanceof TierImpl) {
                tierChanged((TierImpl) e.getInvalidatedObject());

                if (multiTierControlPanel != null) {
                    multiTierControlPanel.tierChanged((TierImpl) e.getInvalidatedObject());
                }
            }

            break;

        // if i'm right for the next three operations the event's
        // invalidated Object should be a tier
        // and the modification object the new annotation...
        case ACMEditEvent.ADD_ANNOTATION_HERE:

            if (e.getInvalidatedObject() instanceof TierImpl &&
                    e.getModification() instanceof Annotation) {
                //annotationAdded((TierImpl)e.getInvalidatedObject(), (Annotation)e.getModification());
                // to accomodate right updates in bulldozer mode stupidly reextract all dependant tiers
                TierImpl invTier = (TierImpl) e.getInvalidatedObject();
                Vector depTiers = invTier.getDependentTiers();

                if (depTiers == null) {
                    depTiers = new Vector();
                }

                depTiers.add(0, invTier);
                annotationsAdded(depTiers);
                // does this lead to exceptions in TextViewer??
                // this cannot be done in the new annotation command, then the inline edit box doesn't popup
                //setActiveAnnotation((Annotation) e.getModification());
                delayedAnnotationActivation = true;
                lastCreatedAnnotation = (Annotation) e.getModification();
                showEditBoxForAnnotation((Annotation) e.getModification());

                if (editBox.isVisible()) {
                    editBox.requestFocus();
                }
                
                if (multiTierControlPanel != null) {
                    multiTierControlPanel.annotationsChanged();
                }
            }

            break;

        case ACMEditEvent.ADD_ANNOTATION_BEFORE:

        // fall through
        //break;
        case ACMEditEvent.ADD_ANNOTATION_AFTER:

            if (e.getInvalidatedObject() instanceof TierImpl &&
                    e.getModification() instanceof Annotation) {
                /*
                   TierImpl invTier = (TierImpl) e.getInvalidatedObject();
                   Vector depTiers = invTier.getDependentTiers(userIdentity);
                
                                   if (depTiers == null) {
                                       depTiers = new Vector();
                                   }
                
                                   depTiers.add(0, invTier);
                                   annotationsAdded(depTiers);
                 */

                //setActiveAnnotation((Annotation) e.getModification());
                delayedAnnotationActivation = true;
                lastCreatedAnnotation = (Annotation) e.getModification();
                // jul 2004: redo all; we can not rely on the fact that only dependent
                // tiers will be effected by this operation...
                // (problem: unaligned annotations on time-subdivision tiers)
                Vector tiers = transcription.getTiers();

                annotationsAdded(tiers);
                showEditBoxForAnnotation((Annotation) e.getModification());

                if (editBox.isVisible()) {
                    editBox.requestFocus();
                }
                
                if (multiTierControlPanel != null) {
                    multiTierControlPanel.annotationsChanged();
                }

                //maybe this could be more finegrained by re-extracting only all
                // RefAnnotations referring to the parent of the modified annotation...
            }

            break;

        case ACMEditEvent.CHANGE_ANNOTATIONS:

            if (e.getInvalidatedObject() instanceof Transcription) {
                transcriptionChanged();
                
                if (multiTierControlPanel != null) {
                    multiTierControlPanel.annotationsChanged();
                }
            }

            break;

        case ACMEditEvent.REMOVE_ANNOTATION:

            if (e.getInvalidatedObject() instanceof Transcription) {
                //System.out.println("Invalidated object: " + e.getInvalidatedObject());
                annotationsRemoved();
                
                if (multiTierControlPanel != null) {
                    multiTierControlPanel.annotationsChanged();
                }
            }

            break;

        case ACMEditEvent.CHANGE_ANNOTATION_TIME:

            if (e.getInvalidatedObject() instanceof AlignableAnnotation) {
                TierImpl invTier = (TierImpl) ((AlignableAnnotation) e.getInvalidatedObject()).getTier();
                Vector depTiers = invTier.getDependentTiers();

                if (depTiers == null) {
                    depTiers = new Vector();
                }

                depTiers.add(0, invTier);
                annotationTimeChanged(depTiers);
            }

            break;

        case ACMEditEvent.CHANGE_ANNOTATION_VALUE:

            if (e.getSource() instanceof Annotation) {
                Annotation a = (Annotation) e.getSource();

                if (a.getTier() instanceof TierImpl) {
                    annotationChanged((TierImpl) a.getTier(), a);

                    // The TimeLineViewer has a special responsibility in showing an
                    // edit box when an annotation has been created.
                    // When in an undo action a deleted annotation is recreated 
                    // and the value restored in a single pass, no component seems
                    // to actually have keyboard focus: no keyboard shortcur works
                    // (can't explain this)
                    // the following code ensures that after finishing the undo action
                    // a component receives the keyboard focus in a separate thread

                    /*
                       SwingUtilities.invokeLater(new Runnable(){
                           public void run() {
                               TimeLineViewer.this.getParent().requestFocus();
                           }
                       });
                     */
                }
            }

            break;

        default:
            break;
        }
        
        // simply remove all selected annotations
        selectedAnnotations.clear();
    }

    //**************************************************************************************//

    /* implement SelectionUser */
    /* (non-Javadoc)
     * @see mpi.eudico.client.annotator.SelectionUser#updateSelection()
     */
    public void updateSelection() {
        //selectionBeginPos = (int) (getSelectionBeginTime() / msPerPixel);
        //selectionEndPos = (int) (getSelectionEndTime() / msPerPixel);
        selectionBeginPos = xAt(getSelectionBeginTime());
        selectionEndPos = xAt(getSelectionEndTime());
        paintBuffer();

        //repaint();
    }

    //**************************************************************************************//

    /* implement ElanLocaleListener */

    /**
     * Update locale sensitive UI elements.
     */
    public void updateLocale() {
        if (popup != null) {
            zoomMI.setText(ElanLocale.getString("TimeScaleBasedViewer.Zoom"));
            timeScaleConMI.setText(ElanLocale.getString(
                    "TimeScaleBasedViewer.Connected"));
            activeAnnStrokeBoldMI.setText(ElanLocale.getString(
                    "TimeLineViewer.ActiveAnnotationBold"));
            reducedTierHeightMI.setText(ElanLocale.getString("TimeLineViewer.ReducedTierHeight"));
            hScrollBarVisMI.setText(ElanLocale.getString("TimeLineViewer.Menu.HScrollBar"));
            fontMenu.setText(ElanLocale.getString("Menu.View.FontSize"));
            activeTierMI.setText(ElanLocale.getString("Menu.Tier.ActiveTier"));
            deleteTierMI.setText(ElanLocale.getString("Menu.Tier.DeleteTier"));
            changeTierMI.setText(ElanLocale.getString(
                    "Menu.Tier.ChangeTier"));
            newAnnoMI.setText(ElanLocale.getString("Menu.Annotation.NewAnnotation"));
            newAnnoBeforeMI.setText(ElanLocale.getString(
                    "Menu.Annotation.NewAnnotationBefore"));
            newAnnoAfterMI.setText(ElanLocale.getString(
                    "Menu.Annotation.NewAnnotationAfter"));
            modifyAnnoMI.setText(ElanLocale.getString(
                    "Menu.Annotation.ModifyAnnotation"));
            //modifyAnnoDCMI.setText(ElanLocale.getString("Menu.Annotation.ModifyAnnotationDatCat"));
            copyAnnoMI.setText(ElanLocale.getString(
    			"Menu.Annotation.CopyAnnotation"));
            copyAnnoTreeMI.setText(ElanLocale.getString(
    			"Menu.Annotation.CopyAnnotationTree"));
            pasteAnnoHereMI.setText(ElanLocale.getString(
    			"Menu.Annotation.PasteAnnotationHere"));
            pasteAnnoTreeHereMI.setText(ElanLocale.getString(
    			"Menu.Annotation.PasteAnnotationTreeHere"));

            //modifyAnnoTimeMI.setText(ElanLocale.getString(
            //       "Menu.Annotation.ModifyAnnotationTime"));
            deleteAnnoMI.setText(ElanLocale.getString(
                    "Menu.Annotation.DeleteAnnotation"));
            shiftActiveAnnMI.setText(ElanLocale.getString("Menu.Annotation.Shift") + " "
            		+ ElanLocale.getString("Menu.Annotation.ShiftActiveAnnotation"));
        }

        if (editBox != null) {
            editBox.updateLocale();
        }
    }

    /**
     * Updates the font that is used in the visualization of the annotations<br>
     * Does not change the font in the time ruler.
     *
     * @param f the new Font
     */
    public void updateFont(Font f) {
        int oldSize = font.getSize();
        font = f;
        setFont(font);
        tooltipFont = font.deriveFont((float) tooltipFontSize);
        metrics = getFontMetrics(font);
        
        Iterator keyIt = prefTierFonts.keySet().iterator();
        String key = null;
        Font prFont = null;
        while (keyIt.hasNext()) {
        	key = (String) keyIt.next();
        	prFont = (Font) prefTierFonts.get(key);
        	if (prFont != null) {
        		prefTierFonts.put(key, new Font(prFont.getName(), Font.PLAIN, 
        				font.getSize()));
        	}
        }
        
        recalculateTagSizes();
        //pixelsForTierHeight = font.getSize() * 3;
        //pixelsForTierHeight = font.getSize() + 24;
        pixelsForTierHeight = calcTierHeight();
        
        //pixelsForTierHeight = pixelsForTierHeight / 2 + 4;
        if (oldSize != f.getSize()) {
            notifyMultiTierControlPanel();
            paintBuffer();
            scrollBar.setBlockIncrement(pixelsForTierHeight);
            updateScrollBar();
        } else {
            paintBuffer();
        }
    }

    /**
     * Sets the font size.
     *
     * @param fontSize the new font size
     */
    public void setFontSize(int fontSize) {
        updateFont(getFont().deriveFont((float) fontSize));

        if (popup != null) {
            Enumeration en = fontSizeBG.getElements();
            JMenuItem item;
            String value;

            while (en.hasMoreElements()) {
                item = (JMenuItem) en.nextElement();
                value = item.getText();

                try {
                    int v = Integer.parseInt(value);

                    if (v == fontSize) {
                        item.setSelected(true);

                        //updateFont(getFont().deriveFont((float) fontSize));
                        break;
                    }
                } catch (NumberFormatException nfe) {
                    //// do nothing
                }
            }
        }
    }

    /**
     * Returns the current font size.
     *
     * @return the current font size
     */
    public int getFontSize() {
        return font.getSize();
    }

    /**
     * Handle scrolling of the viewer image.
     *
     * @param e DOCUMENT ME!
     */
    public void adjustmentValueChanged(AdjustmentEvent e) {
        int value = e.getValue();

        if (e.getSource() == scrollBar) {	        
        if (editBox.isVisible()) {
            Point p = editBox.getLocation();
            p.y += (verticalScrollOffset - value);
            editBox.setLocation(p);
        }

        setVerticalScrollOffset(value);
        notifyMultiTierControlPanel();
        } else if (e.getSource() == hScrollBar) {
            	// editbox is taken care of in setIntervalBeginTime
	        setIntervalBeginTime(pixelToTime(value));
    }
    }
    
	/**
	 * Zoom slider listener.
	 */
	public void stateChanged(ChangeEvent e) {
		if (e.getSource() == zoomSlider) {
			//if (!zoomSlider.getValueIsAdjusting()) {
				int zoomValue = zoomSlider.getValue();
				if (zoomValue > 100) {
					float factor = (zoomValue - 100) / (float) (zoomSlider.getMaximum() - 100);
					zoomValue = 100 + (int) (factor * 900);
				}
				float nmspp = ((100f / zoomValue) * DEFAULT_MS_PER_PIXEL);
				if (nmspp != msPerPixel) {
					setMsPerPixel(nmspp);
				}
				zoomSlider.setToolTipText(String.valueOf(zoomValue));
			//}
		}		
	}

    //**************************************************************************************//

    /* implement MultiTierViewer */
    public void setVisibleTiers(List tiers) {
        //store some old values
        dismissEditBox();

        int oldNum = visibleTiers.size();

        synchronized (tierLock) {
            visibleTiers.clear();

            Tier2D t2d;
            Tier tier;

            Iterator tierIter = tiers.iterator();

            while (tierIter.hasNext()) {
                tier = (Tier) tierIter.next();

                Iterator it = allTiers.iterator();

                while (it.hasNext()) {
                    t2d = (Tier2D) it.next();

                    if (t2d.getTier() == tier) {
                        visibleTiers.add(t2d);

                        break;
                    }
                }
            }
        }

        if (cursorTag2D != null) {
            cursorTierIndex = visibleTiers.indexOf(cursorTag2D.getTier2D());
        }

        notifyMultiTierControlPanel();
        paintBuffer();

        if (oldNum != visibleTiers.size()) {
            updateScrollBar();
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param tier DOCUMENT ME!
     */
    public void setActiveTier(Tier tier) {
        Iterator it = allTiers.iterator(); //visibleTiers??
        Tier2D t2d;

        while (it.hasNext()) {
            t2d = (Tier2D) it.next();

            if (t2d.getTier() == tier) {
                t2d.setActive(true);
                
                int tierIndex = visibleTiers.indexOf(t2d);
                int cy = tierIndex * pixelsForTierHeight;
                if (cy < verticalScrollOffset) {
                    scrollBar.setValue(cy);
                } else if (((cy + pixelsForTierHeight) - verticalScrollOffset) > (getHeight() -
                        rulerHeight - defBarWidth)) {
                    scrollBar.setValue((cy + pixelsForTierHeight + rulerHeight + defBarWidth) -
                        getHeight());
                }
            } else {
                t2d.setActive(false);
            }
        }

        paintBuffer();
    }

    /**
     * DOCUMENT ME!
     *
     * @param controller DOCUMENT ME!
     */
    public void setMultiTierControlPanel(MultiTierControlPanel controller) {
        multiTierControlPanel = controller;

        //paintBuffer();
        notifyMultiTierControlPanel();
    }

    //*************************************************************************************//

    /* implement ComponentListener */
    /*
     * After a resize this is (currently) the order in which methods are called
     * componentResized
     * doLayout()
	 * layout()
	 * paint()
	 * paintComponent()
	 * paint()
	 * paintComponent()
     * Calculate a new BufferedImage taken the new size of the Component
     */
    public void componentResized(ComponentEvent e) {
    	// in the case useBuffer == false calc the image width

    	if (!useBufferedImage) {
    		imageWidth = getWidth() - defBarWidth;
            intervalEndTime = intervalBeginTime + (int) (imageWidth * msPerPixel);
            if (timeScaleConnected) {
                setGlobalTimeScaleIntervalEndTime(intervalEndTime);
            }
    	}
        paintBuffer();
        if (hScrollBarVisible) {
        	if (getWidth() > 2 * ZOOMSLIDER_WIDTH) {
        		if (!zoomSliderPanel.isVisible()) {
        			zoomSliderPanel.setVisible(true);
        		}
	            hScrollBar.setBounds(0, getHeight() - defBarWidth, getWidth() - defBarWidth - ZOOMSLIDER_WIDTH, 
	            		defBarWidth);
	            //hScrollBar.revalidate();
	            updateHorScrollBar();
	            scrollBar.setBounds(getWidth() - defBarWidth, 0, defBarWidth,
	                    getHeight() - defBarWidth);
		        zoomSliderPanel.setBounds(hScrollBar.getWidth(), getHeight() - defBarWidth, ZOOMSLIDER_WIDTH, defBarWidth);
		        zoomSlider.setBounds(0, 0, ZOOMSLIDER_WIDTH, defBarWidth);
		        corner.setBounds(getWidth() - defBarWidth, getHeight() - defBarWidth, 
		        		defBarWidth, defBarWidth);
        	} else {
        		zoomSliderPanel.setVisible(false);
	            hScrollBar.setBounds(0, getHeight() - defBarWidth, getWidth() - defBarWidth, defBarWidth);
	            //hScrollBar.revalidate();
	            updateHorScrollBar();
	            scrollBar.setBounds(getWidth() - defBarWidth - 1, 0, defBarWidth,
	                    getHeight() - defBarWidth);
		        corner.setBounds(getWidth() - defBarWidth, getHeight() - defBarWidth, 
		        		defBarWidth, defBarWidth);
        	}
        } else {
            scrollBar.setBounds(getWidth() - defBarWidth - 1, 0, defBarWidth,
            getHeight());
	        corner.setBounds(getWidth() - defBarWidth, getHeight() - defBarWidth, 
	        		0, 0);
        }
        //scrollBar.revalidate();
        updateScrollBar();
    	
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void componentMoved(ComponentEvent e) {
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void componentShown(ComponentEvent e) {
        componentResized(e);
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void componentHidden(ComponentEvent e) {
    }

    //***********************************************************************************

    /* implement MouseListener and MouseMotionListener
       /*
     * A mouse click in the SignalViewer updates the media time
     * to the time corresponding to the x-position.
     */
    public void mouseClicked(MouseEvent e) {
        if (SwingUtilities.isRightMouseButton(e) || e.isPopupTrigger()) {
//        	checkForSplitAnnotation();
//        	Point p = e.getPoint();
//        	splitTime =  (int) (p.x * msPerPixel);
            return;
        }

        Annotation annotation = null; // new code by AR
        boolean shouldShowEditBox = false;

        // grab keyboard focus
        requestFocus();

        Point pp = e.getPoint();

        if ((e.getClickCount() == 1) && e.isAltDown()) {
        	Point inverse = new Point(pp);
            dismissEditBox();
            inverse.x += timeToPixels(intervalBeginTime);
            inverse.y += verticalScrollOffset;
            
            Tag2D selTag = getTagAt(inverse, getTierIndexForPoint(inverse));
            if (selTag != null) {
            	if (selectedAnnotations.contains(selTag)) {
            		selectedAnnotations.remove(selTag);
            	} else {
            		selectedAnnotations.add(selTag);
            	}
            } else {
            	selectedAnnotations.clear();
            }
            repaint();
        } else 
        if ((e.getClickCount() == 1) && e.isShiftDown()) {
            // change the selection interval
            if (getSelectionBeginTime() != getSelectionEndTime()) {
                long clickTime = timeAt(pp.x);

                if (clickTime > getSelectionEndTime()) {
                    // expand to the right
                    setSelection(getSelectionBeginTime(), clickTime);
                } else if (clickTime < getSelectionBeginTime()) {
                    // expand to the left
                    setSelection(clickTime, getSelectionEndTime());
                } else {
                    // reduce from left or right, whichever boundary is closest
                    // to the click time
                    if ((clickTime - getSelectionBeginTime()) < (getSelectionEndTime() -
                            clickTime)) {
                        setSelection(clickTime, getSelectionEndTime());
                    } else {
                        setSelection(getSelectionBeginTime(), clickTime);
                    }
                }
            } else {
            	// create a selection from media time to click time
            	long clickTime = timeAt(pp.x);
            	long medTime = getMediaTime();
            	if (clickTime > medTime) {
            		setSelection(medTime, clickTime);
            	} else if (clickTime < medTime) {
            		setSelection(clickTime, medTime);
            	}
            }
        } else {
            Point inverse = new Point(pp);
            dismissEditBox();

            if (timeRulerVisible && (pp.y < rulerHeight)) {
                //cursorTag2D = null;
            } else {
                //compensate for the intervalBeginTime
                inverse.x += timeToPixels(intervalBeginTime);
                inverse.y += verticalScrollOffset;
                cursorTierIndex = getTierIndexForPoint(inverse);
                cursorTag2D = getTagAt(inverse, cursorTierIndex);

                if (cursorTag2D != null) {
                    annotation = cursorTag2D.getAnnotation();
                    setActiveAnnotation(annotation);
                } else {
                    setActiveAnnotation(null);

                    //setMediaTime(timeAt(pp.x));
                }

                if ((e.getClickCount() == 2) && (cursorTag2D != null)) {
                    //showEditBoxForTag(cursorTag2D);
                    shouldShowEditBox = true;

                    if (e.isShiftDown()) {
                        forceOpenControlledVocabulary = true;
                    } else {
                        forceOpenControlledVocabulary = false;
                    }
                } else if ((e.getClickCount() >= 2) && (cursorTag2D == null)) {
                    autoInsertAnnotation(inverse, cursorTierIndex);
                }
            }

            //repaint();
            // disabled by AR
            //			setMediaTime(timeAt(pp.x));
            // new code by AR, takes care of setting selection in every mode to boundaries of active annotation
            if ((annotation == null) && !e.isAltDown()) {
                setMediaTime(timeAt(pp.x));
//            	System.out.println("isRightMouseButton: " + SwingUtilities.isRightMouseButton(e));
//            	System.out.println("isMeta: " + (e.getButton() == MouseEvent.BUTTON1 ^ e.isMetaDown()));
//            	System.out.println("Pop up: " + e.isPopupTrigger());
                if ( !SwingUtilities.isRightMouseButton(e) && !(e.getButton() == MouseEvent.BUTTON1 ^ e.isMetaDown()) 
                        && !e.isPopupTrigger()) {
//                	System.out.println("Clear");
//                	System.out.println("isRightMouseButton: " + SwingUtilities.isRightMouseButton(e));
//                	System.out.println("isMeta: " + (e.getButton() == MouseEvent.BUTTON1 ^ e.isMetaDown()));
//                	System.out.println("Pop up: " + e.isPopupTrigger());
                	selectedAnnotations.clear();               	
                }
            }

            if (shouldShowEditBox && (cursorTag2D != null)) {
                showEditBoxForTag(cursorTag2D);
            }
        }
        
        
        // Previously in this method the handling of:
	    // a single click, and isAltdown() or isShiftdown() 
	    // or two or more clicks. Now check if we need to clear 
        // a selection.
	    // Skip clearing the selection if shift or alt is down. Clear on a 
	    // single click only.
        // TODO Note: the current implementation interferes with the creation of an annotation
        // by double clicking inside the selection 
        /*
	    if (clearSelOnSingleClick && e.getClickCount() == 1
		    && !e.isShiftDown() && !e.isAltDown() && annotation == null && 
		    !getViewerManager().getMediaPlayerController().getSelectionMode()) {
	    	setSelection(0, 0);
	    }
		*/
    }

    /*
     * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
     */
    public void mousePressed(MouseEvent e) {
        Point pp = e.getPoint();

        // HS nov 04: e.isPopupTrigger always returns false on my PC...
        // HS jun 2006: change to discriminate Ctrl-click and Command-click on Mac
        // isRightMouseButton() returns true for both, isMetaDown() returns true for Command only 
        // but on Windows isMetaDown() returns true when the right mouse button (BUTTON3) has been pressed
        if ((SwingUtilities.isRightMouseButton(e) && (e.getButton() == MouseEvent.BUTTON1 ^ e.isMetaDown())) 
                || e.isPopupTrigger()) {
            if (popup == null) {
                createPopupMenu();
            }

            updatePopup(pp);

            if (e.isShiftDown()) {
                forceOpenControlledVocabulary = true;
            } else {
                forceOpenControlledVocabulary = false;
            }

            if ((popup.getWidth() == 0) || (popup.getHeight() == 0)) {
                popup.show(this, pp.x, pp.y);
            } else {
                popup.show(this, pp.x, pp.y);
                SwingUtilities.convertPointToScreen(pp, this);

                Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
                Window w = SwingUtilities.windowForComponent(this);

                if ((pp.x + popup.getWidth()) > d.width) {
                    pp.x -= popup.getWidth();
                }

                //this does not account for a desktop taskbar
                if ((pp.y + popup.getHeight()) > d.height) {
                    pp.y -= popup.getHeight();
                }

                //keep it in the window then
                if ((pp.y + popup.getHeight()) > (w.getLocationOnScreen().y +
                        w.getHeight())) {
                    pp.y -= popup.getHeight();
                }

                popup.setLocation(pp);
            }

            return;
        }

        if (playerIsPlaying()) {
            stopPlayer();
        }

        dragStartPoint = e.getPoint();
        dragStartTime = timeAt(dragStartPoint.x);

        if (e.isAltDown() && timeRulerVisible &&
                (dragStartPoint.y < rulerHeight)) {
            panMode = true;
            setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));

            //dismissEditBox();
        } else if (e.isAltDown()) {
            // drag-edit an annotation
            if ((cursorTag2D != null) && (dragEditTag2D == null)) {
                if (getHoverTag(new Point(e.getPoint())) == cursorTag2D) {
                    dragEditTag2D = new DragTag2D(cursorTag2D.getAnnotation());
                    dragEditTag2D.copyFrom(cursorTag2D);
                }
            }
            // hier... temp drag an annotation that is not active
            if (dragEditTag2D == null) {
            	Point inverse = new Point(pp);
                dismissEditBox();
                //inverse.x += timeToPixels(intervalBeginTime);
                inverse.y += verticalScrollOffset;
            	Tag2D hover = getHoverTag(e.getPoint());
            	if (hover != null && (hover.getAnnotation() instanceof AlignableAnnotation)) {
//            		if (hover.getAnnotation() != getActiveAnnotation()) {
//            			setActiveAnnotation(hover.getAnnotation());
//            		}
            		
                    dragEditTag2D = new DragTag2D(hover.getAnnotation());
                    dragEditTag2D.copyFrom(hover);
            		cursorTag2D = hover;
            		cursorTierIndex = getTierIndexForPoint(inverse);
            	}
            }

            if (dragEditTag2D != null) {
                // before changing anything, create a copy
                DragTag2D copy = new DragTag2D(dragEditTag2D.getAnnotation());
                copy.copyFrom(dragEditTag2D);
                //selectedAnnotations.remove(dragEditTag2D);
                dragEditTag2D = copy;
                dragEditing = true;

                // find the parent annotation's boundaries, if this one is not on 
                // a root tier
                if (((TierImpl) dragEditTag2D.getAnnotation().getTier()).hasParentTier()) {
                    //TierImpl pt = (TierImpl) ((TierImpl)dragEditTag2D.getAnnotation().getTier()).getParentTier();
                    //AlignableAnnotation pa = (AlignableAnnotation) pt.getAnnotationAtTime(
                    //	dragEditTag2D.getAnnotation().getBeginTimeBoundary());
                    AlignableAnnotation pa = (AlignableAnnotation) dragEditTag2D.getAnnotation()
                                                                                .getParentAnnotation();

                    if (pa != null) {
                        dragParentBegin = pa.getBeginTimeBoundary();
                        dragParentEnd = pa.getEndTimeBoundary();
                    } else {
                        dragParentBegin = -1L;
                        dragParentEnd = -1L;
                    }
                } else {
                    dragParentBegin = -1L;
                    dragParentEnd = -1L;
                }

                int x = (int) ((dragEditTag2D.getBeginTime() / msPerPixel) -
                    (intervalBeginTime / msPerPixel));
                int x2 = x +
                    (int) ((dragEditTag2D.getEndTime() -
                    dragEditTag2D.getBeginTime()) / msPerPixel);

                if (Math.abs(x - e.getX()) < DRAG_EDIT_MARGIN) {
                    dragEditMode = DRAG_EDIT_LEFT;
                    setCursor(Cursor.getPredefinedCursor(Cursor.W_RESIZE_CURSOR));
                } else if (Math.abs(x2 - e.getX()) < DRAG_EDIT_MARGIN) {
                    dragEditMode = DRAG_EDIT_RIGHT;
                    setCursor(Cursor.getPredefinedCursor(Cursor.E_RESIZE_CURSOR));
                } else {
                    dragEditMode = DRAG_EDIT_CENTER;
                    setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
                }

                if ((x2 - x) < (3 * DRAG_EDIT_MARGIN)) {
                    dragEditMode = DRAG_EDIT_CENTER;
                    setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
                }
            }
        } else {
            panMode = false;

            /* just to be sure a running scroll thread can be stopped */
            stopScroll();
        }
    }

    /*
     * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
     */
    public void mouseReleased(MouseEvent e) {
        //stop scrolling thread
        stopScroll();

        // changing the selection might have changed the intervalBeginTime
        if (timeScaleConnected) {
            setGlobalTimeScaleIntervalBeginTime(intervalBeginTime);
            setGlobalTimeScaleIntervalEndTime(intervalEndTime);
        }

        if (panMode) {
            panMode = false;
            setCursor(Cursor.getDefaultCursor());
        }

        if (dragEditing) {
            setCursor(Cursor.getDefaultCursor());
            if (dragEditTag2D != null) {
	            if (dragEditTag2D.getY() != dragEditTag2D.getOrigY()) {
	            	//System.out.println("Across tier translate!");
	            	Point inverse = new Point(e.getPoint());
	            	inverse.y += verticalScrollOffset;
	            	int tindex = getTierIndexForPoint(inverse);
	            	if (tindex > -1) {
	            		Tier2D t2d = visibleTiers.get(tindex);
	            		if (t2d.getTier() != dragEditTag2D.getAnnotation().getTier()) {
	            			if (t2d.getTier().getParentTier() != null) {
	            				// popup a message that the annotation can only be dragged to a
	            				// top level tier
	            			} else {
	            				doMoveAnnotation(dragEditTag2D.getAnnotation(), t2d.getTier());
	            			}
	            		} 
	            	}
	            } else {
	            	doDragModifyAnnotationTime();
	            }
            }
            dragEditing = false;
            dragEditTag2D = null;
            dragParentBegin = -1L;
            dragParentEnd = -1L;
            repaint();
        }
    }

    /*
     * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
     */
    public void mouseEntered(MouseEvent e) {
    }

    /*
     * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
     */
    public void mouseExited(MouseEvent e) {
        //stop scrolling thread
        stopScroll();
        hoverTag2D = null;
        showEmptySlots = false;
        repaint();
    }

    /*
     * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
     */
    public void mouseDragged(MouseEvent e) {
        if (SwingUtilities.isRightMouseButton(e)) {
            return;
        }

        dragEndPoint = e.getPoint();

        //panning
        if (panMode) {
            int scrolldiff = dragEndPoint.x - dragStartPoint.x;

            // some other viewer may have a media offset...
            long newTime = intervalBeginTime - (int) (scrolldiff * msPerPixel);

            if ((intervalBeginTime < 0) && (newTime < intervalBeginTime)) {
                newTime = intervalBeginTime;
            }

            setIntervalBeginTime(newTime);
            dragStartPoint = dragEndPoint;

            return;
        }

        /*e.getPoint can be outside the image size*/
        if ((dragEndPoint.x <= 0) ||
                (dragEndPoint.x >= (getWidth() - defBarWidth))) {
            stopScroll();

            /*
               if (timeScaleConnected) {
                   setGlobalTimeScaleIntervalBeginTime(intervalBeginTime);
                   setGlobalTimeScaleIntervalEndTime(intervalEndTime);
               }
             */
            return;
        }

        //auto scroll first
        if ((dragEndPoint.x < SCROLL_OFFSET) && (dragEndPoint.x > 0)) {
            /*
               long begin = intervalBeginTime - SCROLL_OFFSET * msPerPixel;
               if (begin < 0) {
                   begin = 0;
               }
               setIntervalBeginTime(begin);
               //paintBuffer();
             */
            if (scroller == null) {
                // if the dragging starts close to the edge call setSelection
                if ((dragStartPoint.x < SCROLL_OFFSET) &&
                        (dragStartPoint.x > 0)) {
                    setSelection(dragStartTime, dragStartTime);
                }

                stopScrolling = false;
                scroller = new DragScroller(-SCROLL_OFFSET / 4, 30);
                scroller.start();
            }

            return;
        } else if ((dragEndPoint.x > (getWidth() - defBarWidth - SCROLL_OFFSET)) &&
                (dragEndPoint.x < (getWidth() - defBarWidth))) {
            /*
               long begin = intervalBeginTime + SCROLL_OFFSET * msPerPixel;
               setIntervalBeginTime(begin);
               //paintBuffer();
             */
            if (scroller == null) {
                // if the dragging starts close to the edge call setSelection
                if ((dragStartPoint.x > (getWidth() - defBarWidth -
                        SCROLL_OFFSET)) &&
                        (dragStartPoint.x < (getWidth() - defBarWidth))) {
                    setSelection(dragStartTime, dragStartTime);
                }

                stopScrolling = false;
                scroller = new DragScroller(SCROLL_OFFSET / 4, 30);
                scroller.start();
            }

            return;
        } else {
            stopScroll();

            if (dragEditing) {
                // don't change the selection
                updateDragEditTag(dragEndPoint);

                return;
            }

            if (timeAt(dragEndPoint.x) > dragStartTime) { //left to right
                selectionEndTime = timeAt(dragEndPoint.x);

                if (selectionEndTime > getMediaDuration()) {
                    selectionEndTime = getMediaDuration();
                }

                selectionBeginTime = dragStartTime;

                if (selectionBeginTime < 0) {
                    selectionBeginTime = 0L;
                }

                if (selectionEndTime < 0) {
                    selectionEndTime = 0L;
                }

                setMediaTime(selectionEndTime);
            } else { //right to left
                selectionBeginTime = timeAt(dragEndPoint.x);

                if (selectionBeginTime > getMediaDuration()) {
                    selectionBeginTime = getMediaDuration();
                }

                selectionEndTime = dragStartTime;

                if (selectionEndTime > getMediaDuration()) {
                    selectionEndTime = getMediaDuration();
                }

                if (selectionBeginTime < 0) {
                    selectionBeginTime = 0L;
                }

                if (selectionEndTime < 0) {
                    selectionEndTime = 0L;
                }

                setMediaTime(selectionBeginTime);
            }

            setSelection(selectionBeginTime, selectionEndTime);
            repaint();
        }
    }

    /*
     * Note: if the alt key is released while the mouse is moving
     * isAltDown() still returns true.
     * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
     */
    public void mouseMoved(MouseEvent e) {
        Point pp = e.getPoint(); //pp is in component coordinates
        hoverTag2D = null;
        dragEditTag2D = null;

        if (timeRulerVisible && (pp.y < rulerHeight)) {
            //setToolTipText(null);
            setToolTipText(TimeFormatter.toString(timeAt(e.getPoint().x)));
            showEmptySlots = false;
            //repaint();

            return;
        }

        Point inverse = new Point(pp);

        //compensate for the intervalBeginTime
        //inverse.x += timeToPixels(intervalBeginTime);
        //inverse.y += verticalScrollOffset;
        //hoverTierIndex = getTierIndexForPoint(inverse);
        //hoverTag2D = getTagAt(inverse, hoverTierIndex);
        hoverTag2D = getHoverTag(inverse);

        /*
           if (e.isShiftDown()) {
               showEmptySlots = true;
           } else {
               showEmptySlots = false;
           }
         */

        // repaint();
        if ((hoverTag2D != null) && (hoverTag2D == cursorTag2D) &&
                e.isAltDown()) {
            setToolTipText(null);

            if (cursorTag2D.getAnnotation() instanceof AlignableAnnotation) {
                // here it is for display only, don't copy 
                dragEditTag2D = new DragTag2D(cursorTag2D.getAnnotation());
                dragEditTag2D.copyFrom(cursorTag2D);
            }
        } else if (hoverTag2D != null) {
            StringBuffer sb = new StringBuffer();

            if (hoverTag2D.getAnnotation() instanceof AlignableAnnotation) {
                sb.append("BT: ");
                sb.append(TimeFormatter.toString(hoverTag2D.getBeginTime()));
                sb.append(", ET: ");
                sb.append(TimeFormatter.toString(hoverTag2D.getEndTime()));
                sb.append(" ");
            }

            sb.append(hoverTag2D.getValue());

            //make tooltip multiline
            final int MAXTOOLTIPLENGTH = 140;
            String strTemp = sb.toString();
            String strEnd = "<html>";

            while (strTemp.length() > MAXTOOLTIPLENGTH) {
                int index = strTemp.lastIndexOf(" ", MAXTOOLTIPLENGTH);
                if (index > 0 && index <= MAXTOOLTIPLENGTH) {
                	strEnd += (strTemp.substring(0, index) + "<br>");
                	strTemp = strTemp.substring(index);
                } else {
                	if (strTemp.length() > MAXTOOLTIPLENGTH) {
                		strEnd += strTemp.substring(0, MAXTOOLTIPLENGTH);
                		strTemp = "";
                		break;
                	}
                }
            }
            strEnd += strTemp;

            if (hoverTag2D.getAnnotation() instanceof AbstractAnnotation) {
            	AbstractAnnotation annot = (AbstractAnnotation) hoverTag2D.getAnnotation(); 
            	String cvName = ((TierImpl) annot.getTier()).getLinguisticType().getControlledVocabylaryName(); 
            	ControlledVocabulary cv = null; 
            	if(cvName != null && !cvName.equals("")) { 
            		cv = transcription.getControlledVocabulary(cvName); 
            	} 
            	if (((AbstractAnnotation) hoverTag2D.getAnnotation()).getExtRef() != null) { 
            		if(cv instanceof ExternalCV) { 
            			ExternalCVEntry entry = (ExternalCVEntry) ((ExternalCV) cv).getEntrybyId(annot.getExtRefValue(ExternalReference.CVE_ID)); 
            			if(entry != null && entry.getDescription() != null && entry.getDescription().length() != 0) {
            				strEnd += "<br>" + entry.getDescription();
            			} 
            		} 
            		if (annot.getExtRefValue(ExternalReference.ISO12620_DC_ID) != null 
            			&& !annot.getExtRefValue(ExternalReference.ISO12620_DC_ID).equals("")) {
            			strEnd += "<br>DCR: " + annot.getExtRefValue(ExternalReference.ISO12620_DC_ID); 
            		} 
            	} 
            	if(cv != null && !(cv instanceof ExternalCV)){
            		CVEntry entry = cv.getEntryWithValue(annot.getValue());
            		if (entry != null && entry.getDescription() != null && entry.getDescription().length() != 0) {
            			strEnd += "<br>" + entry.getDescription(); 
            		} 
            	}
            }
            
            strEnd += ("</html>");
            setToolTipText(strEnd);

            //setToolTipText(t.toString().replace('\n', ' '));
            //System.out.println("Value: " + t.getValues() + "begin: " + t.getBeginTime() + " end: " + t.getEndTime());
        } else {
            setToolTipText(null);
        }
        if (e.isAltDown()) {// hier temp copied from pressed
            // drag-edit an annotation
//            if ((cursorTag2D != null) && (dragEditTag2D == null)) {
//                if (getHoverTag(new Point(e.getPoint())) == cursorTag2D) {
//                    dragEditTag2D = cursorTag2D;
//                }
//            }
//        	if (hoverTag2D != null) {
//        		dragEditTag2D = hoverTag2D;
//        	}

            if (hoverTag2D != null && (hoverTag2D.getAnnotation() instanceof AlignableAnnotation)) {
                // before changing anything, create a copy
                Tag2D copy = new Tag2D(hoverTag2D.getAnnotation());
                copy.setX(hoverTag2D.getX());
                copy.setWidth(hoverTag2D.getWidth());
                //selectedAnnotations.remove(hoverTag2D);
                hoverTag2D = copy;
                dragEditing = true;

                // find the parent annotation's boundaries, if this one is not on 
                // a root tier
                if (((TierImpl) hoverTag2D.getAnnotation().getTier()).hasParentTier()) {
                    //TierImpl pt = (TierImpl) ((TierImpl)dragEditTag2D.getAnnotation().getTier()).getParentTier();
                    //AlignableAnnotation pa = (AlignableAnnotation) pt.getAnnotationAtTime(
                    //	dragEditTag2D.getAnnotation().getBeginTimeBoundary());
                    AlignableAnnotation pa = (AlignableAnnotation) hoverTag2D.getAnnotation()
                                                                                .getParentAnnotation();

                    if (pa != null) {
                        dragParentBegin = pa.getBeginTimeBoundary();
                        dragParentEnd = pa.getEndTimeBoundary();
                    } else {
                        dragParentBegin = -1L;
                        dragParentEnd = -1L;
                    }
                } else {
                    dragParentBegin = -1L;
                    dragParentEnd = -1L;
                }

                int x = (int) ((hoverTag2D.getBeginTime() / msPerPixel) -
                    (intervalBeginTime / msPerPixel));
                int x2 = x +
                    (int) ((hoverTag2D.getEndTime() -
                    		hoverTag2D.getBeginTime()) / msPerPixel);

                if (Math.abs(x - e.getX()) < DRAG_EDIT_MARGIN) {
                    dragEditMode = DRAG_EDIT_LEFT;
                    setCursor(Cursor.getPredefinedCursor(Cursor.W_RESIZE_CURSOR));
                } else if (Math.abs(x2 - e.getX()) < DRAG_EDIT_MARGIN) {
                    dragEditMode = DRAG_EDIT_RIGHT;
                    setCursor(Cursor.getPredefinedCursor(Cursor.E_RESIZE_CURSOR));
                } else {
                    dragEditMode = DRAG_EDIT_CENTER;
                    setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
                }

                if ((x2 - x) < (3 * DRAG_EDIT_MARGIN)) {
                    dragEditMode = DRAG_EDIT_CENTER;
                    setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
                }
            } else {
            	setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
            }
        } else {
        	setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
        }
        //repaint();
    }

    /**
     * The use of a mousewheel needs Java 1.4!<br>
     * The scroll amount of the mousewheel is the height of a tier.
     *
     * @param e DOCUMENT ME!
     */
    public void mouseWheelMoved(MouseWheelEvent e) {
    	if (e.isControlDown()) {
    		if (e.getUnitsToScroll() > 0) {
    			zoomOut();
    		} else {
    			zoomIn();
    		}
    		return;
    	}
        if (e.getUnitsToScroll() > 0) {
            scrollBar.setValue(scrollBar.getValue() + pixelsForTierHeight);
        } else {
            scrollBar.setValue(scrollBar.getValue() - pixelsForTierHeight);
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void keyTyped(KeyEvent e) {
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void keyPressed(KeyEvent e) {
        if (e.getModifiers() == (KeyEvent.ALT_MASK +
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                KeyEvent.SHIFT_MASK)) {
            if (!showEmptySlots) {
                showEmptySlots = true;
                repaint();
            }
        } else {
            if (showEmptySlots) {
                showEmptySlots = false;
                repaint();
            }
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void keyReleased(KeyEvent e) {
        if ((e.getKeyCode() == KeyEvent.VK_CONTROL) ||
                (e.getKeyCode() == KeyEvent.VK_ALT) ||
                (e.getKeyCode() == KeyEvent.VK_SHIFT)) {
            if (showEmptySlots) {
                showEmptySlots = false;
                repaint();
            }
        }

        if (e.getKeyCode() == KeyEvent.VK_ALT) {
            if (!dragEditing) {
                dragEditTag2D = null;
                repaint();
            }
        }
    }

    /*
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent e) {
    	if (e.getSource() == copyAnnoMI) {
    		copyAnno();
    	} else if (e.getSource() == copyAnnoTreeMI) {
    		copyAnnoTree();
    	} else if (e.getSource() == pasteAnnoHereMI) {
    		pasteAnnoHere();
    	} else if (e.getSource() == pasteAnnoTreeHereMI) {
    		pasteAnnoTreeHere();
    	} else if (e.getSource() == deleteSelAnnosMI) {
    		deleteSelectedAnnotations();
    	} else if (e.getSource() == zoomSelectionMI) {
    		zoomToSelection();
    	} else if (e.getSource() == splitAnnotationMI) {
    		Annotation ann = getActiveAnnotation();
    		Object[] arguments = new Object[2];
        	arguments[0] = ann;
        	if(ann.getBeginTimeBoundary() < splitTime && ann.getEndTimeBoundary() > splitTime){
        		arguments[1] = splitTime;
        	}
        	Command command = ELANCommandFactory.createCommand(transcription, ELANCommandFactory.SPLIT_ANNOTATION);
        	command.execute(transcription, arguments);    	
        }     	else if (e.getSource() == tickerModeMI) {
    		tickerMode = tickerModeMI.isSelected();
            setPreference("TimeLineViewer.TickerMode", new Boolean(tickerMode), 
            		transcription);
    	} else if (e.getSource() == timeRulerVisMI) {
    		timeRulerVisible = timeRulerVisMI.isSelected();
    		if (timeRulerVisible) {
    			rulerHeight = ruler.getHeight();
    		} else {
    			rulerHeight = 0;
    		}
    		paintBuffer();
    		notifyMultiTierControlPanel();
    		setPreference("TimeLineViewer.TimeRulerVisible", new Boolean(timeRulerVisible), 
    				transcription);
    	}
    	else if (e.getActionCommand().equals("connect")) {
            boolean connected = ((JCheckBoxMenuItem) e.getSource()).getState();
            setTimeScaleConnected(connected);
            setPreference("TimeLineViewer.TimeScaleConnected", new Boolean(connected), 
            		transcription);
        } else if (e.getActionCommand().equals("aastroke")) {
            aaStrokeBold = ((JCheckBoxMenuItem) e.getSource()).getState();
            repaint();
            setPreference("TimeLineViewer.ActiveAnnotationBold", 
            		new Boolean(aaStrokeBold), transcription);
        } else if (e.getActionCommand().equals("redTH")) {
        	reducedTierHeight = ((JCheckBoxMenuItem) e.getSource()).getState();
            pixelsForTierHeight = calcTierHeight();
            notifyMultiTierControlPanel();
            paintBuffer();
            setPreference("TimeLineViewer.ReducedTierHeight", 
            		new Boolean(reducedTierHeight), transcription);
        } else if (e.getActionCommand().equals("hsVis")) {
            hScrollBarVisible = ((JCheckBoxMenuItem) e.getSource()).getState();
            if (hScrollBarVisible) {
                add(hScrollBar);
            } else {
                remove(hScrollBar);
            }
            componentResized(null);
            setPreference("TimeLineViewer.HorizontalScrollBarVisible", 
            		new Boolean(hScrollBarVisible), transcription);
        } else if (e.getActionCommand().equals("activeTier")) {
            doActiveTier();
        } else if (e.getActionCommand().equals("deleteTier")) {
            doDeleteTier();
        } else if (e.getActionCommand().equals("changeTier")) {
            doChangeTier();
        } else if (e.getActionCommand().equals("newAnn")) {
            doNewAnnotation();
        } else if (e.getActionCommand().equals("annBefore")) {// remove
            doAnnotationBefore();
        } else if (e.getActionCommand().equals("annAfter")) {// remove
            doAnnotationAfter();
        } else if (e.getActionCommand().equals("modifyAnn")) {
            doModifyAnnotation();
        } else if (e.getActionCommand().equals("deleteAnn")) {
            doDeleteAnnotation();
        } else if (e.getActionCommand().indexOf("font") > -1) {
            String sizeString = e.getActionCommand();
            int index = sizeString.indexOf("font") + 4;
            int size = 12;

            try {
                size = Integer.parseInt(sizeString.substring(index));
            } catch (NumberFormatException nfe) {
                System.err.println("Error parsing font size");
            }

            updateFont(getFont().deriveFont((float) size));
            setPreference("TimeLineViewer.FontSize", new Integer(size), 
            		transcription);
        } else {
            /* the rest are zoom menu items*/
            String zoomString = e.getActionCommand();
            int zoom = 100;

            try {
                zoom = Integer.parseInt(zoomString);
            } catch (NumberFormatException nfe) {
                System.err.println("Error parsing the zoom level");
            }

            float newMsPerPixel = ((100f / zoom) * 10);
            setMsPerPixel(newMsPerPixel);
            setPreference("TimeLineViewer.ZoomLevel", new Float(zoom), 
            		transcription);
        }
    }
    
	public void editingCancelled() {
		if (delayedAnnotationActivation) {
			if (activateNewAnnotation) {
				setActiveAnnotation(lastCreatedAnnotation);
			}
			delayedAnnotationActivation = false;
		}
	}

	public void editingCommitted() {
		if (delayedAnnotationActivation) {
			if (activateNewAnnotation) {
				setActiveAnnotation(lastCreatedAnnotation);
			}
			delayedAnnotationActivation = false;
		}		
	}

    /**
     * Set the font size, the "active annotation bold" flag etc.
     */
	public void preferencesChanged() {
		Integer fontSize = (Integer) getPreference("TimeLineViewer.FontSize", 
				transcription);
		if (fontSize != null) {
			setFontSize(fontSize.intValue());
		}
		Boolean aaBold = (Boolean) getPreference("TimeLineViewer.ActiveAnnotationBold",
				transcription);
		if (aaBold == null) {
			aaBold = (Boolean) getPreference("TimeLineViewer.ActiveAnnotationBold",
					null);// application default
		}
		if (aaBold != null) {
			if (activeAnnStrokeBoldMI != null) {
				// will this cause an event?
				activeAnnStrokeBoldMI.setSelected(aaBold.booleanValue());
			}
			aaStrokeBold = aaBold.booleanValue();
		}
		
		Boolean centerAnn = (Boolean) Preferences.get("EditingPanel.ActiveAnnotationInCenter", null);
    	if(centerAnn == null){
    		centerAnnotation = true;
    	} else {
    		centerAnnotation = centerAnn.booleanValue();
    	}
    	
    	Object clearSel = Preferences.get("ClearSelectionAfterCreation", null);
    	
    	if (clearSel instanceof Boolean) {
    		activateNewAnnotation = !((Boolean) clearSel);
    	}
		
		Object redTH = getPreference("TimeLineViewer.ReducedTierHeight", transcription);
		if (redTH == null) {
			redTH = getPreference("TimeLineViewer.ReducedTierHeight", null);
		}
		if (redTH instanceof Boolean) {
			boolean oldReduce = reducedTierHeight;
			reducedTierHeight = ((Boolean) redTH).booleanValue();
			
			if (reducedTierHeightMI != null) {
				reducedTierHeightMI.setSelected(reducedTierHeight);
			}
			if (oldReduce != reducedTierHeight) {
		        pixelsForTierHeight = calcTierHeight(); 
		        notifyMultiTierControlPanel();
			}
		}			

		Boolean hsVis = (Boolean) getPreference("TimeLineViewer.HorizontalScrollBarVisible", 
				transcription);
		if (hsVis != null) {
			hScrollBarVisible = hsVis.booleanValue();
			if (hScrollBarVisMI != null) {
				// will this cause an event??
				hScrollBarVisMI.setSelected(hScrollBarVisible);
			}
			if (!hScrollBarVisible) {
	            remove(hScrollBar);
	        }
		}
		Boolean tsConnect = (Boolean) getPreference("TimeLineViewer.TimeScaleConnected", 
				transcription);
		if (tsConnect != null) {
			if (timeScaleConMI != null) {
				timeScaleConMI.setSelected(tsConnect.booleanValue());
			}
			setTimeScaleConnected(tsConnect.booleanValue());
		}
		Boolean ticMode = (Boolean) getPreference("TimeLineViewer.TickerMode", 
				transcription);
		if (ticMode != null) {
			tickerMode = ticMode.booleanValue();
			if (tickerModeMI != null) {				
				tickerModeMI.setSelected(tickerMode);				
			}
		}
		Boolean rulerVis = (Boolean) getPreference("TimeLineViewer.TimeRulerVisible", 
				transcription);
		if (rulerVis != null) {
			timeRulerVisible = rulerVis.booleanValue();
			if (timeRulerVisMI != null) {
				timeRulerVisMI.setSelected(timeRulerVisible);
			}
			if (timeRulerVisible) {
				rulerHeight = ruler.getHeight();
			} else {
				rulerHeight = 0;
			}
		}
		
		Object zoomLevel = getPreference("TimeLineViewer.ZoomLevel", 
				transcription);
		if (zoomLevel instanceof Float) {
			float zl = ((Float) zoomLevel).floatValue();
            float newMsPerPixel =  ((100f / zl) * 10);
            setMsPerPixel(newMsPerPixel);
			updateZoomPopup((int)zl);
		}
		// preferred fonts
		Object fo = getPreference("TierFonts", transcription);
		if (fo instanceof HashMap) {
			HashMap foMap = (HashMap) fo;
			
			Iterator keyIt = foMap.keySet().iterator();
			String key = null;
			Font ft = null;
			Tier2D t2d = null;				
			
			while (keyIt.hasNext()) {
				key = (String) keyIt.next();
				ft = (Font) foMap.get(key);
				
				if (key != null && ft != null) {
					for (int i = 0; i <allTiers.size(); i++) {
						t2d = (Tier2D) allTiers.get(i);
						if (t2d.getName().equals(key)) {
							break;
						}
					}
					// use the size of the default font
					if (prefTierFonts.containsKey(key)) {
						Font oldF = (Font) prefTierFonts.get(key);
						if (!oldF.getName().equals(ft.getName())) {
							prefTierFonts.put(key, new Font(ft.getName(), 
									Font.PLAIN, font.getSize()));
							reextractTagsForTier(t2d);
						}
					} else {
						prefTierFonts.put(key, new Font(ft.getName(), Font.PLAIN, font.getSize()));
						reextractTagsForTier(t2d);
					}					
				}
			}	
			Iterator keyIt2 = prefTierFonts.keySet().iterator();
			ArrayList remKeys = new ArrayList(4);
			while (keyIt2.hasNext()) {
				key = (String) keyIt2.next();
				if (!foMap.containsKey(key)) {
					remKeys.add(key);
				}
			}
			if (remKeys.size() > 0) {
				for (int j = 0; j < remKeys.size(); j++) {
					key = (String) remKeys.get(j);
					prefTierFonts.remove(key);
					
					for (int i = 0; i <allTiers.size(); i++) {
						t2d = (Tier2D) allTiers.get(i);
						if (t2d.getName().equals(key)) {
							reextractTagsForTier(t2d);
							break;
						}
					}
				}
			}
		}
		
        Object val = getPreference("InlineEdit.EnterCommits", null);

        if (val instanceof Boolean) {
            editBox.setEnterCommits(((Boolean) val).booleanValue());
        }

        val = Preferences.get("InlineEdit.DeselectCommits", null);

        if (val instanceof Boolean) {
            deselectCommits = ((Boolean) val).booleanValue();
        }
        highlightColors =  (Map<String,Color>) Preferences.get(
        		"TierHighlightColors", transcription);
        
        // update CV entry colors, should be handled differently if possible
        ControlledVocabulary cv;
        CVEntry[] entries;
        Tag2D tag2d;
        for (Tier2D t2d : allTiers) {
        	
        	cv = transcription.getControlledVocabulary(t2d.getTier().getLinguisticType().getControlledVocabylaryName());
        	if (cv != null) {
        		entries = cv.getEntries();
        		Iterator it = t2d.getTags();
        		while (it.hasNext()) {
        			tag2d = (Tag2D) it.next();
        			
        			if (tag2d.getAnnotation().getValue() != null && tag2d.getAnnotation().getValue().length() > 0) {
        				for (CVEntry cve : entries) {
        					if (tag2d.getAnnotation().getValue().equals(cve.getValue())) {
        						tag2d.setColor(cve.getPrefColor());
        						break;
        					}
        				}
        			}
        		}
        	} else {
        		TierImpl markTier = TierAssociation.findMarkerTierFor(transcription, t2d.getTier());
        		if (markTier != null) {
        			cv = transcription.getControlledVocabulary(markTier.getLinguisticType().getControlledVocabylaryName());
        			if (cv != null) {
        				entries = cv.getEntries();
        				Iterator it = t2d.getTags();
        				List ch;
                		while (it.hasNext()) {
                			tag2d = (Tag2D) it.next();
                			
                    		ch = ((AbstractAnnotation)tag2d.getAnnotation()).getChildrenOnTier(markTier);
                    		if (ch.size() >= 1) {// should be 1 max
                    			Annotation ma = (Annotation) ch.get(0);
            	            	for (CVEntry e : entries) {
            	            		if (e.getValue().equals(ma.getValue())) {
            	            			tag2d.setColor(e.getPrefColor());
            	            			break;
            	            		}
            	            	}
                    		}
                		}
        			}
        		}

        	}
        	
        } 
        
	val = getPreference("ClearSelectionOnSingleClick",null); 
	if (val instanceof Boolean){
	    clearSelOnSingleClick = ((Boolean) val).booleanValue();
	}
	
	val = getPreference("Preferences.SymAnnColor",null); 
	if (val instanceof Color){
	    symAnnColor = new Color (((Color)val).getRed(),((Color)val).getGreen(),((Color)val).getBlue());
	}
	
	paintBuffer();
}
	
    /**
     * Should we dispose of Graphics object?
     */
    public void finalize() throws Throwable {
        System.out.println("Finalize TimeLineViewer...");
        if (bi != null) {
            bi.flush();
        }
        if (big2d != null) {
            big2d.dispose();
        }
        if (editBox != null) {
        	editBox.removeInlineEditBoxListener(this);
        }
        super.finalize();
    }
/*
	@Override
	public void paint(Graphics g) {
		System.out.println("Paint");		
		super.paint(g);
	}

	@Override
	public void layout() {
		// TODO Auto-generated method stub
		System.out.println("Layout");
		super.layout();
	}

	@Override
	public void doLayout() {
		// TODO Auto-generated method stub
		System.out.println("Do Layout");
		super.doLayout();
		//paintBuffer();
	}
*/
	/**
     * Scrolls the image while dragging to the left or right with the specified
     * number of pixels.<br>
     * This method is called from a separate Thread.
     *
     * @param numPixels the number of pixels to scroll the interval
     *
     * @see DragScroller
     */
    synchronized void scroll(int numPixels) {
        long begin = intervalBeginTime + (int) (numPixels * msPerPixel);

        if (numPixels > 0) {
            // left to right, change selection while scrolling
            setIntervalBeginTime(begin);
            selectionEndTime = getSelectionEndTime() +
                (int) (numPixels * msPerPixel);

            if (selectionEndTime > getMediaDuration()) {
                selectionEndTime = getMediaDuration();
            }

            setMediaTime(selectionEndTime);
            setSelection(getSelectionBeginTime(), selectionEndTime);
        } else {
            // right to left
            if (begin < 0) {
                begin = 0;
            }

            setIntervalBeginTime(begin);
            selectionBeginTime = getSelectionBeginTime() +
                (int) (numPixels * msPerPixel);

            if (selectionBeginTime < 0) {
                selectionBeginTime = 0;
            }

            setMediaTime(selectionBeginTime);
            setSelection(selectionBeginTime, getSelectionEndTime());
        }

        //repaint();
    }

    /**
     * DOCUMENT ME!
     */
    void stopScroll() {
        /*
           if (scroller != null) {
               try {
                   scroller.interrupt();
               } catch (SecurityException se) {
                   System.out.println("TimeLineViewer: could not stop scroll thread");
               } finally {
               }
               scroller = null;
           }
         */
        stopScrolling = true;
        scroller = null;
    }

    // the edit actions from the popup menu, the Commands and Actions will come in here
    private void doActiveTier() {
        if ((rightClickTier == null) || (multiTierControlPanel == null)) {
            return;
        }

        multiTierControlPanel.setActiveTier(rightClickTier.getTier());
    }

    private void doDeleteTier() {
        Command c = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.EDIT_TIER);

        Object[] args = new Object[] {
                new Integer(EditTierDialog.DELETE), rightClickTier.getTier()
            };

        c.execute(transcription, args);
    }

    private void doChangeTier() {
        Command c = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.EDIT_TIER);

        Object[] args = new Object[] {
                new Integer(EditTierDialog.CHANGE), rightClickTier.getTier()
            };

        c.execute(transcription, args);
    }

    private void doNewAnnotation() {
        if (rightClickTier == null) {
            return;
        }

        // use command
        TierImpl tier = rightClickTier.getTier();
        long begin = getSelectionBeginTime();
        long end = getSelectionEndTime();

        if (begin == end) {
            return;
        }

        if (!tier.isTimeAlignable()) {
            begin = rightClickTime;
            end = rightClickTime;
        }

        Command c = null;
        Object val = Preferences.get("CreateDependingAnnotations", null);                
        if (val instanceof Boolean) {
        	 if(((Boolean) val).booleanValue()){
        		 c = ELANCommandFactory.createCommand(transcription,
                         ELANCommandFactory.NEW_ANNOTATION_REC);  
        		 
        	 } else {
        		 c = ELANCommandFactory.createCommand(transcription,
                         ELANCommandFactory.NEW_ANNOTATION);                		 
        	 }
        }  else {
   		 	c = ELANCommandFactory.createCommand(transcription,
                 ELANCommandFactory.NEW_ANNOTATION);                		 
        }
        
        Object[] args = new Object[] {
                new Long(begin), new Long(end)
            };
        c.execute(tier, args);
    }

    /**
     * Annotation before is relative to the cursor tag (active annotation)
     * remove
     */
    private void doAnnotationBefore() {
        if ((rightClickTier == null) || (cursorTag2D == null)) {
            return;
        }

        Command c = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.NEW_ANNOTATION_BEFORE);
        Object[] args = new Object[] { cursorTag2D.getAnnotation() };
        c.execute(rightClickTier.getTier(), args);
    }

    /**
     * Annotation after is relative to the cursor tag (active annotation)
     * remove
     */
    private void doAnnotationAfter() {
        if ((rightClickTier == null) || (cursorTag2D == null)) {
            return;
        }

        Command c = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.NEW_ANNOTATION_AFTER);
        Object[] args = new Object[] { cursorTag2D.getAnnotation() };
        c.execute(rightClickTier.getTier(), args);
    }

    private void doModifyAnnotation() {
        if (cursorTag2D != null) {
            showEditBoxForTag(cursorTag2D);
        }
    }

    private void doDeleteAnnotation() {
        if (cursorTag2D != null) {
            TierImpl tier = cursorTag2D.getTier2D().getTier();
            Annotation aa = cursorTag2D.getAnnotation();

            Command c = ELANCommandFactory.createCommand(transcription,
                    ELANCommandFactory.DELETE_ANNOTATION);
            c.execute(tier, new Object[] { getViewerManager(), aa });
        }
    }
    
    private void deleteSelectedAnnotations() {
    	if (selectedAnnotations.size() == 0) {
    		return;
    	}
    	List<Annotation> selAnnos = new ArrayList<Annotation>(selectedAnnotations.size());
    	for (Tag2D t2d : selectedAnnotations) {
    		selAnnos.add(t2d.getAnnotation());
    	}
    	Command c = ELANCommandFactory.createCommand(transcription, ELANCommandFactory.DELETE_MULTIPLE_ANNOS);
    	c.execute(transcription, new Object[]{selAnnos});
    	
    	selectedAnnotations.clear();
    	repaint();
    }
    
    private void copyAnno() {
    	if (cursorTag2D != null) {
            Annotation aa = cursorTag2D.getAnnotation();
            
            Command c = ELANCommandFactory.createCommand(transcription,
                    ELANCommandFactory.COPY_ANNOTATION);
            c.execute(null, new Object[]{aa});
    	}
    }
    
    private void copyAnnoTree() {
    	if (cursorTag2D != null) {
            Annotation aa = cursorTag2D.getAnnotation();
            
            Command c = ELANCommandFactory.createCommand(transcription,
                    ELANCommandFactory.COPY_ANNOTATION_TREE);
            c.execute(null, new Object[]{aa});
    	}
    }
    
    private void pasteAnnoHere() {
    	if (rightClickTier != null) {
    		TierImpl tier = rightClickTier.getTier();
    		
            Command c = ELANCommandFactory.createCommand(transcription,
                    ELANCommandFactory.PASTE_ANNOTATION_HERE);
            c.execute(transcription, new Object[]{tier.getName(), new Long(rightClickTime)});
    	}
    }
    
    private void pasteAnnoTreeHere() {
    	if (rightClickTier != null) {
    		TierImpl tier = rightClickTier.getTier();
    		
            Command c = ELANCommandFactory.createCommand(transcription,
                    ELANCommandFactory.PASTE_ANNOTATION_TREE_HERE);
            c.execute(transcription, new Object[]{tier.getName(), new Long(rightClickTime)});
    	}	
    }

    /**
     * Calculate the new begin and end time of the active annotation  and
     * create a command.
     */
    private void doDragModifyAnnotationTime() {
    	if (dragEditTag2D == null) {
    		return;
    	}
    	AlignableAnnotation editAnn = null;
    	if (dragEditTag2D != null && dragEditTag2D.getAnnotation() instanceof AlignableAnnotation) {
    		editAnn = (AlignableAnnotation) dragEditTag2D.getAnnotation();
    	}
    	if (editAnn == null) {
    		return;
    	}
//        if ((dragEditTag2D == null) ||
//                !(getActiveAnnotation() instanceof AlignableAnnotation)) {
//            return;
//        }

        long beginTime = 0L;
        long endTime = 0L;

        switch (dragEditMode) {
        case DRAG_EDIT_CENTER:
            beginTime = pixelToTime(dragEditTag2D.getX());
            endTime = pixelToTime(dragEditTag2D.getX() +
                    dragEditTag2D.getWidth());

            break;

        case DRAG_EDIT_LEFT:
            beginTime = pixelToTime(dragEditTag2D.getX());
            endTime = dragEditTag2D.getEndTime();

            break;

        case DRAG_EDIT_RIGHT:
            beginTime = dragEditTag2D.getBeginTime();
            endTime = pixelToTime(dragEditTag2D.getX() +
                    dragEditTag2D.getWidth());
        }

        if (endTime <= beginTime) {
            return;
        }
        // check if the change in begin and/or end time is >= than one pixel
        if (Math.abs(beginTime - dragEditTag2D.getBeginTime()) < msPerPixel &&
        		Math.abs(endTime - dragEditTag2D.getEndTime()) < msPerPixel) {
        	return;
        }
        Command c = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.MODIFY_ANNOTATION_TIME);
//        c.execute(getActiveAnnotation(),
//            new Object[] { new Long(beginTime), new Long(endTime) });
        c.execute(editAnn,
                new Object[] { new Long(beginTime), new Long(endTime) });
    }
    
    /**
     * Moves an (alignable) annotation from one tier to another (top level tier) . 
     * @param ann
     * @param t
     */
    private void doMoveAnnotation(Annotation ann, TierImpl t) {
        Command c = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.MOVE_ANNOTATION_TO_TIER);

        c.execute(t, new Object[] { ann });
    }

    /**
     * A Thread for scrolling near the left and right border of the viewer.
     *
     * @author HS
     * @version 1.0
     */
    class DragScroller extends Thread {
        /** Holds value of property DOCUMENT ME! */
        int numPixels;

        /** Holds value of property DOCUMENT ME! */
        long sleepTime;

        /**
         * Creates a new DragScroller instance
         *
         * @param numPixels DOCUMENT ME!
         * @param sleepTime DOCUMENT ME!
         */
        DragScroller(int numPixels, long sleepTime) {
            this.numPixels = numPixels;
            this.sleepTime = sleepTime;
        }

        /**
         * Periodically scrolls the view.
         */
        public void run() {
            while (!stopScrolling) {
                TimeLineViewer.this.scroll(numPixels);

                try {
                    sleep(sleepTime);
                } catch (InterruptedException ie) {
                    return;
                }
            }
        }
    }

    //##############################################
/*
	public void doLayout() {
		// TODO Auto-generated method stub
		super.doLayout();
		System.out.println("doLayout....");
    	if (!useBufferedImage) {
    		imageWidth = getWidth() - defBarWidth;
            intervalEndTime = intervalBeginTime + (int) (imageWidth * msPerPixel);
            if (timeScaleConnected) {
                setGlobalTimeScaleIntervalEndTime(intervalEndTime);
            }
    	}
        //paintBuffer();
        if (hScrollBarVisible) {
            hScrollBar.setBounds(0, getHeight() - defBarWidth, getWidth() - defBarWidth, defBarWidth);
            //hScrollBar.revalidate();
            updateHorScrollBar();
        scrollBar.setBounds(getWidth() - defBarWidth, 0, defBarWidth,
                    getHeight() - defBarWidth);
        } else {
            scrollBar.setBounds(getWidth() - defBarWidth, 0, defBarWidth,
            getHeight());
        }
        //scrollBar.revalidate();
        updateScrollBar();
        paintBuffer();
        //hScrollBar.setVisible(true);
	}
*/
	class TLLayoutManager implements LayoutManager2 {

		public void addLayoutComponent(Component comp, Object constraints) {
			// TODO Auto-generated method stub
			System.out.println("add...");
		}

		public float getLayoutAlignmentX(Container target) {
			// TODO Auto-generated method stub
			return 0;
		}

		public float getLayoutAlignmentY(Container target) {
			// TODO Auto-generated method stub
			return 0;
		}

		public void invalidateLayout(Container target) {
			// TODO Auto-generated method stub
			System.out.println("invalidate...");
			//hScrollBar.setVisible(false);
		}

		public Dimension maximumLayoutSize(Container target) {
			// TODO Auto-generated method stub
			return null;
		}

		public void addLayoutComponent(String name, Component comp) {
			// TODO Auto-generated method stub
			
		}

		public void layoutContainer(Container parent) {
			// TODO Auto-generated method stub
			System.out.println("layout container...");
			// in the case useBuffer == false calc the image width
			/*
	    	if (!useBufferedImage) {
	    		imageWidth = getWidth() - defBarWidth;
	            intervalEndTime = intervalBeginTime + (int) (imageWidth * msPerPixel);
	            if (timeScaleConnected) {
	                setGlobalTimeScaleIntervalEndTime(intervalEndTime);
	            }
	    	}
	        //paintBuffer();
	        if (hScrollBarVisible) {
	            hScrollBar.setBounds(0, getHeight() - defBarWidth, getWidth() - defBarWidth, defBarWidth);
	            //hScrollBar.revalidate();
	            updateHorScrollBar();
	        scrollBar.setBounds(getWidth() - defBarWidth, 0, defBarWidth,
	                    getHeight() - defBarWidth);
	        } else {
	            scrollBar.setBounds(getWidth() - defBarWidth, 0, defBarWidth,
	            getHeight());
	        }
	        //scrollBar.revalidate();
	        updateScrollBar();
	        paintBuffer();
	        */
		}

		public Dimension minimumLayoutSize(Container parent) {
			return null;
		}

		public Dimension preferredLayoutSize(Container parent) {
			return null;
		}

		public void removeLayoutComponent(Component comp) {
		}
    	
    }
}
