package mpi.eudico.server.corpora.clomimpl.abstr;

/**
 * A descriptor for any kind of file that is linked to or associated with  a
 * transcription, other than audio and video files (or the subjects of
 * transcription).  All fields are public.
 * NB: an interface and abstract super class could/should be created that 
 * is extended by this class and MediaDescriptors.
 * 
 * @author Han Sloetjes
 */
public class LinkedFileDescriptor implements Cloneable {
    /** Holds value of constant unknown MIME type */
    public final static String UNKNOWN_MIME_TYPE = "unknown";

	/** text MIME type */
	public final static String TEXT_TYPE = "text/plain";
	
	/** xml MIME type */
	public final static String XML_TYPE = "text/xml";

	/** svg MIME type */
	public final static String SVG_TYPE = "image/svg+xml";	
	
    /** the url of the linked file */
    public String linkURL;
    
    /** holds the relative URL, relative to the containing document */
    public String relativeLinkURL;

    /** the mimetype string */
    public String mimeType;

    /** the time origin or offset */
    public long timeOrigin;

    /**
     * the url of a media file that this file is associated with  in whichever
     * way
     */
    public String associatedWith;

    /**
     * the url of a configuration file that might contain information  on how
     * to treat this file
     */
    public String configFile; //??

    /**
     * Constructor.
     *
     * @param linkURL the url of the linked file
     * @param mimeType a mimetype string
     *
     * @throws NullPointerException if the link url is null
     */
    public LinkedFileDescriptor(String linkURL, String mimeType) {
        if (linkURL == null) {
            throw new NullPointerException();
        }

        this.linkURL = linkURL;

        if (mimeType != null) {
            this.mimeType = mimeType;
        } else {
            this.mimeType = UNKNOWN_MIME_TYPE;
        }
    }

    /**
     * Returns a string representation of this descriptor
     *
     * @return a string representation of this descriptor
     */
    public String toString() {
        return linkURL + " " + mimeType + " " + timeOrigin + " " +
        associatedWith + " " + configFile;
    }

    /**
     * Returns a deep copy of this LinkedFileDescriptor.
     *
     * @return a deep copy of this LinkedFileDescriptor
     */
    public Object clone() {
        try {
            LinkedFileDescriptor cloneLFD = (LinkedFileDescriptor) super.clone();

            if (linkURL != null) {
                cloneLFD.linkURL = new String(linkURL);
            }
            
            if (relativeLinkURL != null) {
            	cloneLFD.relativeLinkURL = new String(relativeLinkURL);
            }
            
            if (mimeType != null) {
                cloneLFD.mimeType = new String(mimeType);
            }

            if (associatedWith != null) {
                cloneLFD.associatedWith = new String(associatedWith);
            }

            if (configFile != null) {
                cloneLFD.configFile = new String(configFile);
            }

            cloneLFD.timeOrigin = timeOrigin;

            return cloneLFD;
        } catch (CloneNotSupportedException cnse) {
            // should not happen
            // throw an exception?
            return null;
        }
    }

    /**
     * Overrides <code>Object</code>'s equals method by checking all  fields of
     * the other object to be equal to all fields in this  object.
     *
     * @param obj the reference object with which to compare
     *
     * @return true if this object is the same as the obj argument; false
     *         otherwise
     */
    public boolean equals(Object obj) {
        if (obj == null) {
            // null is never equal
            return false;
        }

        if (obj == this) {
            // same object reference 
            return true;
        }

        if (!(obj instanceof LinkedFileDescriptor)) {
            // it should be a LinkedFileDescriptor object
            return false;
        }

        // check the fields
        LinkedFileDescriptor other = (LinkedFileDescriptor) obj;

        if (((this.linkURL != null) && (other.linkURL == null)) ||
                ((this.linkURL == null) && (other.linkURL != null)) ||
                ((this.linkURL != null) && (other.linkURL != null) &&
                !this.linkURL.equals(other.linkURL))) {
            return false;
        }

        if (((this.relativeLinkURL != null) && (other.relativeLinkURL == null)) ||
                ((this.relativeLinkURL == null) && (other.relativeLinkURL != null)) ||
                ((this.relativeLinkURL != null) && (other.relativeLinkURL != null) &&
                !this.relativeLinkURL.equals(other.relativeLinkURL))) {
            return false;
        }
        
        if (((this.mimeType != null) && (other.mimeType == null)) ||
                ((this.mimeType == null) && (other.mimeType != null)) ||
                ((this.mimeType != null) && (other.mimeType != null) &&
                !this.mimeType.equals(other.mimeType))) {
            return false;
        }

        if (((this.associatedWith != null) && (other.associatedWith == null)) ||
                ((this.associatedWith == null) &&
                (other.associatedWith != null)) ||
                ((this.associatedWith != null) &&
                (other.associatedWith != null) &&
                !this.associatedWith.equals(other.associatedWith))) {
            return false;
        }

        if (((this.configFile != null) && (other.configFile == null)) ||
                ((this.configFile == null) && (other.configFile != null)) ||
                ((this.configFile != null) && (other.configFile != null) &&
                !this.configFile.equals(other.configFile))) {
            return false;
        }

        if (this.timeOrigin != other.timeOrigin) {
            return false;
        }

        return true;
    }
}
