package nl.mpi.recognizer.remote.clam;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.swing.JPanel;

import mpi.eudico.client.annotator.recognizer.api.Recognizer;
import mpi.eudico.client.annotator.recognizer.api.RecognizerHost;

public class ClamRecognizer implements Recognizer {
	protected RecognizerHost host;
	private String name = "CLAM remote recognizer";
	protected List<String> mediaPaths;
	private String baseUrl = "http://mpcserver.let.ru.nl/aamlr/";
	private String projectId = null;
	private String currentMediaFile = null;
	private String currentMediaName = null;
	private String outputFile = null;
	private Map<String, String> fileProjectMap;
	private List<String> uploadedProjects;
	
	private float progress = 0f;
	private boolean isRunning = false;
	private boolean interrupted = false;
	private StringBuilder reportBuilder = null;
	private long startTime;
	
	/**
	 * Constructor.
	 */
	public ClamRecognizer() {
		super();
		fileProjectMap = new HashMap<String, String>(5);
		uploadedProjects = new ArrayList<String>(5);
	}

	public boolean canCombineMultipleFiles() {
		return false;
	}

	public boolean canHandleMedia(String mediaFilePath) {
		if (mediaFilePath == null) {
			return false;
		}
		// check extension, or use wav header
		String lower = mediaFilePath.toLowerCase();
		if (lower.endsWith(".wav") || lower.endsWith(".wave")) {
			return true;
		}
		
		return false;
	}

	public void dispose() {
		// TODO Auto-generated method stub

	}

	public JPanel getControlPanel() {
		return null;
	}

	public int getExamplesSupport() {
		return Recognizer.EXAMPLE_SEGMENTS_NOT_SUPPORTED;
	}

	public String getName() {
		return name;
	}

	public Object getParameterValue(String param) {
		return null;
	}

	public int getRecognizerType() {
		return Recognizer.AUDIO_TYPE;
	}

	public String getReport() {
		if (reportBuilder != null) {
			return reportBuilder.toString();
		}
		return null;
	}

	public boolean setMedia(List<String> mediaFilePaths) {
		this.mediaPaths = mediaFilePaths;
		if (mediaFilePaths != null && mediaFilePaths.size() >= 1) {
			currentMediaFile = mediaFilePaths.get(0);
			if (currentMediaFile.startsWith("file:")) {
				currentMediaFile = currentMediaFile.substring(5);
			}
			if (currentMediaFile.startsWith("///")) {
				currentMediaFile = currentMediaFile.substring(2);	
			}
			
			int slashIndex = currentMediaFile.lastIndexOf("/") + 1;
			if (slashIndex > 0) {
				currentMediaName = currentMediaFile.substring(slashIndex);
			}
			projectId = null;
		}
		return true;
	}

	public void setName(String name) {
		this.name = name;
	}

	public void setParameterValue(String param, String value) {
		if (param == null || param.length() == 0 || value == null || value.length() == 0) {
			return;
		}
		if (param.equals("xmloutput")) {
			outputFile = value;
		} else if (param.equals("base_url")) {
			baseUrl = value;
		}
	}

	public void setParameterValue(String param, float value) {

	}

	public void setRecognizerHost(RecognizerHost host) {
		this.host = host;

	}

	/**
	 * Starts the process: create a project, upload the file if necessary,
	 * start the recognizer, check the progress and retrieve the results.
	 */
	public void start() {
		if (isRunning) {
			return;
		}
		if (currentMediaFile == null) {
			host.errorOccurred("There is no audio file.");
			return;
		}
		interrupted = false;
		progress = 0f;
		if (reportBuilder != null) {
			reportBuilder.delete(0, reportBuilder.length());	
		}
		reportBuilder = new StringBuilder();
		startTime = System.currentTimeMillis();
		SimpleDateFormat df = new SimpleDateFormat("d MMM yyyy HH:mm:ss");
		reportBuilder.append("Starting process at " + df.format(new Date(startTime)) +"\n");
		reportBuilder.append("Server url:\t" + baseUrl + "\n");
		reportBuilder.append("Media file:\t" + currentMediaFile + "\n");
		reportBuilder.append("Output file: \t" + outputFile + "\n");
		
		// create valid id and project
		host.setProgress(progress, "Retrieving project ID");
		projectId = fileProjectMap.get(currentMediaFile);
		 		
		if (projectId == null) {
			host.setProgress(progress, "Creating project");
			reportBuilder.append("Creating a project...: ");
			projectId = createProjectID();
			
			if (projectId == null) {
				host.errorOccurred("Could not create project.");
				return;
			} else {
				fileProjectMap.put(currentMediaFile, projectId);
			}
		} else {
			reportBuilder.append("A project exists: ");
		}
		reportBuilder.append(projectId + "\n");
		progress = 0.05f;
		host.setProgress(progress, "Project exists or is created");
		
		if (interrupted) {
			host.setProgress(progress, "Process canceled.");
			reportBuilder.append("The process has been canceled after: " + 
					(System.currentTimeMillis() - startTime) / 1000 + " sec\n");
			return;
		}
		
		if (!uploadedProjects.contains(projectId)) {
			// upload audio file
			int uploadStatus = uploadFile(); // 200
			if (uploadStatus == HttpURLConnection.HTTP_OK) {
				host.setProgress(progress, "File uploaded");
				uploadedProjects.add(projectId);
				reportBuilder.append("The file has been uploaded, uploading took: " + 
						(System.currentTimeMillis() - startTime) / 1000 + " sec\n");
			} else {
				host.errorOccurred("File upload failed.");
				reportBuilder.append("Unable to upload the wav file to the server: " + uploadStatus + "\n");
				return;
			}
		} else {
			progress = 0.2f;
			host.setProgress(progress, "File already uploaded...");
			reportBuilder.append("The wave file is already available on the server\n");
		}
		
		if (interrupted) {
			host.setProgress(progress, "Process canceled.");
			reportBuilder.append("The process has been canceled after: " + 
					(System.currentTimeMillis() - startTime) / 1000 + " sec\n");
			return;
		}
		
		URL url = null;
		try {
			url = new URL(baseUrl+ projectId);
		} catch (MalformedURLException mue) {
			//ClientLogger.LOG.severe("Could not create the project url: " + mue.getMessage());
			reportBuilder.append("Could not create a project url\n");
			host.errorOccurred("Could not create the project url.");
			return;
		}
		// start the project (without further parameters)
		progress += 0.05f;
		long recStart = System.currentTimeMillis();
		reportBuilder.append("Starting the recognizer process...\n");
		int processStatus = startProject(url);
		if (processStatus == HttpURLConnection.HTTP_ACCEPTED) {
			progress += 0.05f;
			float remaining = 1 - progress - 0.1f;
			// start polling for status
			boolean procSucces = checkProgress(url, remaining);
			
			if (procSucces) {
				reportBuilder.append("Recognition process completed successfully, processing took: " 
						+ (System.currentTimeMillis() - recStart) / 1000 + " sec\n");
				host.setProgress(0.9f, "Fetching results");
			} else {
				host.errorOccurred("An error occurred during execution.");
				reportBuilder.append("Exiting after an error during execution...\n");
				return;
			}
		} else {
			//ClientLogger.LOG.severe("Could not start the recognizer process.");
			reportBuilder.append("Failed to start the recognizer process...\n");
			return;
		}
		
		if (interrupted) {
			host.setProgress(progress, "Process canceled.");
			reportBuilder.append("The process has been canceled after: " + 
					(System.currentTimeMillis() - startTime) / 1000 + " sec\n");
			return;
		}
		
		// download the results, if any
		boolean contents = getResult();
		if (!contents) {
			reportBuilder.append("Unable to download the results from the server.\n");
			host.errorOccurred("Unable to get the results.");
			return;
		} else {
			reportBuilder.append("Results downloaded.\n");
			host.setProgress(0.95f, "Retrieved the results");
		}
		// add segments to the host?
		reportBuilder.append("Process finished after: " + 
				(System.currentTimeMillis() - startTime) / 1000 + " sec\n");
		host.setProgress(1.0f, "Process finished.");
		// could delete the project as long as re-running the application 
		// (e.g. with different parameters) makes no sense.
	}

	public void stop() {
		// can stop anywhere while performing the steps mentioned in  start()
		// if the recognizer is already running on the server in can not be interrupted.
		interrupted = true;
	}

	public void updateLocale(Locale locale) {
	}

	/**
	 * The project id is a combination of a fixed string, ELAN and a timestamp
	 * @return the project id or null
	 */
	protected String createProjectID() {
		String nextId = null;
		int respCode = HttpURLConnection.HTTP_BAD_REQUEST;//400
		int retries = 0;
		
		do {
			try {
				nextId = "ELAN_" + String.valueOf(System.currentTimeMillis());
				URL url = new URL(baseUrl + nextId);
				
				respCode = invokeService(url, null, "PUT");
				
		        if (respCode != HttpURLConnection.HTTP_CREATED) {//201
					//ClientLogger.LOG.severe("Could not create project: " + nextId);
					reportBuilder.append("Unable to create a project: " + respCode + "\n");
					nextId = null;
		        }
			} catch (MalformedURLException mue) {
				//ClientLogger.LOG.severe("Could not contact the server: " + mue.getMessage());
				reportBuilder.append("Unable to create a project: " + mue.getMessage()  + "\n");
				nextId = null;
			}
			retries++;
		} while (respCode != HttpURLConnection.HTTP_CREATED && retries < 10);

		return nextId;
	}
	
	/**
	 * Method for a simple call of which only the return code is of interest.
	 * 
	 * @param url the url
	 * @param query the query string
	 * @param method the HTTP method
	 * @return the HTTP return code
	 */
	private int invokeService(URL url, String query, String method) {
		if (url == null) {
			//ClientLogger.LOG.severe("The server URL is null.");
			return HttpURLConnection.HTTP_BAD_REQUEST;//400
		}
		
		try {			
			HttpURLConnection httpConn = (HttpURLConnection) url.openConnection();
	        httpConn.setDefaultUseCaches(false);//??
	        
	        httpConn.setRequestMethod( (method != null ? method : "GET") );
	        httpConn.connect();
	
	        return httpConn.getResponseCode();	            
	        
		} catch (ProtocolException pe) {
			//ClientLogger.LOG.severe("Could not contact the server: " + pe.getMessage());
		} catch (IOException ioe) {
			//ClientLogger.LOG.severe("Could not contact the server: " + ioe.getMessage());
		}
		
		return HttpURLConnection.HTTP_BAD_REQUEST;
	}
	
	/**
	 * Starts the project, i.e. the recognizer
	 * @param url the project url
	 * @return an HTTP return code
	 */
	private int startProject(URL url) {
		host.setProgress(progress, "Starting the recognizer process...");

		if (url == null) {
			//ClientLogger.LOG.severe("The service url is null.");
			reportBuilder.append("The service url is null.\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		
		try {			
			HttpURLConnection httpConn = (HttpURLConnection) url.openConnection();
	        httpConn.setDefaultUseCaches(false);//??
	        httpConn.setRequestMethod("POST");
	        httpConn.connect();
	
	        int startCode = httpConn.getResponseCode();
	        if (startCode == HttpURLConnection.HTTP_ACCEPTED){
	        	// read the output
	        	/*
				BufferedReader procReader = new BufferedReader(new InputStreamReader(httpConn.getInputStream()));
				String line = null;
				while ((line = procReader.readLine()) != null) {
					//ClientLogger.LOG.info(line);
					System.out.println(line);
				}
				*/
				return startCode;
	        } else {
	        	reportBuilder.append("Unable to start the project: " + startCode + "\n");
	        	return startCode;
	        }
	        
		} catch (ProtocolException pe) {
			reportBuilder.append("Unable to start the project: " + pe.getMessage() + "\n");
		} catch (IOException ioe) {
			reportBuilder.append("Unable to start the project: " + ioe.getMessage() + "\n");
		}

		return HttpURLConnection.HTTP_BAD_REQUEST;
	}
	
	/**
	 * Checks the progress of the recognizer by successive calls to the service.
	 * Note: a time out should probably be introduced...
	 * 
	 * @param url the project url, no parameters are used
	 * @return true if the process completed successfully, false otherwise
	 */
	private boolean checkProgress(URL url, float timeRemain) {
		if (url == null) {
			//ClientLogger.LOG.severe("The service url is null.");
			return false;
		}
		reportBuilder.append("Checking the progress of the recognizer process...\n");
		HttpURLConnection httpConn = null;
		int lastCompletion = 0;
		boolean error = false;
		final String status = "<status";
		final String completion = "completion=";
		final String noError = "errors=\"no\"";
		
		while (!error) {
			try {
				httpConn = (HttpURLConnection) url.openConnection();
		        httpConn.setDefaultUseCaches(false);//??
		        httpConn.setRequestMethod("GET");
		        httpConn.connect();
		
		        int startCode = httpConn.getResponseCode();
		        if (startCode == HttpURLConnection.HTTP_OK) {
		        	// read the output
					BufferedReader procReader = new BufferedReader(new InputStreamReader(httpConn.getInputStream()));
					String line = null;
					while ((line = procReader.readLine()) != null) {
						//ClientLogger.LOG.info(line);
						//System.out.println(line);
						// output looks like: 
					    //<status code="1" message="Processing elan-example1.wav..." completion="0" errors="no" errormsg="">
				        //  <log time="26/May/2011 11:27:48" completion="0">Processing elan-example1.wav...</log>
						// code="2" might indicate completion=100?
						if (line.indexOf(status) > -1) {
							// first check error
							if (line.indexOf(noError) < 0) {// an error occurred
								error = true;
								
								String message = "";
								final String mesAttr = "errormsg";
								int mesIndex = line.indexOf(mesAttr);
								if (mesIndex > -1) {
									String rest = line.substring(mesIndex + mesAttr.length());
									int q1 = rest.indexOf("\"");
									if (q1 > -1) {
										int q2 = rest.indexOf("\"", q1 + 1);
										if (q2 > q1 + 1) {
											message = rest.substring(q1 + 1, q2);
											if (message.length() == 0) {
												message = "Unknown error occurred";
											}
										} else {
											message = rest.substring(q1);
										}
									} else {
										message = rest;//??
									}
								} else {
									message = "Unknown error occurred";
								}
								host.setProgress(progress + 
										timeRemain - ((100 - lastCompletion) * timeRemain), message);
								// stop reading, stop connection?
								return false;
							}  else {
								// no error, read completion
								int comIndex = line.indexOf(completion);
								if (comIndex > -1) {
									String com = line.substring(comIndex + completion.length());
									int q1 = com.indexOf("\"");
									if (q1 > -1) {
										int q2 = com.indexOf("\"", q1 + 1);
										if (q2 > q1 + 1) {
											String perc = com.substring(q1 + 1, q2);
											try {
												int percInt = Integer.parseInt(perc);
												if (percInt > lastCompletion) {
													lastCompletion = percInt;													
													host.setProgress(progress + 
															timeRemain - ((100 - lastCompletion) * timeRemain));
													if (lastCompletion == 100) {
														reportBuilder.append("Recognizer process completed successfully...\n");
														return true;
													}
												}
											} catch (NumberFormatException nfe) {}
										}
									}
								}
							}
						}
					}
					try {
						Thread.sleep(1500);
					} catch (InterruptedException ie) {}
		        }
		        if (interrupted) {
		        	reportBuilder.append("The process has been canceled, checking for progress has been stopped...\n");
		        	
		        	return false;
		        }
			} catch (ProtocolException pe) {
				reportBuilder.append("Could not contact the server: " + pe.getMessage() + "\n");
				error = true;
			} catch (IOException ioe) {
				reportBuilder.append("Could not contact the server: " + ioe.getMessage() + "\n");
				error = true;
			}
		}
		reportBuilder.append("An unknown error occurred while checking the progress of the recognizer...\n");
		return !error;
	}
	
	/**
	 * Downloads the result from the server and writes it to file.
	 * 
	 * @return true if the results have successfully been retrieved and stored, false otherwise
	 */
	private boolean getResult() {
		URL url = null;
		try {
			url = new URL(baseUrl+ projectId + "/output/" + currentMediaName + ".out");
		} catch (MalformedURLException mue) {
			reportBuilder.append("Could not create the input file url: " + mue.getMessage() + "\n");
			return false;
		}
		if (url == null) {
			//ClientLogger.LOG.severe("The service url is null.");
			return false;
		}
		HttpURLConnection httpConn = null;
		try {
			httpConn =  (HttpURLConnection) url.openConnection();
	        httpConn.setDefaultUseCaches(false);//??
	        httpConn.setUseCaches(false);
	        httpConn.setDoOutput(true);
	        httpConn.setRequestMethod("GET");
	        httpConn.connect();
			int respCode = httpConn.getResponseCode();
			
			if (respCode != HttpURLConnection.HTTP_OK) {
				reportBuilder.append("Unable to get the results: " + respCode + "\n");
				host.errorOccurred("Unable to get the results.");
			} else {
				BufferedReader procReader = new BufferedReader(new InputStreamReader(httpConn.getInputStream()));
				BufferedWriter bufWriter = null;
				try {
					bufWriter = new BufferedWriter(new OutputStreamWriter(
							new FileOutputStream(outputFile), "UTF-8"));
				} catch (IOException iie) {
					System.out.println("Cannot write to file: " + outputFile + " " + iie.getMessage());
					try {
						bufWriter = new BufferedWriter(new OutputStreamWriter(
								new FileOutputStream(currentMediaFile + ".xml"), "UTF-8"));
					} catch (IOException iie2) {
						System.out.println("Cannot write to file: " + (currentMediaFile + ".xml") + " " + iie2.getMessage());
					}
				}
				if (bufWriter == null) {
					return false;
				}
				
				String line = null;
				final String nl = "\n";
				final String end = "</TIER>";
				boolean xmlStarted = false;
				//StringBuilder builder = new StringBuilder(1000);
				
				while ((line = procReader.readLine()) != null) {
					if (!xmlStarted && line.startsWith("<?xml")) {
						//builder.append(line);
						//builder.append(nl);
						bufWriter.write(line);
						bufWriter.write(nl);
						xmlStarted = true;
						continue;
					}
					if (xmlStarted) {
						//builder.append(line);
						//builder.append(nl);
						bufWriter.write(line);
						bufWriter.write(nl);
						if (line.indexOf(end) > -1) {
							break;
						}
						if (interrupted) {
							bufWriter.write(end);
							reportBuilder.append("The process has been canceled while retrieving the results. Incomplete results have been stored.\n");
							
							break;
						}
					}
				}
				try {
					bufWriter.flush();
					bufWriter.close();
					procReader.close();
				} catch (IOException oe){
				}
				
				return true;
			}
		} catch (ProtocolException pe) {
			reportBuilder.append("Could not contact the server: " + pe.getMessage() + "\n");
		} catch (IOException ioe) {
			reportBuilder.append("Could not contact the server: " + ioe.getMessage() + "\n");
		}
		
		return false;
	}
	
	/**
	 * Uploads a file to the project.
	 * Takes 40% of the progress time.
	 * @return the HTTP response code, 400 (BAD_REQUEST) is used for any other error (null pointer, io etc.)
	 */
	private int uploadFile() {
		if (currentMediaFile == null) {
			reportBuilder.append("There is no audio file url.\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		host.setProgress(progress, "Uploading wav file");
		URL url = null;
		try {
			// template id is either wavinput or WaveFileFormat ??
			url = new URL(baseUrl+ projectId + "/input/" + currentMediaName + 
					"?inputtemplate=wavinput&file=" + currentMediaName);
		} catch (MalformedURLException mue) {
			reportBuilder.append("Could not create the input file url: " + mue.getMessage() + "\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		if (url == null) {
			reportBuilder.append("Could not create the input file url.\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		final String boundary = "DaDa0x";
		final String nl = "\r\n";
		
		
		StringBuilder preFileString = new StringBuilder("--" + boundary + nl);
		preFileString.append("content-disposition: form-data; name=\"file\";"
			      + " filename=\"" + currentMediaName +"\"" + nl);
		preFileString.append(nl);
		byte[] preBytes = preFileString.toString().getBytes();
		byte[] postBytes = (nl + "--" + boundary + "--" + nl).getBytes();
		File f = new File(currentMediaFile);
		long fileLength = 0;
		if (f.exists() && !f.isDirectory()) {
			fileLength = f.length();
		} else {
			reportBuilder.append("Errro: the file does not exist\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		
		try {
			HttpURLConnection httpConn = (HttpURLConnection) url.openConnection();
	        httpConn.setDefaultUseCaches(false);//??
	        httpConn.setUseCaches(false);
	        httpConn.setDoInput(true);
	        httpConn.setDoOutput(true);
	        httpConn.setRequestMethod("POST");
	        httpConn.setInstanceFollowRedirects( false );
	        //httpConn.setChunkedStreamingMode(1024 * 10);
	        httpConn.setFixedLengthStreamingMode((int) (preBytes.length + fileLength + postBytes.length));
	        httpConn.setRequestProperty( "Connection", "Keep-Alive");
	        httpConn.setRequestProperty( "Content-Type", "multipart/form-data;boundary=" + boundary);
	        //httpConn.connect();
	        FileInputStream fis = null;
	        
	        DataOutputStream dos = null;
	        // reserve an estimated 40% of the process for file upload
	        final float startProg = progress;
	        float progPart = 0.4f;
	        try {
	        	//File f = new File(currentMediaFile);
	        	
	        	fis = new FileInputStream(f);
	        	dos = new DataOutputStream(httpConn.getOutputStream());
	        	
//	        	dos.writeBytes("--" + boundary + nl);
//	        	dos.writeBytes("content-disposition: form-data; name=\"file\";"
//	        			      + " filename=\"" + currentMediaName +"\"" + nl);
//	        	dos.writeBytes(nl);
	        	dos.write(preBytes);
	        			   
	        	//long fileLength = f.length();
	        	//System.out.println("File bytes: " + fileLength);
	        	int numBytesRead;
	        	int totalBytesRead = 0;
	        	byte[] chunk = new byte[1024 * 1024];	        	
	        	
	        	while ((numBytesRead = fis.read(chunk, 0, chunk.length)) != -1 && !interrupted) {
	        		dos.write(chunk, 0, numBytesRead);
	        		//System.out.println("Writing bytes: " + numBytesRead);
	        		dos.flush();
	        		totalBytesRead += numBytesRead;
	        		//System.out.println("Total bytes: " + totalBytesRead);
	        		if (fileLength > 0) {
	        			progress = startProg + (progPart * ((float) totalBytesRead / fileLength));
	        			//System.out.println("Progress: " + progress);
	        			host.setProgress(progress);
	        		}
	        	}
	        	if (interrupted) {
	        		reportBuilder.append("The process has been canceled, uploading has been stopped...\n");
	        	}
//	        	dos.writeBytes(nl);
//	        	dos.writeBytes("--" + boundary + "--" + nl);
	        	// if interrupted this will cause an error status code being returned by the server
	        	dos.write(postBytes);
	        	dos.flush();
	        	host.setProgress(progress, "File uploaded, waiting for server confirmation.");
	        } catch (IOException ioe) {
	        	reportBuilder.append("An error occurred while uploading the audio file: " + ioe.getMessage() + "\n");
	        	return HttpURLConnection.HTTP_BAD_REQUEST;
	        } finally {
	        	if (dos != null) {
	        		dos.flush();
	        		dos.close();
	        	}
	        	if (fis != null) {
	        		fis.close();
	        	}
	        }
	        
// BufferedOutputStream	   
	        /*
	        BufferedOutputStream bos = null;
	        // reserve an estimated 40% of the process for file upload
	        final float startProg = progress;
	        float progPart = 0.4f;
	        try {
	        	File f = new File(currentMediaFile);
	        	
	        	fis = new FileInputStream(f);
	        	bos = new BufferedOutputStream(httpConn.getOutputStream());
	        	
	        	byte[] stringBytes = ("--" + boundary + nl).getBytes();
	        	bos.write(stringBytes, 0, stringBytes.length);
	        	stringBytes = ("content-disposition: form-data; name=\"file\";"
      			      + " filename=\"" + currentMediaName +"\"" + nl).getBytes();
	        	bos.write(stringBytes, 0, stringBytes.length);
	        	stringBytes = (nl).getBytes();
	        	bos.write(stringBytes, 0, stringBytes.length);
	        			   
	        	long fileLength = f.length();
	        	int numBytesRead;
	        	int totalBytesRead = 0;
	        	byte[] chunk = new byte[1024 * 10];	        	
	        	
	        	while ((numBytesRead = fis.read(chunk, 0, chunk.length)) != -1) {
	        		bos.write(chunk, 0, numBytesRead);
	        		System.out.println("Writing bytes: " + numBytesRead);
	        		bos.flush();
	        		totalBytesRead += numBytesRead;
	        		if (fileLength > 0) {
	        			progress = startProg + (progPart * (totalBytesRead / fileLength));
	        			host.setProgress(progress);
	        		}
	        	}
	        	// new line
	        	bos.write(stringBytes, 0, stringBytes.length);
	        	stringBytes = ("--" + boundary + "--" + nl).getBytes();
	        	bos.write(stringBytes, 0, stringBytes.length);
	        	host.setProgress(progress, "File uploaded, waiting for server confirmation.");
	        } catch (IOException ioe) {
	        	ClientLogger.LOG.severe("An error occurred while uploading the audio file: " + ioe.getMessage());
	        	return HttpURLConnection.HTTP_BAD_REQUEST;
	        } finally {
	        	if (bos != null) {
	        		bos.flush();
	        		bos.close();
	        	}
	        	if (fis != null) {
	        		fis.close();
	        	}
	        }       
	        // BufferedOutputStream
	         */
	        
	        int responseCode = httpConn.getResponseCode();
	        if (responseCode != HttpURLConnection.HTTP_OK) {
	        	reportBuilder.append("An error occurred while uploading the audio file: " + responseCode + "\n");
	        } else {
//				BufferedReader procReader = new BufferedReader(new InputStreamReader(httpConn.getInputStream()));
//				String line = null;
//				while ((line = procReader.readLine()) != null) {
//					ClientLogger.LOG.info(line);
//				}
				// return contents something like
				//<clamupload>
				//<upload source="pear.wav" filename="pear.wav" inputtemplate="wavinput" templatelabel="Input wave file">
				//<parameters errors="no"></parameters><valid>yes</valid></upload>
				//</clamupload>
	        }
	        
	        return responseCode;	        
		} catch (ProtocolException pe) {
			reportBuilder.append("Could not contact the server: " + pe.getMessage() + "\n");
		} catch (IOException ioe) {
			reportBuilder.append("Could not contact the server: " + ioe.getMessage() + "\n");
		}
		
		return HttpURLConnection.HTTP_BAD_REQUEST;
	}
}
