package nl.mpi.recognizer.remote.maus;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.swing.JPanel;

import mpi.eudico.client.annotator.recognizer.api.Recognizer;
import mpi.eudico.client.annotator.recognizer.api.RecognizerHost;
import mpi.eudico.client.annotator.recognizer.data.RSelection;
import mpi.eudico.client.annotator.recognizer.data.Segment;
import mpi.eudico.client.annotator.recognizer.data.Segmentation;

/**
 * A class that takes care of uploading a wav file and a text file containing 
 * an unaligned transcript and that transforms the returned Praat TextGrid
 * into segmentations/annotations.
 * Input: mono wav file, 16 bits, audio/x-wav
 * Input: orthographic text, charset=UTF-8, text/plain
 * Output: aligned annotations, Praat TextGrid file
 * 
 * @author Han Sloetjes
 *
 */
public class MausWsClient implements Recognizer {
	protected RecognizerHost host;
	private String name = "MAUS web service client";
	protected List<String> mediaPaths;
	private String baseUrl = "http://webapp.phonetik.uni-muenchen.de/BASWebServices/services/";
	private String baseServiceId = "runMAUSBasicGerman";// previously "processfilesotf";
	// for the situation that a file is uploaded first and referenced later, not used
	//private String uploadService = "uploadfile";
	private String signalParamValue = "";// not used if the media have been set via setMedia
	private String textParamName = "TEXT"; // the name of the parameter for the text to upload
	private String signalParamName = "SIGNAL"; // the name of the parameter for the audio to upload 
	
	//private String projectId = null;
	private String currentMediaFile = null;
	private String currentMediaName = null;
	private String currentTextFile = null;
	private String currentTextName = null;
	private String outputFile = null;
	private Map<String, String> parameters;
	
	private float progress = 0f;
	private boolean isRunning = false;
	private boolean interrupted = false;
	private StringBuilder reportBuilder = null;
	private long startTime;
	private List<Segmentation> segmentations;

	/**
	 * Constructor.
	 */
	public MausWsClient() {
		super();
	}

	public boolean canCombineMultipleFiles() {
		return false;
	}

	public boolean canHandleMedia(String mediaFilePath) {
		if (mediaFilePath == null) {
			return false;
		}
		// check extension, or use wav header
		String lower = mediaFilePath.toLowerCase();
		if (lower.endsWith(".wav") || lower.endsWith(".wave")) {
			// check whether it is a mono 16 bits wav file
			return true;
		}
		
		return false;
	}

	public void dispose() {
		// TODO Auto-generated method stub

	}

	public JPanel getControlPanel() {
		return null;
	}

	/**
	 * A text (file) needs to be uploaded. Could be extracted from a tier,
	 * can be just a file.
	 * EXAMPLE_SEGMENTS_REQUIRED could be returned to indicate that tier input is accepted
	 */
	public int getExamplesSupport() {
		return Recognizer.EXAMPLE_SEGMENTS_OPTIONAL;
	}

	public String getName() {
		return name;
	}

	public Object getParameterValue(String param) {
		return null;
	}

	public int getRecognizerType() {
		return Recognizer.AUDIO_TYPE;
	}

	public String getReport() {
		if (reportBuilder != null) {
			return reportBuilder.toString();
		}
		return null;
	}

	public boolean setMedia(List<String> mediaFilePaths) {
		this.mediaPaths = mediaFilePaths;
		if (mediaFilePaths != null && mediaFilePaths.size() >= 1) {
			currentMediaFile = mediaFilePaths.get(0);
			if (currentMediaFile.startsWith("file:")) {
				currentMediaFile = currentMediaFile.substring(5);
			}
			if (currentMediaFile.startsWith("///")) {
				currentMediaFile = currentMediaFile.substring(2);	
			}
			
			int slashIndex = currentMediaFile.lastIndexOf("/") + 1;
			if (slashIndex > 0) {
				currentMediaName = currentMediaFile.substring(slashIndex);
			}
		}
		return true;
	}

	public void setName(String name) {
		this.name = name;
	}

	public void setParameterValue(String param, String value) {
		if (param == null || param.length() == 0 || value == null || value.length() == 0) {
			return;
		}
		if (param.equals("SignalUrl")) {
			signalParamValue = value;
		} else if (param.equals("base_url")) {
			baseUrl = value;
		} else if (param.equals("TextUrl")) {
			currentTextFile = value;
			int slashIndex = currentTextFile.lastIndexOf("/") + 1;
			if (slashIndex > 0) {
				currentTextName = currentTextFile.substring(slashIndex);
			}
		} else if (param.equals("TextParamName")) {
			textParamName = value;
		} else if (param.equals("SignalParamName")) {
			signalParamName = value;
		} else if (param.equals("ServiceName")) {
			baseServiceId = value;
		} else {// the language id e.g.
			if (parameters == null) {
				parameters = new HashMap<String, String>(2);
			}
			parameters.put(param, value);
		}
	}

	public void setParameterValue(String param, float value) {

	}

	public void setRecognizerHost(RecognizerHost host) {
		this.host = host;

	}

	/**
	 * Starts the process: create a project, upload the file if necessary,
	 * start the recognizer, check the progress and retrieve the results.
	 */
	public void start() {
		if (isRunning) {
			return;
		}
		if (currentMediaFile == null) {
			host.errorOccurred("There is no audio file.");
			return;
		}
		interrupted = false;
		progress = 0f;
		if (reportBuilder != null) {
			reportBuilder.delete(0, reportBuilder.length());	
		}
		reportBuilder = new StringBuilder();
		startTime = System.currentTimeMillis();
		SimpleDateFormat df = new SimpleDateFormat("d MMM yyyy HH:mm:ss");
		reportBuilder.append("Starting process at " + df.format(new Date(startTime)) +"\n");
		reportBuilder.append("Server url:\t" + baseUrl + baseServiceId + "\n");
		reportBuilder.append("Media file:\t" + currentMediaFile + "\n");
		reportBuilder.append("Text file:\t" + currentTextFile + "\n");
		//reportBuilder.append("Output file: \t" + outputFile + "\n");
		
		// create valid id and project
		host.setProgress(progress, "Retrieving project ID");

		progress = 0.05f;
		host.setProgress(progress, "Project exists or is created");
		
		if (interrupted) {
			host.setProgress(progress, "Process canceled.");
			reportBuilder.append("The process has been canceled after: " + 
					(System.currentTimeMillis() - startTime) / 1000f + " sec\n");
			return;
		}	

		// upload audio file + text 
		//int uploadStatus = uploadFile(); // 200
		int uploadStatus = uploadAllFiles(); // 200
		if (uploadStatus == HttpURLConnection.HTTP_OK) {
			host.setProgress(progress, "Files processed");
		} else {
			host.errorOccurred("Processing files failed.");
			reportBuilder.append("Unable to process the files: " + uploadStatus + "\n");
			return;
		}
		
		// add segments to the host?
		if (segmentations != null) {
			reportBuilder.append("Number of tiers created: " + segmentations.size() + "\n");
			for (Segmentation s : segmentations) {
				host.addSegmentation(s);
			}
		} else {
			reportBuilder.append("Number of tiers created: 0\n");
		}
		reportBuilder.append("Process finished after: " + 
				(System.currentTimeMillis() - startTime) / 1000f + " sec\n");
		host.setProgress(1.0f, "Process finished.");
		// could delete the project as long as re-running the application 
		// (e.g. with different parameters) makes no sense.
	}

	public void stop() {
		// can stop anywhere while performing the steps mentioned in  start()
		// if the recognizer is already running on the server in can not be interrupted.
		interrupted = true;
	}

	public void updateLocale(Locale locale) {
	}
	
	/**
	 * Uploads a single file to the server. The returned xml contains the url to the file on the server.
	 * 
	 * Takes 40% of the progress time for the wav, 10% for the text file.
	 * @return the HTTP response code, 400 (BAD_REQUEST) is used for any other error (null pointer, io etc.)
	 */
	/*
	private int uploadFile() {
		if (currentMediaFile == null) {
			reportBuilder.append("There is no audio file url.\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		host.setProgress(progress, "Uploading wav file");
		URL url = null;
		try {
			url = new URL(baseUrl+ uploadService);
		} catch (MalformedURLException mue) {
			reportBuilder.append("Could not create the input file url: " + mue.getMessage() + "\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		if (url == null) {
			reportBuilder.append("Could not create the input file url.\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		final String boundary = "DaDa0x";
		final String nl = "\r\n";
		
		
		StringBuilder preFileString = new StringBuilder("--" + boundary + nl);
		preFileString.append("content-disposition: form-data; name=\"signalParamName\";"
			      + " filename=\"" + currentMediaName +"\"" + nl);
		preFileString.append(nl);
		byte[] preBytes = preFileString.toString().getBytes();
		byte[] postBytes = (nl + "--" + boundary + "--" + nl).getBytes();
		File f = new File(currentMediaFile);
		long fileLength = 0;
		if (f.exists() && !f.isDirectory()) {
			fileLength = f.length();
		} else {
			reportBuilder.append("Error: the file does not exist\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		
		try {
			HttpURLConnection httpConn = (HttpURLConnection) url.openConnection();
	        httpConn.setDefaultUseCaches(false);//??
	        httpConn.setUseCaches(false);
	        httpConn.setDoInput(true);
	        httpConn.setDoOutput(true);
	        httpConn.setRequestMethod("POST");
	        httpConn.setInstanceFollowRedirects( false );
	        //httpConn.setChunkedStreamingMode(1024 * 10);
	        httpConn.setFixedLengthStreamingMode((int) (preBytes.length + fileLength + postBytes.length));
	        httpConn.setRequestProperty( "Connection", "Keep-Alive");
	        httpConn.setRequestProperty( "Content-Type", "multipart/form-data;boundary=" + boundary);
	        //httpConn.connect();
	        FileInputStream fis = null;
	        
	        DataOutputStream dos = null;
	        // reserve an estimated 40% of the process for file upload
	        final float startProg = progress;
	        float progPart = 0.4f;
	        try {
	        	//File f = new File(currentMediaFile);
	        	
	        	fis = new FileInputStream(f);
	        	dos = new DataOutputStream(httpConn.getOutputStream());
	        	
//	        	dos.writeBytes("--" + boundary + nl);
//	        	dos.writeBytes("content-disposition: form-data; name=\"file\";"
//	        			      + " filename=\"" + currentMediaName +"\"" + nl);
//	        	dos.writeBytes(nl);
	        	dos.write(preBytes);
	        			   
	        	//long fileLength = f.length();
	        	//System.out.println("File bytes: " + fileLength);
	        	int numBytesRead;
	        	int totalBytesRead = 0;
	        	byte[] chunk = new byte[1024 * 1024];	        	
	        	
	        	while ((numBytesRead = fis.read(chunk, 0, chunk.length)) != -1 && !interrupted) {
	        		dos.write(chunk, 0, numBytesRead);
	        		//System.out.println("Writing bytes: " + numBytesRead);
	        		dos.flush();
	        		totalBytesRead += numBytesRead;
	        		//System.out.println("Total bytes: " + totalBytesRead);
	        		if (fileLength > 0) {
	        			progress = startProg + (progPart * ((float) totalBytesRead / fileLength));
	        			//System.out.println("Progress: " + progress);
	        			host.setProgress(progress);
	        		}
	        	}
	        	if (interrupted) {
	        		reportBuilder.append("The process has been canceled, uploading has been stopped...\n");
	        	}
//	        	dos.writeBytes(nl);
//	        	dos.writeBytes("--" + boundary + "--" + nl);
	        	// if interrupted this will cause an error status code being returned by the server
	        	dos.write(postBytes);
	        	dos.flush();
	        	host.setProgress(progress, "File uploaded, waiting for server confirmation.");
	        } catch (IOException ioe) {
	        	reportBuilder.append("An error occurred while uploading the audio file: " + ioe.getMessage() + "\n");
	        	return HttpURLConnection.HTTP_BAD_REQUEST;
	        } finally {
	        	if (dos != null) {
	        		dos.flush();
	        		dos.close();
	        	}
	        	if (fis != null) {
	        		fis.close();
	        	}
	        }
	        

	        
	        int responseCode = httpConn.getResponseCode();
	        if (responseCode != HttpURLConnection.HTTP_OK) {
	        	reportBuilder.append("An error occurred while uploading the audio file: " + responseCode + "\n");
	        } else {
				BufferedReader procReader = new BufferedReader(new InputStreamReader(httpConn.getInputStream()));
				String line = null;
				while ((line = procReader.readLine()) != null) {
					System.out.println(line);
				}
				// returned contents something like, can extract the url of the uploaded file
//				<result>
//					<response id="systemstate">
//						serviceperformed
//					</response>
//					<text>http://webapp.phonetik.uni-muenchen.de:80/BASWebServices/07BF5F7203225510AB57DBC2DDC06FC8-audio.wav</text>
//				</result>
	        }
	        
	        return responseCode;	        
		} catch (ProtocolException pe) {
			reportBuilder.append("Could not contact the server: " + pe.getMessage() + "\n");
		} catch (IOException ioe) {
			reportBuilder.append("Could not contact the server: " + ioe.getMessage() + "\n");
		}
		
		return HttpURLConnection.HTTP_BAD_REQUEST;
	}
	*/
	
	/**
	 * Uploads the two files, the wav file and the text file.
	 * Uploading takes 50% of the progress time (very rough estimation), then waiting for the response starts.
	 * @return the HTTP response code, 400 (BAD_REQUEST) is used for any other error (null pointer, io etc.)
	 */
	private int uploadAllFiles() {
		if (currentMediaFile == null) {
			reportBuilder.append("There is no audio file url.\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		ArrayList<RSelection> tierSelectionObjects = host.getSelections();

		// let a selected tier take precedence over a selected text file 
		if ((tierSelectionObjects == null || tierSelectionObjects.size() == 0) && currentTextFile == null) {// check text contents
			reportBuilder.append("There is no text input, neither from a selected tier nor from file url.\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;			
		}
		
		host.setProgress(progress, "Uploading wav and text file");
		URL url = null;
		if (parameters == null || parameters.size() == 0) {
			try {
				url = new URL(baseUrl+ baseServiceId);
			} catch (MalformedURLException mue) {
				reportBuilder.append("Could not create the input service url: " + mue.getMessage() + "\n");
				return HttpURLConnection.HTTP_BAD_REQUEST;
			}
		} else {
			// append params to url
			StringBuilder builder = new StringBuilder(baseUrl);
			builder.append(baseServiceId);
			builder.append('?');
			Iterator<String> paramIter = parameters.keySet().iterator();
			String key = null;
			String val = null;
			int i = 0;
			
			while (paramIter.hasNext()) {
				key = paramIter.next();
				val = parameters.get(key);
				if (key != null && val != null) {
					if (i > 0) {// don't add a & for the first parameter
						builder.append('&');
					}
					builder.append(key);
					builder.append('=');
					builder.append(val);
				}
				i++;
			}
			try {
				url = new URL(builder.toString());
			} catch (MalformedURLException mue) {
				reportBuilder.append("Could not create the input service url: " + mue.getMessage() + "\n");
				return HttpURLConnection.HTTP_BAD_REQUEST;
			}
		}
		
		if (url == null) {
			reportBuilder.append("Could not create the input service url.\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		
		final String boundary = "DaDa0x";
		final String nl = "\r\n";
		long totalLength = 0;
		
		StringBuilder preFileString = new StringBuilder("--" + boundary + nl);
		preFileString.append("content-disposition: form-data; name=\"" + signalParamName + "\";"
			      + " filename=\"" + currentMediaName +"\"" + nl);
		preFileString.append(nl);
		byte[] preBytes = preFileString.toString().getBytes();
		totalLength += preBytes.length;
		File f = new File(currentMediaFile);
		long fileLength = 0;
		
		if (f.exists() && !f.isDirectory()) {
			fileLength = f.length();
			totalLength += f.length();
		} else {
			reportBuilder.append("Error: the wav file does not exist\n");
			return HttpURLConnection.HTTP_BAD_REQUEST;
		}
		
		// text file or text contents
		byte[] inputText = null;
		File tf = null;
		StringBuilder subFileString = null; // the string describing the second file's content
		
		if (tierSelectionObjects != null && tierSelectionObjects.size() > 0) {
			inputText = convertToString(tierSelectionObjects);
			if (inputText != null) {
				totalLength += inputText.length;
				fileLength += inputText.length;
				subFileString = new StringBuilder(nl + "--" + boundary + nl);
				subFileString.append("content-disposition: form-data; name=\"" + textParamName + "\";"
					      + " filename=\"" + "tierText.txt" +"\"" + nl);
				subFileString.append(nl);
			}
		}
		
		if (inputText == null) {// use file
			tf = new File(currentTextFile);
			if (tf.exists() && !tf.isDirectory()) {
				totalLength += tf.length();
				fileLength += tf.length();
				subFileString = new StringBuilder(nl + "--" + boundary + nl);
				subFileString.append("content-disposition: form-data; name=\"" + textParamName + "\";"
					      + " filename=\"" + currentTextName +"\"" + nl);
				subFileString.append(nl);
			} else {
				reportBuilder.append("Error: the text file does not exist\n");
				return HttpURLConnection.HTTP_BAD_REQUEST;
			}
		}
		
		byte[] subBytes = subFileString.toString().getBytes();
		totalLength += subBytes.length;
		// end text file portion
		
		byte[] postBytes = (nl + "--" + boundary + "--" + nl).getBytes();
		totalLength += postBytes.length;
		
		try {
			HttpURLConnection httpConn = (HttpURLConnection) url.openConnection();
	        httpConn.setDefaultUseCaches(false);//??
	        httpConn.setUseCaches(false);
	        httpConn.setDoInput(true);
	        httpConn.setDoOutput(true);
	        httpConn.setRequestMethod("POST");
	        httpConn.setInstanceFollowRedirects( false );
	        //httpConn.setChunkedStreamingMode(1024 * 10);
	        httpConn.setFixedLengthStreamingMode((int) totalLength);
	        httpConn.setRequestProperty("User-Agent", "ELAN");
	        httpConn.setRequestProperty( "Connection", "Keep-Alive");
	        httpConn.setRequestProperty( "Content-Type", "multipart/form-data;boundary=" + boundary);
	        //httpConn.connect();
	        FileInputStream fis = null;
	        FileInputStream fis2 = null;
	        
	        DataOutputStream dos = null;
	        // reserve an estimated 50% of the process for file upload
	        final float startProg = progress;
	        float progPart = 0.5f;
	        try {
	        	fis = new FileInputStream(f);
	        	dos = new DataOutputStream(httpConn.getOutputStream());

	        	dos.write(preBytes);
	        			   
	        	//long fileLength = f.length();
	        	//System.out.println("File bytes: " + fileLength);
	        	int numBytesRead;
	        	int totalBytesRead = 0;
	        	byte[] chunk = new byte[1024 * 1024];
	        	
	        	while ((numBytesRead = fis.read(chunk, 0, chunk.length)) != -1 && !interrupted) {
	        		dos.write(chunk, 0, numBytesRead);
	        		//System.out.println("Writing bytes: " + numBytesRead);
	        		dos.flush();
	        		totalBytesRead += numBytesRead;
	        		//System.out.println("Total bytes: " + totalBytesRead);
	        		if (fileLength > 0) {
	        			progress = startProg + (progPart * ((float) totalBytesRead / fileLength));
	        			//System.out.println("Progress: " + progress);
	        			host.setProgress(progress);
	        		}
	        	}
	        	if (interrupted) {
	        		reportBuilder.append("The process has been canceled, uploading has been stopped...\n");
	        		return HttpURLConnection.HTTP_BAD_REQUEST;
	        	}
	        	// file 2 or text string
	        	dos.write(subBytes);
	        	
	        	if (inputText != null) {
	        		numBytesRead = 0;
	        		if (inputText.length <= chunk.length) {
	        			dos.write(inputText);
	        			dos.flush();
	        			totalBytesRead += inputText.length;
		        		if (fileLength > 0) {
		        			progress = startProg + (progPart * ((float) totalBytesRead / fileLength));
		        			//System.out.println("Progress: " + progress);
		        			host.setProgress(progress);
		        		}
	        		} else {
	        			int count = inputText.length / chunk.length;
//	        			System.out.println("input length: " + inputText.length);
//	        			System.out.println("chunk length: " + chunk.length);
//	        			System.out.println("number of write turns: " + count);
	        			for (int i = 0; i < count; i++) {
	        				dos.write(inputText, i * chunk.length, chunk.length);
	        				dos.flush();
		        			totalBytesRead += chunk.length;
		        			numBytesRead += chunk.length;
		        			//System.out.println("Written: " + numBytesRead);
			        		if (fileLength > 0) {
			        			progress = startProg + (progPart * ((float) totalBytesRead / fileLength));
			        			//System.out.println("Progress: " + progress);
			        			host.setProgress(progress);
			        		}
	        			}
	        			// write remaining text
	        			if (numBytesRead < inputText.length) {
	        				//System.out.println("Remainder: " + (inputText.length - numBytesRead));
	        				dos.write(inputText, numBytesRead, inputText.length - numBytesRead);
	        				dos.flush();
	        			}
	        		}
	        	} else {
		        	fis2 = new FileInputStream(tf);
		        	numBytesRead = 0;
		        	while ((numBytesRead = fis2.read(chunk, 0, chunk.length)) != -1 && !interrupted) {
		        		dos.write(chunk, 0, numBytesRead);
		        		//System.out.println("Writing bytes: " + numBytesRead);
		        		dos.flush();
		        		totalBytesRead += numBytesRead;
		        		//System.out.println("Total bytes: " + totalBytesRead);
		        		if (fileLength > 0) {
		        			progress = startProg + (progPart * ((float) totalBytesRead / fileLength));
		        			//System.out.println("Progress: " + progress);
		        			host.setProgress(progress);
		        		}
		        	}
	        	}

        		progress = startProg + progPart;
        		//System.out.println("Progress: " + progress);
        		host.setProgress(progress);

	        	dos.write(postBytes);
	        	dos.flush();
	        	host.setProgress(progress, "Files uploaded, waiting for server response.");
	    		reportBuilder.append("Uploading of files took: " + 
	    				(System.currentTimeMillis() - startTime) / 1000f + " sec\n");
	        } catch (IOException ioe) {
	        	reportBuilder.append("An error occurred while uploading the audio file or text file: " + ioe.getMessage() + "\n");
	        	return HttpURLConnection.HTTP_BAD_REQUEST;
	        } finally {
	        	if (dos != null) {
	        		dos.flush();
	        		dos.close();
	        	}
	        	if (fis != null) {
	        		fis.close();
	        	}
	        }
	        
	        if (interrupted) {
	        	return HttpURLConnection.HTTP_BAD_REQUEST;
	        }
	        
	        // waiting for response
	        host.setProgress(-1, "Waiting for results...");
	        
	        int responseCode = httpConn.getResponseCode();
	        if (responseCode != HttpURLConnection.HTTP_OK) {
	        	reportBuilder.append("An error occurred while uploading the audio file or text file: " + responseCode + "\n");
	        } else {
				BufferedReader procReader = new BufferedReader(new InputStreamReader(httpConn.getInputStream(), "UTF-8"));
				// returned contents is a Praat TextGrid file in an xml stream
				int parseCode = parseResults(procReader);
				// do something with the return value
				if (parseCode != HttpURLConnection.HTTP_OK) {
					responseCode = parseCode;//??
				}
	        }
	        
	        return responseCode;	        
		} catch (ProtocolException pe) {
			reportBuilder.append("Could not contact the server: " + pe.getMessage() + "\n");
		} catch (IOException ioe) {
			reportBuilder.append("Could not contact the server: " + ioe.getMessage() + "\n");
		}
		
		return HttpURLConnection.HTTP_BAD_REQUEST;
	}
	
	/**
	 * Builds a string from a sequence of selections. 
	 * @param tierSelectionObjects list of selections, is assumed not to be null
	 * @return a String or null
	 */
	private byte[] convertToString(ArrayList<RSelection> tierSelectionObjects) {
		StringBuilder sb = new StringBuilder();
		String label = null;
		
		for (RSelection rsel : tierSelectionObjects) {
			if (rsel instanceof Segment) {
				label = ((Segment) rsel).label;
				if (label != null) {
					sb.append(label);
					sb.append(" ");// how about punctuation?
				}
			}
		}
		
		return sb.toString().getBytes();// getBytes("UTF-8")
	}

//#############################################################
// TextGrid parsing could reuse existing class 
// but this way it is easier to deploy as an extension
//#############################################################
    private final String eq = "=";
    private final String tierSpec = "IntervalTier";
    private final String nm = "name";
    private final String min = "xmin";
    private final String max = "xmax";
    private final String tx = "text";
    
    
	private int parseResults(BufferedReader reader) throws IOException {
		if (reader == null) {
			return HttpURLConnection.HTTP_NO_CONTENT;
		}
		reportBuilder.append("Parsing results started after: " + 
				(System.currentTimeMillis() - startTime) / 1000f + " sec\n");
		host.setProgress(0.9f, "Parsing results");
		// returned contents is something like this, a Praat TextGrid file in an xml stream
//		<result> // xml removed 26 jan 2012?
//		<response id="systemstate">
//		serviceperformed
//		</response>
//		<text>File type = "ooTextFile"
//		Object class = "TextGrid"
//
//		xmin = 0 
//		xmax = 1.970000
//		tiers? <exists> 
//		size = 3
//		item []:
//		    item [1]:
//		        class = "IntervalTier"
//		        name = "ORT"
//		        	...
//		     </text>
//		</result>
		segmentations = new ArrayList<Segmentation>(6);
		String line = null;
		//boolean inGrid = false;
		boolean startNewTier = false;
		boolean inTier = false;
		long begin = -1;
		long end = -1;
		String label;
		ArrayList<RSelection> segments = null;
		Segmentation curSegmentation = null;
		
		while ((line = reader.readLine()) != null) {
			//System.out.println(line);
			
			int index = -1;
			if (line.indexOf(tierSpec) > -1) {//IntervalTier
				startNewTier = true;
				inTier = false;
				continue;
			}
			index = line.indexOf(nm);// name
			if (startNewTier && index > -1) {
				int eqIndex = line.indexOf(eq);
				if (eqIndex > index && eqIndex + 1 < line.length()) {
					String tier = removeQuotes(line.substring(eqIndex + 1).trim());
					segments = new ArrayList<RSelection>();
					curSegmentation = new Segmentation(tier, segments, currentMediaFile);
					segmentations.add(curSegmentation);
					inTier = true;
				}
				startNewTier = false;
				continue;
			}
			if (inTier) {
				index = line.indexOf(min);
				if (index > -1) {
					int eqIndex = line.indexOf(eq);
					if (eqIndex > index && eqIndex + 1 < line.length()) {
						begin = extractLong(line, eqIndex);
					}
					continue;
				}
				index = line.indexOf(max);
				if (index > -1) {
					int eqIndex = line.indexOf(eq);
					if (eqIndex > index && eqIndex + 1 < line.length()) {
						end = extractLong(line, eqIndex);
					}
					continue;
				}
				index = line.indexOf(tx);
				if (index > -1) {
					int eqIndex = line.indexOf(eq);
					if (eqIndex > index && eqIndex + 1 < line.length()) {
						label = removeQuotes(line.substring(eqIndex + 1).trim());
						if (begin > -1 && end > begin) {
							Segment seg = new Segment(begin, end, label);
							if (segments != null) {
								segments.add(seg);
							}
						}
					}
					continue;
				}
			}
			
		}
		
		return HttpURLConnection.HTTP_OK;
	}
	
    /**
     * Extracts a double time value, multiplies by 1000 (sec to ms) and
     * converts to long.
     *
     * @param value the raw value
     * @param eqPos the index of the equals sign
     *
     * @return the time value rounded to milliseconds
     */
    private long extractLong(String value, int eqPos) {
        if (value.length() > (eqPos + 1)) {
            String v = value.substring(eqPos + 1).trim();
            long l = -1;

            try {
                Double d = new Double(v);
                l = Math.round(d.doubleValue() * 1000);
            } catch (NumberFormatException nfe) {
                reportBuilder.append("Not a valid numeric value: " + value + "\n");
            }

            return l;
        }

        return -1;
    }
	
    /**
     * Removes a beginning and end quote mark from the specified string. Does
     * no null check nor are spaces trimmed.
     *
     * @param value the value of which leading and trailing quote chars should
     *        be removed
     *
     * @return the value without the quotes
     */
    private String removeQuotes(String value) {
        if (value.charAt(0) == '"') {
            if (value.charAt(value.length() - 1) == '"' && value.length() > 1) {
                return value.substring(1, value.length() - 1);
            } else {
                return value.substring(1);
            }
        } else {
            if (value.charAt(value.length() - 1) == '"') {
                return value.substring(0, value.length() - 1);
            } else {
                return value;
            }
        }
    }
}
