<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl"
    exclude-result-prefixes="xs xd"
    version="2.0">
    <xd:doc scope="stylesheet">
        <xd:desc>
            <xd:p><xd:b>Created on:</xd:b> Jun 30, 2016</xd:p>
            <xd:p><xd:b>Author:</xd:b> Han Sloetjes</xd:p>
            <xd:p><xd:b>Description:</xd:b> An xslt stylesheet for converting a lexicon in the
                format of the lexicon component of ELAN to LIFT format.</xd:p>
        </xd:desc>
    </xd:doc>
    <xsl:output method="xml" indent="yes"/>
    
    <xsl:template match="/"> 
        <!-- <xsl:apply-templates/> -->
        <xsl:for-each select="lexicon">
            <xsl:call-template name="lexicon_template" />
        </xsl:for-each>
    </xsl:template>
    
    <xd:doc>
        <xd:desc><xd:p>The order of the sub-elements of lift as defined by 
            the lift_13.pdf documentation</xd:p>
        <xd:p>
            <xd:ul>
                <xd:li>header (0..1)</xd:li>
                <xd:li>entry (0..*)</xd:li>
            </xd:ul>
        </xd:p>
         <xd:p>The order within header</xd:p>
         <xd:p>
            <xd:ul>
                <xd:li>description (0..1)</xd:li>
                <xd:li>ranges (0..1)</xd:li>
                <xd:li>fields (0..1)</xd:li>
            </xd:ul>
         </xd:p>
        </xd:desc>
    </xd:doc>
    <xsl:template name="lexicon_template" match="lexicon">
        <xsl:variable name="main_language">
            <xsl:value-of select="header/language"/>
        </xsl:variable>
        <xsl:element name="lift">
            <!-- use header/author ? -->
            <xsl:attribute name="producer">ELAN-Lexicon to LIFT Transformer</xsl:attribute>
            <!-- the version number is lift language version use header/version 0.13 -->
            <xsl:attribute name="version">0.13</xsl:attribute>
                <!--  
                <xsl:choose>
                    <xsl:when test="header/version">
                        <xsl:value-of select="header/version"/>
                    </xsl:when>
                    <xsl:otherwise>0.13</xsl:otherwise>
                </xsl:choose>
            </xsl:attribute>-->
             <xsl:element name="header">
                 <xsl:if test="header/description">
                     <xsl:element name="description">
                         <xsl:call-template name="multitext_with_params">
                             <xsl:with-param name="lang_param" select="$main_language"/>
                             <xsl:with-param name="text_content" select="header/description"/>
                         </xsl:call-template>
                     </xsl:element>
                 </xsl:if>
             </xsl:element>   
            <!-- what about LIFT ranges and fields? -->
            <!-- what about ELAN Lexicon "name" and "language" -->
            <!-- the entries -->
            <xsl:for-each select="entry">
                <xsl:call-template name="entry_template">
                    <xsl:with-param name="main_language" select="$main_language"/>
                </xsl:call-template>
            </xsl:for-each>
        </xsl:element>
    </xsl:template>
    
    <!-- entry template -->
    <xd:doc scope="stylesheet">
        <xd:desc>
            <xd:p>The order of subelements of entry as defined by 
                the lift_13.pdf documentation</xd:p>
            <xd:p>
                <xd:ul>
                    <xd:li>lexical-unit (0..1)</xd:li>
                    <xd:li>citation (0..1)</xd:li>
                    <xd:li>pronunciation (0..*)</xd:li>
                    <xd:li>variant (0..*)</xd:li>
                    <xd:li>sense (0..*)</xd:li>
                    <xd:li>note (0..*)</xd:li>
                    <xd:li>relation (0..*)</xd:li>
                    <xd:li>etymology (0..*)</xd:li>
                </xd:ul>
            </xd:p>
        </xd:desc>
    </xd:doc>
    <xsl:template name="entry_template" match="entry">
        <xsl:param name="main_language"/>
        <xsl:element name="entry">
            <xsl:attribute name="dateCreated">
                <xsl:value-of select="@dateCreated"/>
            </xsl:attribute>
            <xsl:attribute name="dateModified">
                <xsl:value-of select="@dateModified"/>
            </xsl:attribute>
            <!-- special treatment of "guid" and "id" -->
            <xsl:variable name="guid_id">
                <xsl:choose>
                    <!-- test length of the id and the prefix -->
                    <xsl:when test="string-length(@id) = 38 and starts-with(@id, 'e_')">
                        <xsl:value-of select="substring(@id, 3)"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="@id"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:variable>
            <!-- a value based prefix -->
            <xsl:variable name="value_prefix">
                <!-- needs to be normalized? -->
                <xsl:choose>
                    <xsl:when test="citation">
                        <xsl:value-of select="citation"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="lexical-unit"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:variable>
            <!-- guid and or id attributes -->
            <xsl:choose>
                <!-- if it has the length of a guid -->
                <xsl:when test="string-length($guid_id)=36">
                    <!-- assume it is a guid, treat it as such  -->
                    <xsl:attribute name="guid">
                        <xsl:value-of select="$guid_id"/>
                    </xsl:attribute>
                    <!-- construct an id of the (citation or lexical-unit) + entry order (if any) + $guid_id -->
                    <xsl:attribute name="id">
                        <xsl:choose>
                            <xsl:when test="@order">
                                <xsl:value-of select="concat($value_prefix, @order ,'_', $guid_id)"/>   
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="concat($value_prefix, '_', $guid_id)"/>                           
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:attribute>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:attribute name="id">
                        <xsl:value-of select="@id"/><!-- raw copy of the id -->
                    </xsl:attribute>
                </xsl:otherwise>
            </xsl:choose>
            <!-- subelements -->
            <!-- lexical unit -->
            <xsl:element name="lexical-unit">
                <xsl:element name="form">
                    <xsl:attribute name="lang">
                        <xsl:value-of select="$main_language"/>
                    </xsl:attribute>
                    <xsl:element name="text">
                        <xsl:value-of select="lexical-unit"/>
                    </xsl:element>
                </xsl:element>               
            </xsl:element>
            <!-- morph type -->
            <xsl:element name="trait">
                <xsl:attribute name="name">morph-type</xsl:attribute>
                <xsl:attribute name="value">
                    <xsl:choose>
                        <xsl:when test="morph-type">
                            <xsl:value-of select="morph-type"/>
                        </xsl:when>
                        <xsl:otherwise>root</xsl:otherwise>
                    </xsl:choose>
                </xsl:attribute>
            </xsl:element>
            <xsl:if test="citation">
                <xsl:element name="citation">
                    <xsl:for-each select="citation"><!-- although there is only one citation, make use of the "select" -->
                        <xsl:call-template name="form_and_text_template"/>
                    </xsl:for-each>
                </xsl:element>
            </xsl:if>
            <!-- TODO variant etc. -->
            <!-- senses -->
            <xsl:for-each select="sense">
                <xsl:call-template name="sense_template"/>
            </xsl:for-each>
            <!-- note -->
            <xsl:for-each select="note">
                <xsl:call-template name="note_template"/>
            </xsl:for-each>
            <!-- relation can be used for parent ref? -->
            <xsl:if test="@entryRef">
                <xsl:call-template name="relation_template"/>
            </xsl:if>
        </xsl:element>
    </xsl:template>
    
    <!-- sense template -->
    <xd:doc>
        <xd:desc>
            <xd:p>The order of sub-elements of sense as defined by 
            the lift_13.pdf documentation</xd:p>
            <xd:p>
                <xd:ul>
                    <xd:li>grammatical-info (0..1)</xd:li>
                    <xd:li>gloss (0..*)</xd:li>
                    <xd:li>definition (0..1)</xd:li>
                    <xd:li>relation (0..*)</xd:li>
                    <xd:li>note (0..*)</xd:li>
                    <xd:li>example (0..*)</xd:li>
                    <xd:li>reversal (0..*)</xd:li>
                    <xd:li>illustration (0..*)</xd:li>
                    <xd:li>subsense (0..*)</xd:li>
                </xd:ul>
            </xd:p>
        </xd:desc>       
    </xd:doc>
    <xsl:template name="sense_template" match="sense">
        <xsl:element name="sense">
            <xsl:attribute name="id">
                <xsl:choose>
                    <xsl:when test="string-length(@id) = 38 and starts-with(@id, 's_')">
                        <xsl:value-of select="substring(@id, 3)"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="@id"/>
                    </xsl:otherwise>
                </xsl:choose>              
            </xsl:attribute>
            <!--lift does not need order if there is only one sense, could check for that here -->
            <xsl:attribute name="order">
                <xsl:value-of select="@order"/>
            </xsl:attribute>
            <!-- the traits are currently lost on "import" -->
            <xsl:element name="grammatical-info">
                <xsl:attribute name="value">
                    <xsl:value-of select="grammatical-category"/>
                </xsl:attribute>
            </xsl:element>
            <!-- glosses -->
            <xsl:for-each select="gloss">
                <xsl:call-template name="gloss_template">
                </xsl:call-template>
            </xsl:for-each>
            <!-- definitions -->
            <xsl:if test="definition">
                <!-- call the defintion template once -->
                <xsl:call-template name="definition_template"/>
            </xsl:if>
            <!-- note's / internal-note's -->
            <xsl:for-each select="internal-note">
                <xsl:call-template name="internal_note_template"/>
            </xsl:for-each>
            <!-- examples -->
            <xsl:if test="example">
                <xsl:for-each select="example">
                    <xsl:call-template name="example_template"/>
                </xsl:for-each>
            </xsl:if>
            <!-- illustration from fragment-ref? -->
            <xsl:if test="example/fragment-ref">
                <xsl:for-each select="example/fragment-ref">
                    <xsl:call-template name="illustration_template"/>
                </xsl:for-each>
            </xsl:if>
            <!-- here subsense could be added? Senses of entries that refer to this sense?  -->
        </xsl:element>    
    </xsl:template>
    
    <!-- gloss template -->
    <xsl:template name="gloss_template" match="gloss">
        <xsl:element name="gloss">
            <xsl:if test="@lang">
                <xsl:attribute name="lang">
                    <xsl:value-of select="@lang"/>
                </xsl:attribute>
            </xsl:if>
            <xsl:element name="text">
                <xsl:value-of select="text()"/>
            </xsl:element>
        </xsl:element>    
    </xsl:template>
    <!-- definition template -->
    <xsl:template name="definition_template" match="definition">
        <xsl:element name="definition">
            <xsl:for-each select="definition">
                <xsl:call-template name="form_and_text_template"/>
            </xsl:for-each>
        </xsl:element>
    </xsl:template>
    <!-- example template -->
    <xsl:template name="example_template" match="example">
        <xsl:element name="example">
            <xsl:if test="source-ref">
                <xsl:attribute name="source">
                    <xsl:value-of select="source-ref"/>
                </xsl:attribute>
            </xsl:if>
            <!-- or call the multipletext template ? -->
            <xsl:element name="form">
                <xsl:if test="text/@lang">
                    <xsl:attribute name="lang">
                        <xsl:value-of select="text/@lang"/>
                    </xsl:attribute>
                </xsl:if>
                <xsl:element name="text">
                    <xsl:value-of select="text"/>
                </xsl:element>
            </xsl:element>
            <xsl:if test="translation">
                <xsl:call-template name="translation_template"/>
            </xsl:if>
            <xsl:if test="source-ref">
                <xsl:for-each select="source-ref">
                    <xsl:element name="note">
                        <xsl:attribute name="type">reference</xsl:attribute>
                        <xsl:call-template name="form_and_text_template"/>
                    </xsl:element>
                </xsl:for-each>
            </xsl:if>
        </xsl:element>
    </xsl:template>
    <!-- translation template -->
    <xsl:template name="translation_template" match="translation">
        <xsl:element name="translation">
            <xsl:for-each select="translation">
                <xsl:call-template name="form_and_text_template"/>
            </xsl:for-each>
        </xsl:element>
    </xsl:template>
    <!-- note template -->
    <xsl:template name="internal_note_template" match="internal-note">
        <xsl:element name="note">
            <xsl:call-template name="form_and_text_template"/>
        </xsl:element>    
    </xsl:template>
    <!-- note template -->
    <xsl:template name="note_template" match="note">
        <xsl:element name="note">
            <xsl:call-template name="form_and_text_template"/>
        </xsl:element>    
    </xsl:template>
    <!-- illustration template -->
    <xsl:template name="illustration_template" match="fragment-ref">
        <xsl:element name="illustration">
            <xsl:element name="URLRef">
                <xsl:attribute name="URL">
                    <xsl:value-of select="fragment-ref"/>
                </xsl:attribute>
            </xsl:element>
        </xsl:element>
    </xsl:template>
    <!-- relation template -->
    <xsl:template name="relation_template">
        <xsl:element name="relation">
            <xsl:attribute name="ref">
                <xsl:value-of select="@entryRef"/>    
            </xsl:attribute>
            <!-- 
            <xsl:attribute name="order">
                what order would be mentioned here
            </xsl:attribute>
             -->
        </xsl:element>    
    </xsl:template>
    
    <!-- a template that converts a single element (e.g. <text lang="">) into <form lang=""><text/></form>.
         Could pass the main language from each template to then next template that is called? -->
    <xsl:template name="form_and_text_template">
        <xsl:element name="form">
            <!-- use choose/when/otherwise and write the default language attribute? 
            of xsl:if and only write if there is a lang attribute -->
            <xsl:attribute name="lang">
                <xsl:choose>
                    <xsl:when test="@lang">
                        <xsl:value-of select="@lang"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="/lexicon/header/language"/><!-- $main_language -->
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:attribute>
            <!--  
            <xsl:if test="@lang">
                <xsl:attribute name="lang" select="@lang"/>
            </xsl:if>
            -->
            <xsl:element name="text">
                <xsl:value-of select="text()"/>
            </xsl:element>
        </xsl:element>
    </xsl:template>
    
    <!-- Creates a single form element with a (optional) language attribute and an 
        enclosed text element. Roughly the "multitext" class in lift. -->
    <xsl:template name="multitext_with_params">
        <xsl:param name="lang_param"/>
        <xsl:param name="text_content"/>
        <xsl:element name="form">
            <xsl:if test="$lang_param">
                <xsl:attribute name="lang">
                    <xsl:value-of select="$lang_param"/>
                </xsl:attribute>
            </xsl:if>
            <xsl:element name="text">
                <xsl:value-of select="$text_content"/>
            </xsl:element>
        </xsl:element>
    </xsl:template>
    
</xsl:stylesheet>