<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl"
    exclude-result-prefixes="xs xd"
    version="2.0">

    <xd:doc scope="stylesheet">
        <xd:desc>
            <xd:p><xd:b>Created on:</xd:b> Jun 7, 2016</xd:p>
            <xd:p><xd:b>Author:</xd:b> Han Sloetjes</xd:p>
            <xd:p><xd:b>Description:</xd:b> An xslt stylesheet for converting a lexicon in LIFT format to the
            format lexicon component of ELAN. This is the XSLT 2.0 version, it uses a few function not
            available in 1.0. 
                <xd:ul>
                    <xd:li>'xsl:namespace' elements in lexicon element</xd:li>
                    <xd:li>the 'distinct-values' function used to detect the most often used lang attribute for lexical units</xd:li>
                    <xd:li>the 'compare()' function in the example-template</xd:li>
                </xd:ul></xd:p>
            
        </xd:desc>
    </xd:doc>

    <xsl:output method="xml" indent="yes"/>
    
    <xsl:variable name="schema_url">http://www.mpi.nl/tools/elan/LexiconComponent-1.0.xsd</xsl:variable>
    <xsl:variable name="schema_version">1.0</xsl:variable>
    
    <xsl:template match="/"> 
        <!-- uses xsl:call-template instead of xsl:apply-templates, 
            that makes a difference in how the select="" syntax is interpreted-->
        <xsl:for-each select="lift">
            <xsl:call-template name="lift_template" />
        </xsl:for-each>
    </xsl:template>
    
    <xsl:template name="lift_template" match="lift"> <!--  match="lift" superfluous -->
        <!-- try to determine the main language for the lexical-units -->
        <xsl:variable name="main_language">                
            <xsl:variable name="all_lang" select="distinct-values(entry/lexical-unit/form/@lang)"/> 
            <xsl:variable name="num_lang" select="count($all_lang)"/>
            <xsl:choose>
                <xsl:when test="$num_lang=1">
                    <xsl:value-of select="$all_lang[1]"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:for-each-group select="entry/lexical-unit/form" group-by="@lang">
                        <!-- sorts ascending, last item is the most frequent language -->
                        <xsl:sort select="count(current-group())"/>
                        <xsl:if test="position() = last()">
                            <xsl:value-of select="@lang"/>
                        </xsl:if>
                    </xsl:for-each-group>
                </xsl:otherwise>
            </xsl:choose>                   
        </xsl:variable>
        
        <xsl:element name="lexicon">
            <xsl:namespace name="vc" select="'http://www.w3.org/2007/XMLSchema-versioning'"/>
            <xsl:namespace name="xsi" select="'http://www.w3.org/2001/XMLSchema-instance'"/>         
            <xsl:attribute namespace="http://www.w3.org/2001/XMLSchema-instance" name="noNamespaceSchemaLocation">
                <xsl:value-of select="$schema_url"/>
            </xsl:attribute>
            <xsl:attribute name="schemaVersion">
                <xsl:value-of select="$schema_version"/>
            </xsl:attribute>
            <xsl:attribute name="producer">Lift XSLT Transformer</xsl:attribute>
            <xsl:element name="header">
                <xsl:element name="name">Unnamed Lift Lexicon</xsl:element>
                <xsl:element name="language">
                    <xsl:value-of select="$main_language"/>
                </xsl:element>
                <xsl:element name="author">
                    <xsl:value-of select="@producer"/>
                </xsl:element>
                <xsl:element name="version">
                    <xsl:value-of select="@version"/>
                </xsl:element>
            </xsl:element>
            <!-- <xsl:apply-templates/> -->
            <xsl:for-each select="entry">
                <xsl:call-template name="entry_template"/>
            </xsl:for-each>
        </xsl:element>
    </xsl:template>
    
    <xsl:template name="entry_template" match="entry">
        <xsl:element name="entry">
            <xsl:attribute name="id">
                <xsl:choose>
                    <xsl:when test="@guid">
                        <xsl:value-of select="concat('e_', @guid)"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <!-- FLEx export uses id's that are a concatenation of lexical-unit and the guid.
                        lexical-unit can contain characters that are invalid in ID fields (NCName) -->
                        <xsl:choose>
                            <xsl:when test="string-length(@id) >= 36">
                                <xsl:value-of select="concat('e_', substring(@id, string-length(@id) - 35))"/><!-- character index is 1-based -->
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="concat('e_', @id)"/>
                            </xsl:otherwise>
                        </xsl:choose>                      
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:attribute>
            
            <xsl:attribute name="dateCreated">
                <xsl:value-of select="@dateCreated"/>
            </xsl:attribute>
            <xsl:attribute name="dateModified">
                <xsl:value-of select="@dateModified"/>
            </xsl:attribute>
            <xsl:if test="@order">
                <xsl:attribute name="order">
                    <xsl:value-of select="@order"/>
                </xsl:attribute>
            </xsl:if>
            
            <xsl:element name="lexical-unit">
                <xsl:value-of select="lexical-unit/form/text"/>
            </xsl:element>
            
            <xsl:if test="citation/form/text">
                <xsl:element name="citation">
                    <xsl:value-of select="citation/form/text"/>
                </xsl:element>
            </xsl:if>
            <xsl:element name="morph-type">
                <xsl:call-template name="trait_template"/>
            </xsl:element>
            <xsl:for-each select="variant">
                <xsl:call-template name="variant_template"/>
            </xsl:for-each>
            <xsl:for-each select="field">
                <xsl:call-template name="field_template"/>
            </xsl:for-each>
            <xsl:for-each select="sense">
                <xsl:call-template name="sense_template"/>
            </xsl:for-each>         
        </xsl:element>
    </xsl:template>
    
    <!-- the 'trait' template is called from within the 'entry' element -->
    <xsl:template name="trait_template" match="trait">
        <xsl:choose>
            <xsl:when test="trait[@name='morph-type']">
                <xsl:value-of select="trait/@value"/>
            </xsl:when>
            <xsl:otherwise>stem</xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template name="variant_template">
        <xsl:element name="variant">
            <xsl:value-of select="form/text"/>
        </xsl:element>    
    </xsl:template>
    
    <xsl:template name="sense_template" match="sense">
        <xsl:element name="sense">
            <xsl:attribute name="id">
                <!-- <xsl:value-of select="sense/@id"/> -->
                <xsl:value-of select="concat('s_', @id)"/>
            </xsl:attribute>
            <xsl:choose>
                <xsl:when test="@order">
                    <xsl:attribute name="order">
                        <xsl:value-of select="@order"/>
                    </xsl:attribute> 
                </xsl:when>
                <xsl:otherwise>
                    <xsl:attribute name="order">1</xsl:attribute> 
                </xsl:otherwise>
            </xsl:choose>           
            <xsl:element name="grammatical-category">
                <xsl:value-of select="grammatical-info/@value"/>
            </xsl:element>
            <xsl:for-each select="gloss">
                <xsl:call-template name="gloss_template"/>
            </xsl:for-each>            
            <xsl:for-each select="definition">
                <xsl:call-template name="definition_template"/>
            </xsl:for-each>
            <xsl:for-each select="example">
                <xsl:call-template name="example_template"/>
            </xsl:for-each>
            <xsl:for-each select="note">
                <xsl:call-template name="note_template"/>
            </xsl:for-each>
        </xsl:element>
    </xsl:template>

    <xsl:template name="gloss_template">
        <xsl:element name="gloss">
            <xsl:attribute name="lang">
                <xsl:value-of select="@lang"/>
            </xsl:attribute>
            <xsl:value-of select="text"/>
        </xsl:element>
    </xsl:template>
    
    <xsl:template name="definition_template">
        <xsl:for-each select="form">
            <xsl:element name="definition">
                <xsl:attribute name="lang">
                    <xsl:value-of select="@lang"/>
                </xsl:attribute>
                <xsl:value-of select="text"/>
            </xsl:element>    
        </xsl:for-each>      
    </xsl:template>
    <!-- example template -->
    <xsl:template name="example_template">
        <xsl:if test="count(child::form) > 0 or count(child::translation) > 0 or count(child::note) > 0">
            <xsl:element name="example">
                <xsl:choose>
                    <xsl:when test="form/text">
                        <xsl:element name="text">
                            <xsl:attribute name="lang">
                                <xsl:value-of select="form/@lang"/>
                            </xsl:attribute>
                            <xsl:value-of select="form/text"/>
                        </xsl:element>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:element name="text"/><!-- required child element -->
                    </xsl:otherwise>
                </xsl:choose>  
                <xsl:for-each select="translation">
                    <xsl:call-template name="translation_template"/>          
                </xsl:for-each>
                <xsl:choose>
                    <xsl:when test="@source">
                        <xsl:element name="source-ref">
                            <xsl:value-of select="@source"/>
                        </xsl:element>
                    </xsl:when>
                    <xsl:otherwise>
                        <!-- note in combination with type attribute reference, converts to source-ref -->
                        <!-- for XSLT 2.0 use compare(note/@type, 'reference') = 0 -->                               
                        <xsl:if test="note and compare(note/@type, 'reference') = 0">
                            <xsl:element name="source-ref">
                                <xsl:value-of select="note/form/text"/>
                            </xsl:element>
                        </xsl:if>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:element>  
        </xsl:if>
    </xsl:template>
    
    <xsl:template name="note_template">
        <xsl:element name="internal-note">
            <xsl:value-of select="form/text"/>
        </xsl:element>
    </xsl:template>
    
    <xsl:template name="field_template">
        <xsl:element name="field">
            <xsl:attribute name="name">
                <xsl:value-of select="@type"/>
            </xsl:attribute>
            <xsl:attribute name="lang">
                <xsl:value-of select="form/@lang"/>
            </xsl:attribute>
            <xsl:value-of select="form/text"/>
        </xsl:element>
    </xsl:template>
    
    <!-- there can be more than one translation elements in an example and there can be more 
    than one form elements within a translation -->
    <xsl:template name="translation_template">
        <xsl:for-each select="form">
            <xsl:element name="translation">
                <xsl:attribute name="lang">
                    <xsl:value-of select="@lang"/>
                </xsl:attribute>
                <xsl:value-of select="text"/>
            </xsl:element>
        </xsl:for-each>
    </xsl:template>
    
</xsl:stylesheet>